package org.gcube.indexmanagement.featureindexlibrary.vafile.io;

import java.io.DataInputStream;
import java.io.EOFException;
import java.io.RandomAccessFile;

import org.apache.log4j.Logger;
import org.gcube.indexmanagement.featureindexlibrary.commons.FIEnums;
import org.gcube.indexmanagement.featureindexlibrary.vafile.VAFileParams;
import org.gcube.indexmanagement.featureindexlibrary.vafile.elements.ApproximationFileEntry;
import org.gcube.indexmanagement.featureindexlibrary.vafile.elements.VectorFileEntry;

/**
 * IO Helper class
 * 
 * @author UoA
 */
public class IOHelper {
	/**
	 * Logger this class uses
	 */
	private static Logger log = Logger.getLogger(VectorFileRandomAccess.class);
	/**
	 * The id length
	 */
	private int idlength=-1;
	/**
	 * The vector length
	 */
	private int vectorlength=-1;
	
	/**
	 * Creates a new instance
	 */
	public IOHelper(){}
	
	/**
	 * Sets the property value
	 * 
	 * @param idlength The property value
	 * @throws Exception An error
	 */
	public void setIDLength(int idlength) throws Exception{
		this.idlength=idlength;
	}
	
	/**
	 * Sets the property value
	 * 
	 * @param vectorlength The property value
	 * @throws Exception An error
	 */
	public void setVectorLength(int vectorlength) throws Exception{
		this.vectorlength=vectorlength;
	}
	
	/**
	 * Writes an id
	 * 
	 * @param toWrite The string to write
	 * @param rand Th efile pointer
	 * @throws Exception An error
	 */
	private void writeID(String toWrite,RandomAccessFile rand) throws Exception{
		rand.writeChars(toWrite);
	}
	
	/**
	 * Read an id
	 * 
	 * @param rand The pointer
	 * @return the id
	 * @throws Exception An error
	 */
	private String readID(RandomAccessFile rand) throws Exception{
		char []buf=new char[this.idlength];
		for(int i=0;i<this.idlength;i+=1){
			buf[i]=rand.readChar();
		}
		return new String(buf);
	}
	
	/**
	 * Read an id
	 * 
	 * @param in The pointer
	 * @return the id
	 * @throws Exception An error
	 */
	private String readID(DataInputStream in) throws Exception{
		char []buf=new char[this.idlength];
		for(int i=0;i<this.idlength;i+=1){
			buf[i]=in.readChar();
		}
		return new String(buf);
	}
	
	/**
	 * Writes a string
	 * 
	 * @param toWrite The string to write
	 * @param rand TYhe pointer
	 * @throws Exception An error
	 */
	private void writeString(String toWrite,RandomAccessFile rand) throws Exception{
		rand.writeInt(toWrite.toCharArray().length);
		rand.writeChars(toWrite);
	}
	
	/**
	 * Reads a string
	 * 
	 * @param rand The pointer
	 * @return The string
	 * @throws Exception An error
	 */
	private String readString(RandomAccessFile rand) throws Exception{
		int count=rand.readInt();
		char []buf=new char[count];
		for(int i=0;i<count;i+=1){
			buf[i]=rand.readChar();
		}
		return new String(buf);
	}
	
	/**
	 * Writes a vector
	 * 
	 * @param toWrite The vector
	 * @param rand The pointer
	 * @throws Exception An error
	 */
	private void writeVector(float []toWrite,RandomAccessFile rand) throws Exception{
		for(int i=0;i<this.vectorlength;i+=1){
			rand.writeFloat(toWrite[i]);
		}
	}
	
	/**
	 * Reads a vector
	 * 
	 * @param rand The pointer
	 * @return The vector
	 * @throws Exception An error
	 */
	private float []readVector(RandomAccessFile rand) throws Exception{
		float []vector=new float[this.vectorlength];
		for(int i=0;i<this.vectorlength;i+=1){
			vector[i]=rand.readFloat();
		}
		return vector;
	}
	
	/**
	 * Reads a vector
	 * 
	 * @param in The pointer
	 * @return The vector
	 * @throws Exception An error
	 */
	private float []readVector(DataInputStream in) throws Exception{
		float []vector=new float[this.vectorlength];
		for(int i=0;i<this.vectorlength;i+=1){
			vector[i]=in.readFloat();
		}
		return vector;
		
	}
	
	/**
	 * Writes a boolean value
	 * 
	 * @param active The value
	 * @param rand The pointer
	 * @throws Exception An error
	 */
	private void writeFlag(boolean active,RandomAccessFile rand) throws Exception{
		rand.writeBoolean(active);
	}
	
	/**
	 * Reads a boolean value
	 * 
	 * @param rand The pointer
	 * @return The boolean
	 * @throws Exception An error
	 */
	private boolean readFlag(RandomAccessFile rand) throws Exception{
		return rand.readBoolean();
	}
	
	/**
	 * Reads a boolean value
	 * 
	 * @param in The pointer
	 * @return The boolean
	 * @throws Exception An error
	 */
	private boolean readFlag(DataInputStream in) throws Exception{
		return in.readBoolean();
	}
	
	/**
	 * Writes a short
	 * 
	 * @param toWrite The value
	 * @param rand The pointer
	 * @throws Exception An error
	 */
	private void writeShort(short toWrite,RandomAccessFile rand) throws Exception{
		rand.writeShort(toWrite);
	}
	
	/**
	 * Reads a short
	 * 
	 * @param rand The pointer
	 * @return The short
	 * @throws Exception An error
	 */
	private short readShort(RandomAccessFile rand) throws Exception{
		return rand.readShort();
	}
	
	/**
	 * Writes an int
	 * 
	 * @param toWrite The int to write
	 * @param rand The pointer
	 * @throws Exception An error
	 */
	private void writeInt(int toWrite,RandomAccessFile rand) throws Exception{
		rand.writeInt(toWrite);
	}
	
	/**
	 * Reads an int
	 * 
	 * @param rand The pointer
	 * @return the int
	 * @throws Exception an error
	 */
	private int readInt(RandomAccessFile rand) throws Exception{
		return rand.readInt();
	}
	
	/**
	 * Writes a long
	 * 
	 * @param toWrite The long
	 * @param rand The pointer
	 * @throws Exception An error
	 */
	private void writeLong(long toWrite,RandomAccessFile rand) throws Exception{
		rand.writeLong(toWrite);
	}
	
	/**
	 * Reads a long
	 * 
	 * @param rand The pointer
	 * @return The long
	 * @throws Exception An error
	 */
	private long readLong(RandomAccessFile rand) throws Exception{
		return rand.readLong();
	}
	
	/**
	 * Writer the partition points
	 * 
	 * @param partitionPoints The partition points
	 * @param rand The pointer
	 * @throws Exception An error
	 */
	private void writePartitionPoints(float [][]partitionPoints,RandomAccessFile rand) throws Exception{
		for(int i=0;i<partitionPoints.length;i+=1){
			for(int q=0;q<partitionPoints[i].length;q+=1){
				rand.writeFloat(partitionPoints[i][q]);
			}
		}
	}
	
	/**
	 * Reads partition points
	 * 
	 * @param dimentions The dimentions
	 * @param buckets The buckets
	 * @param rand The pointer
	 * @return The partition points
	 * @throws Exception An error
	 */
	private float [][]readPartitionPoints(int dimentions,int buckets,RandomAccessFile rand) throws Exception{
		float [][]partitionPoints=new float[dimentions][buckets];
		for(int i=0;i<partitionPoints.length;i+=1){
			for(int q=0;q<partitionPoints[i].length;q+=1){
				partitionPoints[i][q]=rand.readFloat();
			}
		}
		return partitionPoints;
	}
	
	/**
	 * Writes the header
	 * 
	 * @param params The VAFile header params
	 * @param rand The pointer
	 * @throws Exception An error
	 */
	public void writeHeader(VAFileParams params,RandomAccessFile rand) throws Exception{
		if(params.getPartitionPoints().length!=params.getVectorLength()|| params.getPartitionPoints()[0].length!=params.getBuckets()){
			throw new Exception("invalid partition points lengths");
		}
		try{
			rand.seek(0);
			long start=rand.getFilePointer();
			this.writeLong(start,rand);
			this.writeString(params.getIndexName(),rand);
			this.writeString(params.getIndexID(),rand);
			this.writeString(params.getDistanceMeasure().toString(),rand);
			this.writeString(params.getFeatureName().toString(),rand);
			this.writeString(params.getStorage(),rand);
			this.writeInt(params.getIDLength(),rand);
			this.writeInt(params.getVectorLength(),rand);
			this.writeLong(params.getElementCount(),rand);
			this.writeShort(params.getBits(),rand);
			this.writeInt(params.getBuckets(),rand);
			this.writeLong(params.getApproximationEntryLength(),rand);
			this.writeLong(params.getVectorEntryLength(),rand);
			this.writePartitionPoints(params.getPartitionPoints(),rand);
			this.writeVector(params.getXSum(),rand);
			this.writeVector(params.getX2Sum(),rand);
			this.writeVector(params.getWeights(),rand);
			this.writeFlag(params.getWeighted(),rand);
			long offset=rand.getFilePointer();
			rand.seek(start);
			this.writeLong(offset,rand);
		}catch(Exception e){
			log.error("Could not write header. Thrpwing Exception",e);
			throw new Exception("Could not write header");
		}
//		logger.info("header offset"+offset);
	}
	
	/**
	 * Skips the header
	 * 
	 * @param rand The pointer
	 * @throws Exception An error
	 */
	public void skipHeader(RandomAccessFile rand) throws Exception{
		try{
			rand.seek(0);
			long offset=this.readLong(rand);
			rand.seek(offset);
		}catch(Exception e){
			log.error("Could not skip header. Thrpwing Exception",e);
			throw new Exception("Could not skip header");
		}
	}
	
	/**
	 * Reads a header
	 * 
	 * @param rand The pointer
	 * @return The header
	 * @throws Exception An error
	 */
	public VAFileParams readHeader(RandomAccessFile rand) throws Exception{
		try{
			rand.seek(0);
			this.readLong(rand);
			VAFileParams params=new VAFileParams();
			params.setIndexName(this.readString(rand));
			params.setIndexID(this.readString(rand));
			params.setDistanceMeasure(FIEnums.DistanceTypes.valueOf(this.readString(rand)));
			params.setFeatureName(this.readString(rand));
			params.setStorage(this.readString(rand));
			params.setIDLength(this.readInt(rand));
			this.idlength=params.getIDLength();
			params.setVectorLength(this.readInt(rand));
			this.vectorlength=params.getVectorLength();
			params.setElementCount(this.readLong(rand));
			params.setBits(this.readShort(rand));
			params.setBuckets(this.readInt(rand));
			params.setApproximationEntryLength(this.readLong(rand));
			params.setVectorEntryLength(this.readLong(rand));
			params.setPartitionPoints(this.readPartitionPoints(params.getVectorLength(),params.getBuckets(),rand));
			params.setXSum(this.readVector(rand));
			params.setX2Sum(this.readVector(rand));
			params.setWeights(this.readVector(rand));
			params.setWeighted(this.readFlag(rand));
			return params;
		}catch(Exception e){
			log.error("Could not read header. Thrpwing Exception",e);
			throw new Exception("Could not read header");
		}
	}
	
	/**
	 * Writes an Approximation 
	 * 
	 * @param entry The entry
	 * @param rand The pointer
	 * @throws Exception An error
	 */
	public void writeApproximation(ApproximationFileEntry entry,RandomAccessFile rand) throws Exception{
		try{
			this.writeFlag(entry.isActive(),rand);
			rand.write(entry.getBitString());
		}catch(Exception e){
			log.error("Could not write approximation. Thrpwing Exception",e);
			throw new Exception("Could not write approximation");
		}
	}
	
	/**
	 * Reads an approximation
	 * 
	 * @param rand The pointer
	 * @return The entry
	 */
	public ApproximationFileEntry readApproximation(RandomAccessFile rand){
		try{
			byte []bitString=new byte[this.vectorlength];
			boolean active=this.readFlag(rand);
			for(int i=0;i<bitString.length;i+=1){
				bitString[i]=rand.readByte();
			}
			return new ApproximationFileEntry(bitString,active);
		}catch(EOFException e){
			return null;
		}catch(Exception e){
			log.debug("Could not read apprixmiation. assuming EOF",e);
			return null;
		}
	}
	
	/**
	 * Reads an approximation
	 * 
	 * @param in The pointer
	 * @return The entry
	 */
	public ApproximationFileEntry readApproximation(DataInputStream in){
		try{
			byte []bitString=new byte[this.vectorlength];
			boolean active=in.readBoolean();
			for(int i=0;i<bitString.length;i+=1){
				bitString[i]=in.readByte();
			}
			return new ApproximationFileEntry(bitString,active);
		}catch(EOFException e){
			return null;
		}catch(Exception e){
			log.debug("Could not read apprixmiation. assuming EOF",e);
			return null;
		}
	}
	
	/**
	 * Writes a record
	 * 
	 * @param entry The entry
	 * @param rand The pointer
	 * @throws Exception An error
	 */
	public void writeRecord(VectorFileEntry entry,RandomAccessFile rand) throws Exception{
		if(this.idlength != entry.getId().length() || this.vectorlength!=entry.getVector().length){
			throw new Exception("dimention length missmatch");
		}
		try{
			this.writeFlag(entry.isActive(),rand);
			this.writeID(entry.getId(),rand);
			this.writeVector(entry.getVector(),rand);
		}catch(Exception e){
			log.error("Could not write record. Thrpwing Exception",e);
			throw new Exception("Could not write record");
		}
	}
	
	/**
	 * Reads a record
	 * 
	 * @param rand The pointer
	 * @return Th erecord
	 */
	public VectorFileEntry readRecord(RandomAccessFile rand){
		try{
			boolean active=this.readFlag(rand);
			String id=this.readID(rand);
			float []vector=this.readVector(rand);
			return new VectorFileEntry(id,vector,active);
		}catch(EOFException e){
			return null;
		}catch(Exception e){
			log.debug("Could not read record. assuming EOF",e);
			return null;
		}
	}
	
	/**
	 * Reads a record
	 * 
	 * @param in The pointer
	 * @return The record
	 */
	public VectorFileEntry readRecord(DataInputStream in){
		try{
			boolean active=this.readFlag(in);
			String id=this.readID(in);
			float []vector=this.readVector(in);
			return new VectorFileEntry(id,vector,active);
		}catch(EOFException e){
			return null;
		}catch(Exception e){
			log.debug("Could not read apprixmiation. assuming EOF",e);
			return null;
		}
	}
}
