package org.gcube.indexmanagement.featureindexlibrary.vafile.algo;

import java.io.RandomAccessFile;

import org.apache.log4j.Logger;
import org.gcube.indexmanagement.featureindexlibrary.commons.FeatureVectorElement;
import org.gcube.indexmanagement.featureindexlibrary.commons.FileHelper;
import org.gcube.indexmanagement.featureindexlibrary.vafile.VAFileParams;
import org.gcube.indexmanagement.featureindexlibrary.vafile.elements.ApproximationFileEntry;
import org.gcube.indexmanagement.featureindexlibrary.vafile.elements.VectorFileEntry;
import org.gcube.indexmanagement.featureindexlibrary.vafile.io.FileBufferReader;
import org.gcube.indexmanagement.featureindexlibrary.vafile.io.IOHelper;

/**
 * Populates an index
 * 
 * @author UoA
 */
public class PopulateIndex {
	/**
	 * The Looger used by this class
	 */
	private static Logger logger = Logger.getLogger(PopulateIndex.class);
	/**
	 * VAFile params
	 */
	private VAFileParams params=null;
	/**
	 * TWheter or not this index is firstly populated
	 */
	private boolean first=true;
	/**
	 * File pointer
	 */
	private RandomAccessFile vrand=null;
	/**
	 * File pointer
	 */
	private RandomAccessFile arand=null;
	/**
	 * IO helping class
	 */
	private IOHelper helper=null;
	
	/**
	 * Creates a new instance
	 * 
	 * @param params The VAFile parameters
	 * @param firstcreated wheter or not this index has been previously populated
	 * @throws Exception An error
	 */
	public PopulateIndex(VAFileParams params,boolean firstcreated) throws Exception{
		try{
			this.params=params;
			this.first=firstcreated;
			this.helper=new IOHelper();
			this.helper.setIDLength(params.getIDLength());
			this.helper.setVectorLength(params.getVectorLength());
		}catch(Exception e){
			logger.error("Could not initialize index population. throwing Exception",e);
			throw new Exception("Could not initialize index population");
		}
	}
	
	/**
	 * Open for population
	 * 
	 * @throws Exception An error
	 */
	public void open() throws Exception{
		try{
			vrand=new RandomAccessFile(FileHelper.getVAFVectorFile(params.getStorage(),params.getIndexID()),"rw");
			arand=new RandomAccessFile(FileHelper.getVAFApproxFile(params.getStorage(),params.getIndexID()),"rw");
			if(this.first) helper.writeHeader(params,vrand);
			vrand.seek(vrand.length());
			arand.seek(arand.length());
//			helper.skipHeader(vrand);
//			logger.info("position after opening "+vrand.getFilePointer());
//			logger.info("position after opening "+arand.getFilePointer());
		}catch(Exception e){
			logger.error("Could not open files to populate. Throwing Exception",e);
			throw new Exception ("Could not open files to populate");
		}
	}
	
	/**
	 * populate the index
	 * 
	 * @throws Exception An error
	 */
	public void populate() throws Exception{
		FileBufferReader reader=null;
		try{
			reader=new FileBufferReader(FileHelper.getVAFBufferFile(params.getStorage(),params.getIndexID()),params);
			reader.openForReading();
			long count=params.getElementCount();
			float []xsum=params.getXSum();
			float []x2sum=params.getX2Sum();
			while(true){
				FeatureVectorElement elem=reader.readElement();
				if(elem==null) break;
				byte []bitString=ApproximationGenerator.generate(params.getPartitionPoints(),elem.getVector(),params);
				long apstart=arand.getFilePointer();
				helper.writeApproximation(new ApproximationFileEntry(bitString,true),arand);
				long apstop=arand.getFilePointer();
				long vstart=vrand.getFilePointer();
				helper.writeRecord(new VectorFileEntry(elem.getId(),elem.getVector(),true),vrand);
				long vstop=vrand.getFilePointer();
				if(params.getApproximationEntryLength()==0) params.setApproximationEntryLength(apstop-apstart);
				else if(params.getApproximationEntryLength()!=(apstop-apstart)) throw new Exception("different approximation disk length");
				if(params.getVectorEntryLength()==0) params.setVectorEntryLength(vstop-vstart);
				else if(params.getVectorEntryLength()!=(vstop-vstart)) throw new Exception("different vector disk length");
				count+=1;
				for(int i=0;i<elem.getVector().length;i+=1){
					xsum[i]+=elem.getVector()[i];
					x2sum[i]+=(elem.getVector()[i]*elem.getVector()[i]);
				}
			}
			reader.close();
			params.setElementCount(count);
			params.setXSum(xsum);
			params.setX2Sum(x2sum);
			float []weights=params.getWeights();
			for(int i=0;i<params.getVectorLength();i+=1){
				weights[i]=((float)1)/((float)Math.sqrt(Math.abs((((float)x2sum[i])/((float)count))-((((float)xsum[i])/((float)count))*(((float)xsum[i])/((float)count))))));
//				logger.info(weights[i]);
			}
			params.setWeights(weights);
//			for(int i=0;i<ApproximationGenerator.dimCount.length;i+=1){
//				for(int q=0;q<ApproximationGenerator.dimCount[i].length;q+=1){
//					logger.info("dimCount["+i+"]["+q+"]="+ApproximationGenerator.dimCount[i][q]);
//				}
//			}
		}catch(Exception e){
			logger.error("Could not finish population. throwing Exception",e);
			if(reader!=null) reader.close();
			throw new Exception("Could not finish population");
		}
	}
	
	/**
	 * Close population
	 * 
	 * @throws Exception An error
	 */
	public void close() throws Exception{
		try{
			helper.writeHeader(params,vrand);
			vrand.getFD().sync();
			arand.getFD().sync();
			vrand.close();
			arand.close();
		}catch(Exception e){
			logger.error("Could not close population. throwing Exception",e);
			throw new Exception("Could not close population");
		}
	}
}
