package org.gcube.indexmanagement.bdbwrapper;

import gr.uoa.di.madgik.grs.buffer.IBuffer.Status;
import gr.uoa.di.madgik.grs.events.KeyValueEvent;
import gr.uoa.di.madgik.grs.record.GenericRecord;
import gr.uoa.di.madgik.grs.record.field.Field;
import gr.uoa.di.madgik.grs.record.field.StringField;
import gr.uoa.di.madgik.grs.writer.RecordWriter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Vector;
import java.util.Map.Entry;
import java.util.concurrent.TimeUnit;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.indexmanagement.bdbwrapper.BDBGcqlQueryContainer.SingleTerm;
import org.gcube.indexmanagement.common.ForwardIndexField;
import org.gcube.indexmanagement.common.ForwardIndexType;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.common.PropertyElementForwardIndex;
import org.gcube.indexmanagement.common.XMLTokenReplacer;
import org.gcube.indexmanagement.common.dupelimination.ResultsIdentifierSet;

import com.sleepycat.bind.tuple.FloatBinding;
import com.sleepycat.bind.tuple.IntegerBinding;
import com.sleepycat.bind.tuple.StringBinding;
import com.sleepycat.je.Cursor;
import com.sleepycat.je.Database;
import com.sleepycat.je.DatabaseEntry;
import com.sleepycat.je.OperationStatus;

public class BDBQueryExecutor extends Thread{
	
	static GCUBELog logger = new GCUBELog(BDBQueryExecutor.class);
	
	/** The Berkeley DB info that is needed to perform the query
	 * 
	 */
	private HashMap<String, Integer> keyTypes;
	private ForwardIndexType[] dbT;
	private Database[] db;
	private Database dbMain;
	private Comparator[] comparator;
	private LinkedHashMap<String, String> projections;
	
	/**
	 * indicates if the query has a "project distinct" part
	 */
	private boolean distinct = false;
	
	/**
	 * info for a specific query set
	 */
	private ArrayList<?> queries = null;
	private RecordWriter<GenericRecord> rsWriter;
	
	static final String delimiter = " AND ";
	static final String orderBy = " ORDERBY ";
	static final String prefix = "GET ";
	static final String gt = ">";
	static final String lt = "<";
	static final String eq = "=";
	static final String wild = "*";
	static final String termReplacer = "$";
	static final String termContainer = "\"";
	
	private static final long RSTIMEOUT = 10;
	
	public BDBQueryExecutor(HashMap<String, Integer> keyTypes,
			ForwardIndexType[] dbT, Database[] db, Database dbMain,
			Comparator[] comparator, ArrayList<?> queries,
			RecordWriter<GenericRecord> rsWriter, 
			LinkedHashMap<String, String> projections, boolean distinct) {
		super();
		this.keyTypes = keyTypes;
		this.dbT = dbT;
		this.db = db;
		this.dbMain = dbMain;
		this.comparator = comparator;
		this.rsWriter = rsWriter;
		this.queries = queries;
		this.projections = projections;
		this.distinct = distinct;
	}
	
	public void run() {
		
		long resultsRetrieved = 0;
		ResultsIdentifierSet resultsIdSet = new ResultsIdentifierSet();
		
		//run the queries one by one
		for(Object query : queries) {
			
			logger.debug(" >>> process Query");
			logger.debug("Query received: " +  query);
			
			LinkedHashMap<String,ArrayList<String>> keyNames = new LinkedHashMap<String, ArrayList<String>>();
			ArrayList<String> orderedList = new ArrayList<String>();
			try {
				//if the query is a String
				if(query instanceof ArrayList<?>) {
					//if the parsing indicated a query that returns an empty set
					if(parseQuery((ArrayList<BDBGcqlQueryContainer.SingleTerm>)query, keyNames, orderedList) == false)
						continue;
				} else if(query instanceof String) { 
					//if the parsing indicated a query that returns an empty set
					if(parseStringQuery((String)query, keyNames, orderedList) == false)
						continue;
				} else {
					throw new Exception("Unsupported type of queries");
				}
			} catch(Exception e) {
				logger.error("exception while parsing: ", e);
				return;
			}
			//currently ordering only on one field can be applied
			String ordered = null;
			if(orderedList.size() > 0)
				ordered = orderedList.get(0);
			
			try{
				//for each keyName find the key/ValueID pairs that satisfy the conditions
				ArrayList<HashSet<String>> ids = new ArrayList<HashSet<String>>(keyNames.size());
				ArrayList<LinkedHashMap<String,String>> pairs = new ArrayList<LinkedHashMap<String,String>>(keyNames.size());
				Iterator<String> iter = keyNames.keySet().iterator();
				Integer order = null;
				int j = 0;
				while(iter.hasNext())
				{
					//the two bounds
					DatabaseEntry greater = null;
					DatabaseEntry lower = null;
					boolean gequal = false;
					boolean lequal = false;
					//current keyName
					String keyName = iter.next();
					if(ordered != null && ordered.equals(keyName))
						order = j;
					Integer pos = keyTypes.get(keyName);
					//define the bounds
					ArrayList<String> args = keyNames.get(keyName);
					int index = 0;
					while(true){
						if(args.get(index).equals(gt))
						{
							greater = new DatabaseEntry();
    						if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.DATE)
    							greater.setData(BDBWrapper.convertToDate(args.get(index+2), dbT[pos].getKeyField().getDataTypeFormat()).getBytes());
    						else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.FLOAT)
    							FloatBinding.floatToEntry(BDBWrapper.convertToFloat(args.get(index+2)), greater);
    						else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.INT)      
    							IntegerBinding.intToEntry(BDBWrapper.convertToInt(args.get(index+2)), greater);
    						else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.STRING)        
    							StringBinding.stringToEntry(args.get(index+2), greater);
    						if(args.get(index+1).equals("1"))
    							gequal = true;
    						else
    							gequal = false;
						}else if(args.get(index).equals(lt)){
							lower =  new DatabaseEntry();
    						if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.DATE)
    							lower.setData(BDBWrapper.convertToDate(args.get(index+2), dbT[pos].getKeyField().getDataTypeFormat()).getBytes());
    						else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.FLOAT)
    							FloatBinding.floatToEntry(BDBWrapper.convertToFloat(args.get(index+2)), lower);
    						else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.INT)      
    							IntegerBinding.intToEntry(BDBWrapper.convertToInt(args.get(index+2)), lower);
    						else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.STRING)        
    							StringBinding.stringToEntry(args.get(index+2), lower);
    						if(args.get(index+1).equals("1"))
    							lequal = true;
    						else
    							lequal = false;
						}else if(args.get(index).equals(eq)){
							//check for wildcard
							if(args.get(index+1).equals("1"))
							{
								greater =  new DatabaseEntry();
								if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.DATE)
									greater.setData(BDBWrapper.convertToDate(args.get(index+2), dbT[pos].getKeyField().getDataTypeFormat()).getBytes());
								else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.FLOAT)
									FloatBinding.floatToEntry(BDBWrapper.convertToFloat(args.get(index+2)), greater);
								else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.INT)      
									IntegerBinding.intToEntry(BDBWrapper.convertToInt(args.get(index+2)), greater);
								else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.STRING)        
									StringBinding.stringToEntry(args.get(index+2), greater);
								lower = greater;
								gequal = true;
								lequal = true;
							}
    						break;
						}
						if(index == 3)
							break;
						else if(args.size()==6)
							index = 3;
						else
							break;    							
					}
					//scan the corresponding index to retrieve pairs
					DatabaseEntry keyValue = new DatabaseEntry();
					DatabaseEntry bindKey = new DatabaseEntry();
					Object key = null;
					OperationStatus opSt = OperationStatus.SUCCESS;
					//new cursor for this index
					Cursor cursor = db[pos].openCursor(null, null);
					try{    						
					//Move the cursor to a specific point if we want results greater than a key
					if(greater != null)
					{
						keyValue.setData(greater.getData());
						opSt = cursor.getSearchKeyRange(keyValue, bindKey, null);
						//if we don't want equal keys
						if(!gequal)
						{
							if(comparator[pos].compare(keyValue.getData(), greater.getData()) == 0)
							{
								opSt = cursor.getNextNoDup(keyValue, bindKey, null);
							}
						}
					}else{
						opSt = cursor.getNext(keyValue, bindKey, null);
					}
					HashSet<String> set = new HashSet<String>();
					LinkedHashMap<String, String> map = new LinkedHashMap<String, String>();
					while(opSt == OperationStatus.SUCCESS)
					{
						//if we don't want results greater than lower
						if(lower != null)
						{
							if(comparator[pos].compare(keyValue.getData(), lower.getData()) > 0)
								break;
							if(comparator[pos].compare(keyValue.getData(), lower.getData())==0 && !lequal)
								break;
						}
						String id = StringBinding.entryToString(bindKey);
						//retrieve the key value as a String
						if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.DATE)
				        	key = CustomDate.toCustomDate(keyValue.getData());
				        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.FLOAT)
				        	key = FloatBinding.entryToFloat(keyValue); 
				        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.INT)      
				        	key = IntegerBinding.entryToInt(keyValue);
				        else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.STRING)        
				        	key = StringBinding.entryToString(keyValue);
						String resolvedKey = XMLTokenReplacer.XMLResolve(key.toString());
						//retrieve the key for this id
						String keyString = map.get(id);
						if(keyString == null)
							keyString = "";
						keyString += "<key><keyname>" + keyName + "</keyname><keyvalue>" + resolvedKey + "</keyvalue></key>";
						map.put(id, keyString);
						set.add(id);
						logger.debug("Found key/id:"+resolvedKey+"/"+id+" for keyname:"+keyName);
						//get the next key from index
						opSt = cursor.getNext(keyValue, bindKey, null);    							
					}
					//store ids and key/id pairs
					ids.add(set);
					pairs.add(map);
					//close the cursor
					cursor.close();
					}catch(Exception e){
						cursor.close();
						throw e;
					}
					//indicates the step
					j++;
				}
				//calculate the intersection of all the id sets
				//sort the id sets based on their size
				Collections.sort(ids, new Comparator<HashSet<String>>(){
					public int compare(HashSet<String> a, HashSet<String> b) {
					  int sdif = a.size() - b.size();
					  return sdif;
					 }});
				//intersection of sets
				HashSet<String> intersection = ids.get(0);
				for(int i=1; i<ids.size(); i++)
					intersection.retainAll(ids.get(i));
				//retrieve values for ids in the intersection
				int next = 0;
				boolean firstStep = true;
				if(ordered != null)
					next = order;
				LinkedHashMap<String, String> masterMap = null;
				//scan all sets of pairs
				for(int i=0; i<pairs.size(); i++)
				{
					//if we are in the first step
					if(firstStep)
					{
						masterMap = pairs.get(next);
						//leave only value ids that are contained in the intersection
						Iterator<String> valueIter = masterMap.keySet().iterator();
						while(valueIter.hasNext())
						{
							if(!intersection.contains(valueIter.next()))
								valueIter.remove();
						}
						//the next map to be processed
						if(next != 0)
							next = 0;
						else
							next = 1; 
						firstStep = false;
					}else{
						Iterator<String> valueIter = intersection.iterator();
						//loop through the elements of the intersection
						while(valueIter.hasNext())
						{
							String id = valueIter.next();
							//if there are keys for this id add them
							if(pairs.get(next).containsKey(id))
							{
								String keyString = pairs.get(next).get(id);
								String keyString2 = masterMap.get(id);
								keyString2 += keyString;
								masterMap.put(id, keyString2);
								logger.debug("Added key:"+keyString+" for id:"+id);
							}
						}
						//the next map to be processed
						next++;
						if(ordered != null && next == order)
							next++;
						
					}    						
				}
				//create the ResultSet
				Iterator<String> valueIter = masterMap.keySet().iterator();
				while(valueIter.hasNext())
				{
					DatabaseEntry idEntry = new DatabaseEntry();
					DatabaseEntry dataEntry = new DatabaseEntry();
					String id = valueIter.next();
					//the keys for this id
					String key = masterMap.get(id);
					StringBinding.stringToEntry(id, idEntry);
					//the value for this id
					if(dbMain.get(null, idEntry, dataEntry, null) == OperationStatus.NOTFOUND)
					{
						logger.error("Reference to actual data not found. Index is corrupted!");
						throw new Exception("Reference to actual data not found. Index is corrupted!");
					}
					String data = BDBWrapper.extractValueFromEntry(StringBinding.entryToString(dataEntry));
					String stringXML = XMLTokenReplacer.XMLResolve(data);
		    		
					//get the map of presentable fields for this document
					HashMap<String, String> fieldsPresentable = BDBWrapper.getPresentationFieldsFromXML(stringXML);
					
					//while the reader hasn't stopped reading
                    if(rsWriter.getStatus() != Status.Open) {
                    	rsWriter.emit(new KeyValueEvent(IndexType.RESULTSNOFINAL_EVENT, "" + resultsRetrieved));
                    	if(rsWriter.getStatus() != Status.Dispose) {
                        	rsWriter.close();
                    	}
                    	return;
                    }
					
                    //if there is a project distinct for this query
                    boolean canSend = false;
                    GenericRecord rec = null;
                    if(distinct) {
                    	
                    	rec = createRecordFromDistinctProjections(fieldsPresentable, resultsIdSet);
                    	
                    	if(rec != null)
                    		canSend = true;
                    	
                    //if there is no indication for distinct	
                    } else {
						
                    	rec = createRecordFromProjections(fieldsPresentable);
						
                    	canSend = resultsIdSet.canSend(fieldsPresentable.get(IndexType.LANGUAGE_FIELD), 
    							fieldsPresentable.get(IndexType.DOCID_FIELD));					
                    }
                    
                    if(canSend) {
					
						resultsRetrieved++;
						if(IndexType.sendIndication(resultsRetrieved)) {
							rsWriter.emit(new KeyValueEvent(IndexType.RESULTSNO_EVENT, "" + resultsRetrieved));
						}
						
						while(!rsWriter.put(rec, RSTIMEOUT, TimeUnit.SECONDS)) {
		                   	//while the reader hasn't stopped reading
							 if(rsWriter.getStatus() != Status.Open) {
								rsWriter.emit(new KeyValueEvent(IndexType.RESULTSNOFINAL_EVENT, "" + resultsRetrieved));
								if(rsWriter.getStatus() != Status.Dispose) {
			                       	rsWriter.close();
			                    }
			                    return;
			                 }
		                }					
					}
					
				}
				
			}
			catch(Exception e){
				logger.error(" *** exception in process Query: ", e);
			}
			logger.debug(" <<< process Query");
		}
		
		try {
			rsWriter.emit(new KeyValueEvent(IndexType.RESULTSNOFINAL_EVENT, "" + resultsRetrieved));
			if(rsWriter.getStatus() != Status.Dispose) {
               	rsWriter.close();
            }
		}
		catch (Exception e1) { logger.error("could not close RSWriter", e1); }
		
		logger.debug(" <<< processed All Queries");
	}
	
	private GenericRecord createRecordFromDistinctProjections(
			HashMap<String, String> fieldsPresentable,
			ResultsIdentifierSet resultsIdSet) {
		
		GenericRecord rec =  new GenericRecord();
		ArrayList<Field> fields = new ArrayList<Field>();
		
		//for the projected fields
		StringBuilder concatProjections = new StringBuilder();
		for(Entry<String, String> current : this.projections.entrySet()) {
			
			String proj = current.getValue();
			//get this projection
			String value = fieldsPresentable.get(proj);
			if(value == null) 
				value = "";
			
			concatProjections.append(value.trim()).append("-");
			
			fields.add(new StringField(value));
		}
		
		if(resultsIdSet.canSend("DISTINCT", concatProjections.toString())) {
			
			rec.setFields(fields.toArray(new Field[fields.size()]));
			return rec;
			
		} else {
			
			return null;
			
		}
	}

	private GenericRecord createRecordFromProjections(
			HashMap<String, String> fieldsPresentable) {
		
		GenericRecord rec =  new GenericRecord();
		ArrayList<Field> fields = new ArrayList<Field>();
		
		//set the docID in the first field
		String id = fieldsPresentable.get(IndexType.DOCID_FIELD);
		if(id == null)
			id = BDBWrapper.NODOCID;
		fields.add(new StringField(id));
		
		logger.trace("returning doc with ID: " + id);
			
		//add the projected fields
		for(Entry<String, String> current : this.projections.entrySet()) {
			
			String proj = current.getValue();
			//get this projection
			String value = fieldsPresentable.get(proj);
			if(value == null) 
				value = "";
			
			fields.add(new StringField(value));
		}
		
		rec.setFields(fields.toArray(new Field[fields.size()]));
		
		return rec;
	}

	private boolean parseQuery(ArrayList<SingleTerm> query,
			LinkedHashMap<String, ArrayList<String>> keyNames,
			ArrayList<String> orderedList) throws Exception{
		//for each condition find the keyName to which it refers
		for(SingleTerm condition : query)
		{
			logger.trace("Condition - Field: " + condition.getField() + ", Relation: " + condition.getRelation() + ", Value: " + condition.getValue());
			
			if(condition.getRelation().startsWith(gt))
			{
				String keyName = condition.getField();
				if(keyTypes.get(keyName) == null)
				{
					throw new IndexException("keyName: " + keyName + " was not declared when the constructor was called. " +
					"There is no index for this keyName");
				}
				//store the arguments for this condition
				ArrayList<String> args = keyNames.get(keyName);
				if(args == null)
					args = new ArrayList<String>(3);
				
				//the type of the new condition is gt
				String type = gt;
				//the term of the new condition
				String term;
				//equality flag
				String equals;
				//check for equality
				if(!condition.getRelation().equals(gt))
				{
					equals = "1";
					//store the bound
					String bound = condition.getValue();
					if(bound.equals(wild))
						throw new Exception("for a non equality condition the wildcard is used ");
					term = bound;
				}else{
					equals = "0";
					//store the bound
					String bound = condition.getValue();
					if(bound.equals(wild))
						throw new Exception("for a non equality condition the wildcard is used: ");
					term = bound;
				}
				
				//store the arguments and check how they relate to previously stored arguments
				try {
					if(!checkAndStore(keyName, args, type, equals, term)) {
						logger.info("Contradicting new bound for keyName: " 
								+ keyName + ", Type, equals, term: " 
								+ type + ", " + equals + ", " +  term);
						return false;
					}    									
				} catch (Exception e) {
					logger.fatal("Something that was not predicted happened: ", e);
					throw new Exception("Something that was not predicted happened: ", e);
				}
				
				logger.debug("Stored condition for keyName: " + keyName + ", term: " + term);
				//store arguments
				keyNames.put(keyName, args);
			}else if(condition.getRelation().startsWith(lt))
			{
				String keyName = condition.getField();
				if(keyTypes.get(keyName) == null)
				{
					throw new IndexException("keyName: " + keyName + " was not declared when the constructor was called. " +
					"There is no index for this keyName");
				}
				//store the arguments for this condition
				ArrayList<String> args = keyNames.get(keyName);
				if(args == null)
					args = new ArrayList<String>(3);
				//the type of the new condition is lt
				String type = lt;
				//the term of the new condition
				String term;
				//equality flag
				String equals;
				
				//check for equality
				if(!condition.getRelation().equals(lt))
				{
					equals = "1";
					//store the bound
					String bound = condition.getValue();
					if(bound.equals(wild))
						throw new Exception("for a non equality condition the wildcard is used");
					term = bound;
				}else{
					equals = "0";
					//store the bound
					String bound = condition.getValue();
					if(bound.equals(wild))
						throw new Exception("for a non equality condition the wildcard is used");
					term = bound;
				}
				
				//store the arguments and check how they relate to previously stored arguments
				try {
					if(!checkAndStore(keyName, args, type, equals, term)) {
						logger.info("Contradicting new bound for keyName: " 
								+ keyName + ", Type, equals, term: " 
								+ type + ", " + equals + ", " +  term);
						return false;
					}    									
				} catch (Exception e) {
					logger.fatal("Something that was not predicted happened: ", e);
					throw new Exception("Something that was not predicted happened: ", e);
				}
				
				logger.debug("Stored condition for keyName: " + keyName + ", term: " + term);
				//store arguments
				keyNames.put(keyName, args);
			}else if(condition.getRelation().equals(eq))
			{
				String keyName = condition.getField();
				if(keyTypes.get(keyName) == null)
				{
					throw new IndexException("keyName: " + keyName + " was not declared when the constructor was called. " +
					"There is no index for this keyName");
				}
				//store the arguments for this condition
				ArrayList<String> args = keyNames.get(keyName);
				if(args == null)
					args = new ArrayList<String>(3);	    							
				
				//the type of the new condition is eq
				String type = eq;
				//the term of the new condition
				String term;
				//equality flag
				String equals;
				
				String bound = condition.getValue(); 
				//check for wildcard
				if(bound.equals(wild))
				{
					equals = "0";
					term = bound;
				}else{
					equals = "1";
					term = bound;
				}
				
				//store the arguments and check how they relate to previously stored arguments
				try {
					if(!checkAndStore(keyName, args, type, equals, term)) {
						logger.info("Contradicting new bound for keyName: " 
								+ keyName + ", Type, equals, term: " 
								+ type + ", " + equals + ", " +  term);
						return false;
					}    									
				} catch (Exception e) {
					logger.fatal("Something that was not predicted happened: ", e);
					throw new Exception("Something that was not predicted happened: ", e);
				}
				
				logger.debug("Stored condition for keyName: " + keyName + ", term: " + term);
				//store arguments
				keyNames.put(keyName, args);
			}
		}
		return true;
	}

	private boolean parseStringQuery(String query, LinkedHashMap<String,ArrayList<String>> keyNames, ArrayList<String> orderedList) throws Exception{
		
			//parse the query
			//find the conditions separated by the delimiter
			Vector<String> conditions = new Vector<String>();
			StringBuffer q = new StringBuffer(query.trim());
			String ordered = null;
			//check if there is the prefix instruction
			if(q.indexOf(prefix) != 0)
			{
				throw new Exception("Prefix:" + prefix + " is missing - Invalid query expression: " + query);
			}
			//cut the prefix instruction
			q.delete(0,q.indexOf(prefix)+ prefix.length());
			//check how many are the occurrences of the termContainer character
			if(BDBWrapper.numberOfChars(q.toString(), termContainer)%2 == 1)
			{
				throw new Exception("Unmatched " + termContainer + " character - Invalid query expression: " + query);
			}
			//replace the terms contained in termContainer chars with $
			int current = q.indexOf(termContainer);
			int current2;
			ArrayList<String> terms = new ArrayList<String>();
			while(current != -1)
			{
				//there will certainly be a matching termContainer char
				current2 = q.indexOf(termContainer, current+1);
				//add the first term in the terms list
				terms.add(q.substring(current+1, current2));
				//replace this term with the termReplacer
				q.delete(current, current2+1);
				q.insert(current, termReplacer);
				current = q.indexOf(termContainer);
			}
			logger.debug("Query after replacing terms: " + q);
			//check if there is an orderBy instruction
			if(q.indexOf(orderBy) != -1)
			{
				ordered = q.substring(q.indexOf(orderBy)+ orderBy.length()).trim();
				orderedList.add(ordered);
				q.delete(q.indexOf(orderBy), q.length()); 
				if(keyTypes.get(ordered) == null)
				{
					throw new IndexException("keyName: " + ordered + " was not declared when the constructor was called. " +
					"There is no index for this keyName");
				}
			}
			logger.debug("Query after removing orderby: " + q);
			//store the conditions
			while(q.indexOf(delimiter) != -1)
			{
				conditions.add(q.substring(0, q.indexOf(delimiter)).trim());
				q.delete(0, q.indexOf(delimiter)+delimiter.length());
			}
			conditions.add(q.toString().trim());
			//for each condition find the keyName to which it refers
			for(int i=0; i< conditions.size(); i++)
			{
				if(conditions.get(i).indexOf(gt)==-1 && conditions.get(i).indexOf(lt)==-1 && conditions.get(i).indexOf(eq)==-1)
					throw new Exception("Invalid query expression: " + query);
				if(conditions.get(i).indexOf(gt)!=-1)
				{
					String keyName = conditions.get(i).substring(0, conditions.get(i).indexOf(gt)).trim();
					if(keyTypes.get(keyName) == null)
					{
						throw new IndexException("keyName: " + keyName + " was not declared when the constructor was called. " +
						"There is no index for this keyName");
					}
					//store the arguments for this condition
					ArrayList<String> args = keyNames.get(keyName);
					if(args == null)
						args = new ArrayList<String>(3);
					
					//the type of the new condition is gt
					String type = gt;
					//the term of the new condition
					String term;
					//equality flag
					String equals;
					//check for equality
					if(conditions.get(i).substring(conditions.get(i).indexOf(gt) + gt.length(), conditions.get(i).indexOf(gt) + gt.length() + eq.length()).equals(eq))
					{
						equals = "1";
						//store the bound
						String bound = conditions.get(i).substring(conditions.get(i).indexOf(gt) + gt.length() + eq.length()).trim();
						if(bound.equals(wild))
							throw new Exception("for a non equality condition the wildcard is used: " + conditions.get(i));
						term = terms.remove(0);
					}else{
						equals = "0";
						//store the bound
						String bound = conditions.get(i).substring(conditions.get(i).indexOf(gt) + gt.length()).trim();
						if(bound.equals(wild))
							throw new Exception("for a non equality condition the wildcard is used: " + conditions.get(i));
						term = terms.remove(0);
					}
					
					//store the arguments and check how they relate to previously stored arguments
					try {
						if(!checkAndStore(keyName, args, type, equals, term)) {
							logger.info("Contradicting new bound for keyName: " 
									+ keyName + ", Type, equals, term: " 
									+ type + ", " + equals + ", " +  term);
							return false;
						}    									
					} catch (Exception e) {
						logger.fatal("Something that was not predicted happened: ", e);
						throw new Exception("Something that was not predicted happened: ", e);
					}
					
					logger.debug("Stored condition for keyName: " + keyName + ", term: " + term);
					//store arguments
					keyNames.put(keyName, args);
				}else if(conditions.get(i).indexOf(lt)!=-1)
				{
					String keyName = conditions.get(i).substring(0, conditions.get(i).indexOf(lt)).trim();
					if(keyTypes.get(keyName) == null)
					{
						throw new IndexException("keyName: " + keyName + " was not declared when the constructor was called. " +
						"There is no index for this keyName");
					}
					//store the arguments for this condition
					ArrayList<String> args = keyNames.get(keyName);
					if(args == null)
						args = new ArrayList<String>(3);
					//the type of the new condition is lt
					String type = lt;
					//the term of the new condition
					String term;
					//equality flag
					String equals;
					
					//check for equality
					if(conditions.get(i).substring(conditions.get(i).indexOf(lt) + lt.length(), conditions.get(i).indexOf(lt) + lt.length() + eq.length()).equals(eq))
					{
						equals = "1";
						//store the bound
						String bound = conditions.get(i).substring(conditions.get(i).indexOf(lt) + lt.length() + eq.length()).trim();
						if(bound.equals(wild))
							throw new Exception("for a non equality condition the wildcard is used: " + conditions.get(i));
						term = terms.remove(0);
					}else{
						equals = "0";
						//store the bound
						String bound = conditions.get(i).substring(conditions.get(i).indexOf(lt) + lt.length()).trim();
						if(bound.equals(wild))
							throw new Exception("for a non equality condition the wildcard is used: " + conditions.get(i));
						term = terms.remove(0);
					}
					
					//store the arguments and check how they relate to previously stored arguments
					try {
						if(!checkAndStore(keyName, args, type, equals, term)) {
							logger.info("Contradicting new bound for keyName: " 
									+ keyName + ", Type, equals, term: " 
									+ type + ", " + equals + ", " +  term);
							return false;
						}    									
					} catch (Exception e) {
						logger.fatal("Something that was not predicted happened: ", e);
						throw new Exception("Something that was not predicted happened: ", e);
					}
					
					logger.debug("Stored condition for keyName: " + keyName + ", term: " + term);
					//store arguments
					keyNames.put(keyName, args);
				}else if(conditions.get(i).indexOf(eq)!=-1)
				{
					String keyName = conditions.get(i).substring(0, conditions.get(i).indexOf(eq)).trim();
					if(keyTypes.get(keyName) == null)
					{
						throw new IndexException("keyName: " + keyName + " was not declared when the constructor was called. " +
						"There is no index for this keyName");
					}
					//store the arguments for this condition
					ArrayList<String> args = keyNames.get(keyName);
					if(args == null)
						args = new ArrayList<String>(3);	    							
					
					//the type of the new condition is eq
					String type = eq;
					//the term of the new condition
					String term;
					//equality flag
					String equals;
					
					String bound = conditions.get(i).substring(conditions.get(i).indexOf(eq) + eq.length()).trim(); 
					//check for wildcard
					if(bound.equals(wild))
					{
						equals = "0";
						term = bound;
					}else{
						equals = "1";
						term = terms.remove(0);
					}
					
					//store the arguments and check how they relate to previously stored arguments
					try {
						if(!checkAndStore(keyName, args, type, equals, term)) {
							logger.info("Contradicting new bound for keyName: " 
									+ keyName + ", Type, equals, term: " 
									+ type + ", " + equals + ", " +  term);
							return false;
						}    									
					} catch (Exception e) {
						logger.fatal("Something that was not predicted happened: ", e);
						throw new Exception("Something that was not predicted happened: ", e);
					}
					
					logger.debug("Stored condition for keyName: " + keyName + ", term: " + term);
					//store arguments
					keyNames.put(keyName, args);
				}
			}
			return true;
	}

	//returns false if the new bound is contradicting with the old ones
	//stores the new bound and returns true otherwise
	//throws Exception in case something that was not predicted, happened
	private boolean checkAndStore(String keyName, ArrayList<String> args, String type,
			String equals, String term) throws Exception{
		
		//in case the new bound is a wild card, there is no meaning to do anything
		if(term.equals(wild)) {
			logger.trace("New bound is a wildcard");
			//if nothing is stored yet, store the wildcard condition
			if(args.size() == 0)
			{
				args.add(type);
				args.add(equals);
				args.add(term);
			}
			return true;
		}
		//in case there is an old bound and it is a wild card, we must clear args and add the new bound
		if(args.size() == 3 && args.get(2).equals(wild)) {
			args.clear();
			args.add(type);
			args.add(equals);
			args.add(term);
			return true;
		}					
		
		//if there is nothing stored yet
		if(args.size() == 0)
		{
			args.add(type);
			args.add(equals);
			args.add(term);
			return true;
		//if there is one bound stored	
		} else if(args.size() == 3) {
			//the previous bound info
			String prevType = args.get(0);
			String prevEquals =  args.get(1);
			String prevTerm = args.get(2);
			
			//compare the terms
			int comp = compare(keyName, prevTerm, term);
			//if the terms and types are equal
			if(type.equals(prevType) && comp==0) {
				//in case of equality there is nothing more to do
				if(type.equals(eq))
					return true;
				//else check if we must change the equality flag
				if(equals.equals("0"))
				{
					args.remove(1);
					args.add(1, "0");
				}
				return true;
			}
			
			//if we have two gts
			if(prevType.equals(gt) && type.equals(gt)) {
				//if previous is less than the new, replace the previous with the new one
				if(comp < 0) {
					args.clear();
					args.add(type);
					args.add(equals);
					args.add(term);
					return true;
				//nothing to do	
				} else if(comp > 0) {
					return true;
				}
			//if we have two lts	
			} else if (prevType.equals(lt) && type.equals(lt)){
				//if new is less than the previous, replace the previous with the new one
				if(comp > 0) {
					args.clear();
					args.add(type);
					args.add(equals);
					args.add(term);
					return true;
				//nothing to do	
				} else if(comp < 0) {
					return true;
				}
			//if we have a gt and a lt	
			} else if (prevType.equals(gt) && type.equals(lt)){
				//the only case that these two are contradicting 
				//is the gt to be greater than the lt
				if(comp > 0) {
					return false;
				}
				//else add the new bound
				args.add(type);
				args.add(equals);
				args.add(term);
				return true;
			
			//if we have an lt and a gt		
			} else if (prevType.equals(lt) && type.equals(gt)){
				//the only case that these two are contradicting 
				//is the gt to be greater than the lt
				if(comp < 0) {
					return false;
				}
				//else add the new bound
				args.add(type);
				args.add(equals);
				args.add(term);
				return true;
				
			//if we have an eq and a gt	
			} else if (prevType.equals(eq) && type.equals(gt)){
				//the only case that these two are contradicting 
				//is the gt to be greater than the eq
				if(comp < 0) {
					return false;
				}
				//else there is nothing to do
				return true;
			
			//if we have an eq and an lt
			} else if (prevType.equals(eq) && type.equals(lt)){
				//the only case that these two are contradicting 
				//is the lt to be less than the eq
				if(comp > 0) {
					return false;
				}
				//else there is nothing to do
				return true;
			
			//if we have two eqs
			} else if (prevType.equals(eq) && type.equals(eq)){
				//the only case that these two are contradicting 
				//is the terms are not equal
				if(comp != 0) {
					return false;
				}
				//the case with same types and comp == 0 is examined before
				logger.error("The case with same types and comp == 0 is examined before." + 
						" This point should not have reached. Possible Bug. ");
				throw new Exception("The case with same types and comp == 0 is examined before." + 
						" This point should not have reached. Possible Bug. ");
				
			//if we have a eq and a gt
			} else if (prevType.equals(gt) && type.equals(eq)){ 
				//the only case that these two are contradicting 
				//is the gt to be greater than the eq
				if(comp > 0) {
					return false;
				}
				//replace with the new eq bound
				args.clear();
				args.add(type);
				args.add(equals);
				args.add(term);
				return true;
				
			//if we have a eq and a lt
			} else if (prevType.equals(lt) && type.equals(eq)){
				//the only case that these two are contradicting 
				//is the lt to be less than the eq
				if(comp < 0) {
					return false;
				}
				//replace with the new eq bound
				args.clear();
				args.add(type);
				args.add(equals);
				args.add(term);
				return true;
				
			}
		//if there are two bounds	
		} else if(args.size() == 6) {
			//the previous gt and lt bound info
			int gtIndex = 0;
			int ltIndex = 3;
			if(args.get(0).equals(lt)){
				gtIndex = 3;
				ltIndex = 0;
			}
			String gtEquals =  args.get(gtIndex + 1);
			String gtTerm = args.get(gtIndex + 2);
			String ltEquals =  args.get(ltIndex + 1);
			String ltTerm = args.get(ltIndex + 2);
			int gtComp = compare(keyName, gtTerm, term);
			int ltComp = compare(keyName, ltTerm, term);
			
			//if the new bound is eq
			if(type.equals(eq)) {
				//in case gt is  greater than eq
				if(gtComp > 0) {
					return false;
				}
				//in case lt is less than eq
				if(ltComp < 0) {
					return false;
				}
				//since the bounds are not contradicting
				//clear the old bounds and store the eq
				args.clear();
				args.add(type);
				args.add(equals);
				args.add(term);
				return true;
				
			//if the new bound is lt	
			} else if(type.equals(lt)) {
				//if new lt is greater than the old lt, there is nothing to do
				if(ltComp < 0) {
					return true;
				}
				//if old and new lt are equal and the old ltEquals is "0" there is nothing to do
				if(ltComp == 0 && ltEquals.equals("0")) {
					return true;
				}
				//if new lt is less than the old gt, the bounds are contradicting
				if(gtComp > 0) {
					return false;
				}
				//the new lt must replace the old one
				//remove type
				args.remove(ltIndex);
				//remove equals
				args.remove(ltIndex);
				//remove term
				args.remove(ltIndex);
				//add the new (in the reverse order)
				args.add(ltIndex, term);
				args.add(ltIndex, equals);
				args.add(ltIndex, type);
				return true;
				
			//if the new bound is gt	
			} else if(type.equals(gt)) {
				//if new gt is less than the old gt, there is nothing to do
				if(gtComp > 0) {
					return true;
				}
				//if old and new gt are equal and the old gtEquals is "0" there is nothing to do
				if(gtComp == 0 && gtEquals.equals("0")) {
					return true;
				}
				//if new gt is greater than the old lt, the bounds are contradicting
				if(ltComp < 0) {
					return false;
				}
				//the new gt must replace the old one
				//remove type
				args.remove(gtIndex);
				//remove equals
				args.remove(gtIndex);
				//remove term
				args.remove(gtIndex);
				//add the new (in the reverse order)
				args.add(gtIndex, term);
				args.add(gtIndex, equals);
				args.add(gtIndex, type);
				return true;
			}						
			
		//this means we have a bug at parsing	
		} else {
			throw new Exception("Bug. args size = " + args.size());
		}
		//this point should not have been reached
		logger.warn("Normally this point should not have been reached");
		return true;
	}
	
	//compare two terms based on their type
	private int compare(String keyName, String term1, String term2) throws Exception{
		//find where info for this keyName is stored
        int pos = keyTypes.get(keyName);
        if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.DATE)
        {
        	CustomDate t1 = BDBWrapper.convertToDate(term1, dbT[pos].getKeyField().getDataTypeFormat());
        	CustomDate t2 = BDBWrapper.convertToDate(term2, dbT[pos].getKeyField().getDataTypeFormat());
        	return t1.compareTo(t2);
        } else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.FLOAT) {
        	Float t1 = new Float(term1);
        	Float t2 = new Float(term2);
        	return t1.compareTo(t2);
        } else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.INT) {      
        	Integer t1 = new Integer(term1);
        	Integer t2 = new Integer(term2);
        	return t1.compareTo(t2);
        } else if(dbT[pos].getKeyField().getDataType() == ForwardIndexField.DataType.STRING) {        
        	return term1.compareTo(term2);
        }
        throw new Exception("Unsupported type: " + dbT[pos].getKeyField().getDataType() 
        		+ ", for keyName: " + keyName);
	}
	
	//for test purposes
	//compare two terms based on their type
	private static int compareTestLocally(String keyName, String term1, String term2) throws Exception{
		return term1.compareTo(term2);
	}
	
	//for test purposes - package visibility
	static boolean parseQueryTestLocally(ArrayList<SingleTerm> query,
			LinkedHashMap<String, ArrayList<String>> keyNames,
			ArrayList<String> orderedList) throws Exception{
		//for each condition find the keyName to which it refers
		for(SingleTerm condition : query)
		{
			if(condition.getRelation().startsWith(gt))
			{
				String keyName = condition.getField();
				//store the arguments for this condition
				ArrayList<String> args = keyNames.get(keyName);
				if(args == null)
					args = new ArrayList<String>(3);
				
				//the type of the new condition is gt
				String type = gt;
				//the term of the new condition
				String term;
				//equality flag
				String equals;
				//check for equality
				if(!condition.getRelation().equals(gt))
				{
					equals = "1";
					//store the bound
					String bound = condition.getValue();
					if(bound.equals(wild))
						throw new Exception("for a non equality condition the wildcard is used ");
					term = bound;
				}else{
					equals = "0";
					//store the bound
					String bound = condition.getValue();
					if(bound.equals(wild))
						throw new Exception("for a non equality condition the wildcard is used: ");
					term = bound;
				}
				
				//store the arguments and check how they relate to previously stored arguments
				try {
					if(!checkAndStoreTestLocally(keyName, args, type, equals, term)) {
						logger.info("Contradicting new bound for keyName: " 
								+ keyName + ", Type, equals, term: " 
								+ type + ", " + equals + ", " +  term);
						return false;
					}    									
				} catch (Exception e) {
					logger.fatal("Something that was not predicted happened: ", e);
					throw new Exception("Something that was not predicted happened: ", e);
				}
				
				logger.debug("Stored condition for keyName: " + keyName + ", term: " + term);
				//store arguments
				keyNames.put(keyName, args);
			}else if(condition.getRelation().startsWith(lt))
			{
				String keyName = condition.getField();
				//store the arguments for this condition
				ArrayList<String> args = keyNames.get(keyName);
				if(args == null)
					args = new ArrayList<String>(3);
				//the type of the new condition is lt
				String type = lt;
				//the term of the new condition
				String term;
				//equality flag
				String equals;
				
				//check for equality
				if(!condition.getRelation().equals(lt))
				{
					equals = "1";
					//store the bound
					String bound = condition.getValue();
					if(bound.equals(wild))
						throw new Exception("for a non equality condition the wildcard is used");
					term = bound;
				}else{
					equals = "0";
					//store the bound
					String bound = condition.getValue();
					if(bound.equals(wild))
						throw new Exception("for a non equality condition the wildcard is used");
					term = bound;
				}
				
				//store the arguments and check how they relate to previously stored arguments
				try {
					if(!checkAndStoreTestLocally(keyName, args, type, equals, term)) {
						logger.info("Contradicting new bound for keyName: " 
								+ keyName + ", Type, equals, term: " 
								+ type + ", " + equals + ", " +  term);
						return false;
					}    									
				} catch (Exception e) {
					logger.fatal("Something that was not predicted happened: ", e);
					throw new Exception("Something that was not predicted happened: ", e);
				}
				
				logger.debug("Stored condition for keyName: " + keyName + ", term: " + term);
				//store arguments
				keyNames.put(keyName, args);
			}else if(condition.getRelation().equals(eq))
			{
				String keyName = condition.getField();
				//store the arguments for this condition
				ArrayList<String> args = keyNames.get(keyName);
				if(args == null)
					args = new ArrayList<String>(3);	    							
				
				//the type of the new condition is eq
				String type = eq;
				//the term of the new condition
				String term;
				//equality flag
				String equals;
				
				String bound = condition.getValue(); 
				//check for wildcard
				if(bound.equals(wild))
				{
					equals = "0";
					term = bound;
				}else{
					equals = "1";
					term = bound;
				}
				
				//store the arguments and check how they relate to previously stored arguments
				try {
					if(!checkAndStoreTestLocally(keyName, args, type, equals, term)) {
						logger.info("Contradicting new bound for keyName: " 
								+ keyName + ", Type, equals, term: " 
								+ type + ", " + equals + ", " +  term);
						return false;
					}    									
				} catch (Exception e) {
					logger.fatal("Something that was not predicted happened: ", e);
					throw new Exception("Something that was not predicted happened: ", e);
				}
				
				logger.debug("Stored condition for keyName: " + keyName + ", term: " + term);
				//store arguments
				keyNames.put(keyName, args);
			}
		}
		return true;
	}
	
	//for test purposes
	//returns false if the new bound is contradicting with the old ones
	//stores the new bound and returns true otherwise
	//throws Exception in case something that was not predicted, happened
	private static boolean checkAndStoreTestLocally(String keyName, ArrayList<String> args, String type,
			String equals, String term) throws Exception{
		
		//in case the new bound is a wild card, there is no meaning to do anything
		if(term.equals(wild))
			return true;
		//in case there is an old bound and it is a wild card, we must clear args and add the new bound
		if(args.size() == 3 && args.get(2).equals(wild)) {
			args.clear();
			args.add(type);
			args.add(equals);
			args.add(term);
			return true;
		}					
		
		//if there is nothing stored yet
		if(args.size() == 0)
		{
			args.add(type);
			args.add(equals);
			args.add(term);
			return true;
		//if there is one bound stored	
		} else if(args.size() == 3) {
			//the previous bound info
			String prevType = args.get(0);
			String prevEquals =  args.get(1);
			String prevTerm = args.get(2);
			
			//compare the terms
			int comp = compareTestLocally(keyName, prevTerm, term);
			//if the terms and types are equal
			if(type.equals(prevType) && comp==0) {
				//in case of equality there is nothing more to do
				if(type.equals(eq))
					return true;
				//else check if we must change the equality flag
				if(equals.equals("0"))
				{
					args.remove(1);
					args.add(1, "0");
				}
				return true;
			}
			
			//if we have two gts
			if(prevType.equals(gt) && type.equals(gt)) {
				//if previous is less than the new, replace the previous with the new one
				if(comp < 0) {
					args.clear();
					args.add(type);
					args.add(equals);
					args.add(term);
					return true;
				//nothing to do	
				} else if(comp > 0) {
					return true;
				}
			//if we have two lts	
			} else if (prevType.equals(lt) && type.equals(lt)){
				//if new is less than the previous, replace the previous with the new one
				if(comp > 0) {
					args.clear();
					args.add(type);
					args.add(equals);
					args.add(term);
					return true;
				//nothing to do	
				} else if(comp < 0) {
					return true;
				}
			//if we have a gt and a lt	
			} else if (prevType.equals(gt) && type.equals(lt)){
				//the only case that these two are contradicting 
				//is the gt to be greater than the lt
				if(comp > 0) {
					return false;
				}
				//else add the new bound
				args.add(type);
				args.add(equals);
				args.add(term);
				return true;
			
			//if we have an lt and a gt		
			} else if (prevType.equals(lt) && type.equals(gt)){
				//the only case that these two are contradicting 
				//is the gt to be greater than the lt
				if(comp < 0) {
					return false;
				}
				//else add the new bound
				args.add(type);
				args.add(equals);
				args.add(term);
				return true;
				
			//if we have an eq and a gt	
			} else if (prevType.equals(eq) && type.equals(gt)){
				//the only case that these two are contradicting 
				//is the gt to be greater than the eq
				if(comp < 0) {
					return false;
				}
				//else there is nothing to do
				return true;
			
			//if we have an eq and an lt
			} else if (prevType.equals(eq) && type.equals(lt)){
				//the only case that these two are contradicting 
				//is the lt to be less than the eq
				if(comp > 0) {
					return false;
				}
				//else there is nothing to do
				return true;
			
			//if we have two eqs
			} else if (prevType.equals(eq) && type.equals(eq)){
				//the only case that these two are contradicting 
				//is the terms are not equal
				if(comp != 0) {
					return false;
				}
				//the case with same types and comp == 0 is examined before
				logger.error("The case with same types and comp == 0 is examined before." + 
						" This point should not have reached. Possible Bug. ");
				throw new Exception("The case with same types and comp == 0 is examined before." + 
						" This point should not have reached. Possible Bug. ");
				
			//if we have a eq and a gt
			} else if (prevType.equals(gt) && type.equals(eq)){ 
				//the only case that these two are contradicting 
				//is the gt to be greater than the eq
				if(comp > 0) {
					return false;
				}
				//replace with the new eq bound
				args.clear();
				args.add(type);
				args.add(equals);
				args.add(term);
				return true;
				
			//if we have a eq and a lt
			} else if (prevType.equals(lt) && type.equals(eq)){
				//the only case that these two are contradicting 
				//is the lt to be less than the eq
				if(comp < 0) {
					return false;
				}
				//replace with the new eq bound
				args.clear();
				args.add(type);
				args.add(equals);
				args.add(term);
				return true;
				
			}
		//if there are two bounds	
		} else if(args.size() == 6) {
			//the previous gt and lt bound info
			int gtIndex = 0;
			int ltIndex = 3;
			if(args.get(0).equals(lt)){
				gtIndex = 3;
				ltIndex = 0;
			}
			String gtEquals =  args.get(gtIndex + 1);
			String gtTerm = args.get(gtIndex + 2);
			String ltEquals =  args.get(ltIndex + 1);
			String ltTerm = args.get(ltIndex + 2);
			int gtComp = compareTestLocally(keyName, gtTerm, term);
			int ltComp = compareTestLocally(keyName, ltTerm, term);
			
			//if the new bound is eq
			if(type.equals(eq)) {
				//in case gt is  greater than eq
				if(gtComp > 0) {
					return false;
				}
				//in case lt is less than eq
				if(ltComp < 0) {
					return false;
				}
				//since the bounds are not contradicting
				//clear the old bounds and store the eq
				args.clear();
				args.add(type);
				args.add(equals);
				args.add(term);
				return true;
				
			//if the new bound is lt	
			} else if(type.equals(lt)) {
				//if new lt is greater than the old lt, there is nothing to do
				if(ltComp < 0) {
					return true;
				}
				//if old and new lt are equal and the old ltEquals is "0" there is nothing to do
				if(ltComp == 0 && ltEquals.equals("0")) {
					return true;
				}
				//if new lt is less than the old gt, the bounds are contradicting
				if(gtComp > 0) {
					return false;
				}
				//the new lt must replace the old one
				//remove type
				args.remove(ltIndex);
				//remove equals
				args.remove(ltIndex);
				//remove term
				args.remove(ltIndex);
				//add the new (in the reverse order)
				args.add(ltIndex, term);
				args.add(ltIndex, equals);
				args.add(ltIndex, type);
				return true;
				
			//if the new bound is gt	
			} else if(type.equals(gt)) {
				//if new gt is less than the old gt, there is nothing to do
				if(gtComp > 0) {
					return true;
				}
				//if old and new gt are equal and the old gtEquals is "0" there is nothing to do
				if(gtComp == 0 && gtEquals.equals("0")) {
					return true;
				}
				//if new gt is greater than the old lt, the bounds are contradicting
				if(ltComp < 0) {
					return false;
				}
				//the new gt must replace the old one
				//remove type
				args.remove(gtIndex);
				//remove equals
				args.remove(gtIndex);
				//remove term
				args.remove(gtIndex);
				//add the new (in the reverse order)
				args.add(gtIndex, term);
				args.add(gtIndex, equals);
				args.add(gtIndex, type);
				return true;
			}						
			
		//this means we have a bug at parsing	
		} else {
			throw new Exception("Bug. args size = " + args.size());
		}
		//this point should not have been reached
		logger.warn("Normally this point should not have been reached");
		return true;
	}

}
