/*
 * ForwardIndexType.java
 *
 * $Author: tsakas $
 * $Date: 2007/12/20 14:37:39 $
 * $Id: ForwardIndexType.java,v 1.1 2007/12/20 14:37:39 tsakas Exp $
 *
 * <pre>
 *             Copyright (c) : 2006 Fast Search & Transfer ASA
 *                             ALL RIGHTS RESERVED
 * </pre>
 */

package org.gcube.indexmanagement.common;
   
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.utils.logging.GCUBELog;
   
/**   
 * This class returns the index types that are supported by the 
 * ForwardIndex.
 */
public class ForwardIndexType extends IndexType {
	
	/** logger */
	static GCUBELog logger = new GCUBELog(ForwardIndexType.class);
	
	/** The name of the "key" field */
	private static final String KEY_FIELD_NAME = "key";
	
	/** The name of the "value" field */
	private static final String VALUE_FIELD_NAME = "value";
	
	
	 /** The fields of this GeoIndexType */
    public ForwardIndexField[] fields;

    /** The parser used to parse the xml representation of the ForwardIndexType */
    private XMLProfileParser parser = null;

    /** The object representing the "key" field */
    private ForwardIndexField keyField = null;
    
    /** The object representing the "value" field */
    private ForwardIndexField valueField = null;
    
    /**
     * Standard constructor which build a ForwardIndexType object an index type name
     * 
     * @param indexTypeName -
     *            The name to assign to the GeoIndexType
     * @param scope -
     *            the current scope
     */
//    public ForwardIndexType(String indexTypeName, GCUBEScope scope) {
//    	super(indexTypeName);
//    	
//    	try {
//	    	String indexType = retrieveIndexTypeGenericResource(scope);
//	        if (parser == null) {
//	            parser = new XMLProfileParser();
//	        }
//        
//            parser.readString(indexType, null);
//            readIndexType();
//        } catch (Exception ex) {
//            logger.error("Error initializing ForwardIndexType.", ex);
//        }
//    }
    
    public ForwardIndexType(String indexTypeName, String scopeStr){
    	super(indexTypeName);
    	
    	GCUBEScope scope = GCUBEScope.getScope(scopeStr);
    	
    	try {
	    	String indexType = retrieveIndexTypeGenericResource(scope);
	        if (parser == null) {
	            parser = new XMLProfileParser();
	        }
        
            parser.readString(indexType, null);
            readIndexType();
        } catch (Exception ex) {
            logger.error("Error initializing ForwardIndexType.", ex);
        }
    	
    }

    /**
     * The method which actually traverses the created DOM document in order to
     * fill the fields
     * 
     * @throws Exception
     */
    private void readIndexType() throws Exception {
        int position = 0;
        try {
            // Extract the fields
            parser.setRootNode("field-list");
            int nrOfFields = parser.countDescendants("field");
            if (nrOfFields != 2)
            	throw new Exception("Invalid forward index type: number of fields must be equal to 2.");

            fields = new ForwardIndexField[nrOfFields];
            while (parser.setNextField()) {
                fillField(position++);
            }

        } catch (Exception ex) {
            logger.error("Error while reading indexType, position: " + position, ex);
            throw ex;
        }
    }
    
    /**
     * The method which actually fills a field of this ForwardIndexType object
     * 
     * @param position -
     *            The index (in the fields array) of the ForwardIndexField to fill
     * @throws Exception -
     *             Unable to get the field information from the parser
     */
    private void fillField(int position) throws Exception {
        fields[position] = new ForwardIndexField();
        fields[position].name = parser.getFieldByValue("name");
        fields[position].dataType = ForwardIndexField.DataType.getByName(parser.getFieldByValue("type"));
        fields[position].size = fields[position].dataType.getDefaultSize();
        fields[position].dataTypeFormat = parser.getFieldByValue("format");
        
        if (fields[position].name.equals(KEY_FIELD_NAME))
        	keyField = fields[position];
        else if (fields[position].name.equals(VALUE_FIELD_NAME))
        	valueField = fields[position];
        else
        	throw new Exception("Unrecongisable field found: " + fields[position].name);
    }

    /**
     * A method to check if this ForwardIndexType object contains a certain field
     * 
     * @param fieldName -
     *            The name of the field to look for
     * @param fieldDataType -
     *            The DataType of the field to look for
     * @return - TRUE if this GeoIndexType object contains a field with the
     *         specified name and data type
     */
    public boolean containsField(String fieldName, ForwardIndexField.DataType fieldDataType) {
        for (ForwardIndexField field : fields) {
            if (field.name.equals(fieldName) && field.dataType.equals(fieldDataType)) {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Returns the object representing the "key" field of this forward index type
     * @return
     */
    public ForwardIndexField getKeyField() {
    	return keyField;
    }
    
    /**
     * Returns the object representing the "value" field of this forward index type
     * @return
     */
    public ForwardIndexField getValueField() {
    	return valueField;
    }
}
