package org.gcube.informationsystem.icproxy.resources;

import static org.gcube.resources.discovery.icclient.ICFactory.client;
import static org.gcube.resources.discovery.icclient.ICFactory.clientFor;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;

import java.util.List;

import javax.validation.constraints.NotNull;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;

import lombok.extern.slf4j.Slf4j;

import org.gcube.common.resources.gcore.ServiceEndpoint;
import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.api.SimpleQuery;

@Slf4j
@Path("ServiceEndpoint")
public class ServiceEndpointResource {
	
	@GET
	@Path("/{category}")
	@Produces(MediaType.APPLICATION_XML)
	public List<ServiceEndpoint> retrieve(@NotNull @PathParam("category") String resourceCategory) {
		log.info("ServiceEndpoint called with category {} in context {}",resourceCategory, ScopeProvider.instance.get());
		
		DiscoveryClient<ServiceEndpoint> client = clientFor(ServiceEndpoint.class);

		List<ServiceEndpoint> endpoints = client.submit(getQuery(resourceCategory));
		log.debug("retrieved resources are "+endpoints.size());
		return endpoints; 
	}

	@GET
	@Path("/{category}/{name}")
	@Produces(MediaType.APPLICATION_XML)
	public List<ServiceEndpoint> retrieve(@NotNull @PathParam("name") String resourceName, 
			@NotNull @PathParam("category") String resourceCategory) {
		log.info("ServiceEndpoint called with category {} and name {} in scope {}",resourceCategory, resourceName, ScopeProvider.instance.get());
		
		DiscoveryClient<ServiceEndpoint> client = clientFor(ServiceEndpoint.class);

		List<ServiceEndpoint> endpoints = client.submit(getQuery(resourceName, resourceCategory));
		log.debug("retrieved resources are "+endpoints.size());
		return endpoints; 
	}

	@GET
	@Path("/{category}/{name}/Result/{result:([^$\\?]+)}")
	@Produces(MediaType.TEXT_XML)
	public String retrieveCustom(@NotNull @PathParam("name") String resourceName, 
			@NotNull @PathParam("category") String resourceCategory, @NotNull @PathParam("result") String resultXPath) {
		log.info("ServiceEndpoint called with category {} and name {} and result {} in scope {}"
				,resourceCategory, resourceName, resultXPath, ScopeProvider.instance.get());
		
		SimpleQuery query = getQuery(resourceName, resourceCategory);
				
		if (resultXPath.startsWith("/"))
			query.setResult("$resource"+resultXPath);
		else
			query.setResult("$resource/"+resultXPath);
		
		DiscoveryClient<String> client = client();
		List<String> endpoints = client.submit(query);
		StringBuilder builder = new StringBuilder("<Results>");
		for (String single: endpoints)
			builder.append("<Result>").append(single.replaceAll("\n", "")).append("</Result>");
		
		builder.append("</Results>");
		
		log.debug("retrieved resources are "+endpoints.size());
		return builder.toString(); 
	}

	private SimpleQuery getQuery(String resourceName, String resourceCategory){
		SimpleQuery query = queryFor(ServiceEndpoint.class);
		query.addCondition(String.format("$resource/Profile/Name/text() eq '%s'",resourceName));
		query.addCondition(String.format("$resource/Profile/Category/text() eq '%s'",resourceCategory));
		return query;
	}
	
	private SimpleQuery getQuery(String resourceCategory){
		SimpleQuery query = queryFor(ServiceEndpoint.class);
		query.addCondition(String.format("$resource/Profile/Category/text() eq '%s'",resourceCategory));
		return query;
	}
	
}
