package org.gcube.informationsystem.resourceregistry.contexts;

import java.util.List;
import java.util.UUID;

import org.gcube.informationsystem.contexts.reference.ContextState;
import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.resourceregistry.api.contexts.ContextCache;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.request.RequestInfo;

/**
 * Client interface for managing contexts in the Resource Registry system.
 * 
 * <strong>Purpose:</strong>
 * This interface provides methods for creating, reading, updating, deleting, and managing the state of contexts
 * within the Resource Registry. Contexts represent organizational units that contain and organize Information
 * System resources, facets, and relations.
 * 
 * <strong>Instantiation:</strong>
 * Clients should be created using the factory pattern for proper configuration:
 * <pre>
 * // Recommended way to create a client
 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
 * </pre>
 * 
 * <strong>Supported Operations:</strong>
 * <ul>
 * <li><strong>CRUD Operations</strong>: Create, Read, Update, Delete contexts with full lifecycle management</li>
 * <li><strong>State Management</strong>: Change context states (ACTIVE, READY, DOWN) with proper validation</li>
 * <li><strong>Query Operations</strong>: List all contexts with filtering and pagination support</li>
 * <li><strong>Current Context</strong>: Access the context associated with the current authorization token</li>
 * </ul>
 * 
 * <strong>Query Parameter Configuration:</strong>
 * Most methods support optional query parameters that can be configured via the client's header mechanism:
 * <ul>
 * <li><strong>includeMeta</strong>: Include metadata in responses (role-based filtering applied)</li>
 * <li><strong>limit</strong>: Maximum number of results per request</li>
 * <li><strong>offset</strong>: Number of results to skip for pagination</li>
 * </ul>
 * 
 * <strong>Authorization:</strong>
 * All operations respect strict authorization rules based on user roles:
 * <ul>
 * <li><strong>IS-Manager</strong>: Full access to all contexts and operations</li>
 * <li><strong>Infrastructure-Manager</strong>: Full access to all contexts and operations  </li>
 * <li><strong>Context-Manager</strong>: Full access only within their managed contexts</li>
 * <li><strong>Other Users</strong>: Limited read access based on context membership</li>
 * </ul>
 * 
 * <strong>Context States:</strong>
 * Contexts can be in different states managed through state change operations:
 * <ul>
 * <li><strong>ACTIVE</strong>: Context is fully operational and accessible</li>
 * <li><strong>READY</strong>: Context is prepared but not yet active</li>
 * <li><strong>DOWN</strong>: Context is temporarily unavailable</li>
 * </ul>
 *
 * <strong>Context Support:</strong>
 * The client automatically operates within the context determined by the authorization token.
 *
 * @author Luca Frosini (ISTI - CNR)
 */
public interface ResourceRegistryContextClient extends RequestInfo {

	/**
	 * Adds a custom HTTP header to be included in requests.
	 * 
	 * @param name Header name
	 * @param value Header value
	 */
	public void addHeader(String name, String value);
	
	/**
	 * Adds a custom HTTP header to be included in requests.
	 * 
	 * @param name Header name
	 * @param value boolean value
	 */
	public void addHeader(String name, boolean value);

	/**
	 * Gets the context cache used for context-related operations.
	 * 
	 * @return The context cache instance
	 */
	public ContextCache getContextCache();
	
	/**
	 * Retrieves the list of all contexts in the system.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /contexts[?limit={limit}&offset={offset}&includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Returns all contexts that are accessible to the current user</li>
	 * <li>Results are filtered based on user authorization and context access permissions</li>
	 * <li>Supports pagination and metadata inclusion based on client configuration</li>
	 * <li>Returns strongly-typed Context objects with role-appropriate detail level.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Contexts successfully retrieved</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access context information</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response context</li>
	 * <li>Default value: false (basic context information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.ContextPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>limit</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Maximum number of contexts to return in a single response</li>
	 * <li>Default value: server configuration dependent</li>
	 * <li>Usage: {@code client.setLimit(50)}</li>
	 * <li>Example: Set to 50 to get at most 50 contexts per request</li>
	 * <li>Unlimited results: Set to {@code RequestInfo.UNBOUNDED_LIMIT} (-1) for unlimited results (may cause timeout if results are too many)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.ContextPath#LIMIT_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>offset</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Number of contexts to skip from the beginning of the result set</li>
	 * <li>Default value: 0 (starts from the first context)</li>
	* <li>Usage: {@code client.setOffset(10)}</li>
	 * <li>Example: Set to 10 to skip the first 10 contexts (useful for pagination)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.ContextPath#OFFSET_QUERY_PARAMETER}.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>No input parameters required for this method</li>
	 * <li>All contexts accessible to the current user are returned based on authorization level</li>
	 * <li>Context filtering is automatically applied based on user permissions.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can retrieve all contexts in the system with complete metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can retrieve all contexts in the system with complete metadata</li>
	 * <li><strong>Context-Manager</strong>: Can retrieve contexts within their management scope</li>
	 * <li><strong>Other Users</strong>: Can retrieve contexts they have explicit access to with filtered metadata.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * // Basic usage - get all contexts with default settings
	 * List&lt;Context&gt; contexts = client.all();
	 * 
	 * // With metadata for non-admin users
	 * client.setIncludeMeta(true);
	 * List&lt;Context&gt; contextsWithMeta = client.all();
	 * 
	 * // With pagination
	 * client.setLimit(50);
	 * client.setOffset(10);
	 * List&lt;Context&gt; paginatedContexts = client.all();
	 * </pre>
	 * 
	 * @return list of Context objects with role-appropriate detail level accessible to the current user
	 * @throws ResourceRegistryException if an error occurs during context retrieval or if there are authorization issues
	 */
	public List<Context> all() throws ResourceRegistryException;
	
	/**
	 * Creates a new context in the system.
	 * This method invokes the REST API endpoint {@code PUT /contexts/{context-uuid}} from the Resource Registry service.
	 * If the context object does not have a UUID, one will be automatically generated.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>If the context UUID is null, a new UUID will be automatically generated</li>
	 * <li>The operation will create a new context or update an existing one with the same UUID</li>
	 * <li>All context properties and relationships are established based on the provided context object</li>
	 * <li>Returns the created context object with complete information and assigned UUID.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: Context successfully created</li>
	 * <li><strong>200 OK</strong>: Context successfully updated (if UUID already existed)</li>
	 * <li><strong>400 Bad Request</strong>: Invalid context data or validation failure</li>
	 * <li><strong>409 Conflict</strong>: Context with the same name already exists in the parent context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to create contexts in the specified location</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response</li>
	 * <li>Default value: false (basic context information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.ContextPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Context object is validated against system constraints</li>
	 * <li>All context properties and relationships are processed</li>
	 * <li>UUID assignment is handled automatically if not provided.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can create any context without restrictions and automatically receive full context information including complete metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can create any context without restrictions and automatically receive full context information including complete metadata</li>
	 * <li><strong>Context-Manager</strong>: Can create contexts only if they are Context-Manager of the parent context (the parent context must be the current context retrieved from client's authorization token)</li>
	 * <li><strong>Other Users</strong>: Cannot create contexts and will receive authorization errors if attempting this operation.</li>
	 * </ul>
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>If the context UUID is null, a new UUID will be automatically generated</li>
	 * <li>The operation will create a new context or update an existing one with the same UUID</li>
	 * <li>All context properties and relationships are established based on the provided context object</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * // Create a new context object
	 * Context newContext = new ContextImpl();
	 * newContext.setName("MyNewContext");
	 * newContext.setParent(parentContext);
	 * 
	 * // Basic usage - create context
	 * Context createdContext = client.create(newContext);
	 * 
	 * // With metadata inclusion for non-admin users
	 * client.setIncludeMeta(true);
	 * Context contextWithMeta = client.create(newContext);
	 * </pre>
	 * 
	 * @param context the Context object to create
	 * @return the created Context object with complete information and assigned UUID
	 * @throws ContextAlreadyPresentException if a context with the same UUID already exists and creation fails (HTTP 409)
	 * @throws ResourceRegistryException if the context cannot be created due to authorization issues or other errors
	 * 
	 * @see ResourceRegistryContextClient#create(String) for creating contexts from JSON representation
	 */
	public Context create(Context context) throws ContextAlreadyPresentException, ResourceRegistryException;

	/**
	 * Creates a new context in the system from its JSON representation.
	 * This method invokes the REST API endpoint {@code PUT /contexts/{context-uuid}} from the Resource Registry service.
	 * If the context JSON does not contain a UUID, one will be automatically generated.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>The JSON is parsed to extract context information including UUID, name, parent, and other properties</li>
	 * <li>If the context UUID is null or missing, a new UUID will be automatically generated</li>
	 * <li>The operation will create a new context or update an existing one with the same UUID</li>
	 * <li>All context properties and relationships are established based on the provided JSON</li>
	 * <li>Returns the created context as JSON string with complete information and assigned UUID.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>201 Created</strong>: Context successfully created</li>
	 * <li><strong>200 OK</strong>: Context successfully updated (if UUID already existed)</li>
	 * <li><strong>400 Bad Request</strong>: Invalid context data, JSON format, or validation failure</li>
	 * <li><strong>409 Conflict</strong>: Context with the same name already exists in the parent context</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to create contexts in the specified location</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response</li>
	 * <li>Default value: false (basic context information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.ContextPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>JSON must be a valid Context representation</li>
	 * <li>JSON is parsed to extract context information including UUID, name, parent, and other properties</li>
	 * <li>All context properties are validated against system constraints.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can create any context without restrictions and automatically receive full context information including complete metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can create any context without restrictions and automatically receive full context information including complete metadata</li>
	 * <li><strong>Context-Manager</strong>: Can create contexts only if they are Context-Manager of the parent context (the parent context must be the current context retrieved from client's authorization token)</li>
	 * <li><strong>Other Users</strong>: Cannot create contexts and will receive authorization errors if attempting this operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * // Create context from JSON string
	 * String contextJson = "{\"@class\":\"Context\",\"name\":\"MyNewContext\",\"parent\":{\"id\":\"parent-uuid\"}}";
	 * 
	 * // Basic usage - create context from JSON
	 * String createdContextJson = client.create(contextJson);
	 * 
	 * // With metadata inclusion for non-admin users
	 * client.setIncludeMeta(true);
	 * String contextWithMeta = client.create(contextJson);
	 * </pre>
	 * 
	 * @param context the JSON string representation of the context to create
	 * @return the created context as JSON string with complete information and assigned UUID
	 * @throws ContextAlreadyPresentException if a context with the same UUID already exists and creation fails
	 * @throws ResourceRegistryException if the context cannot be created due to authorization issues, invalid JSON, or other errors
	 * 
	 * @see #create(Context) for creating contexts from Context objects
	 */
	public String create(String context) throws ContextAlreadyPresentException, ResourceRegistryException;

	/**
	 * Checks if a context with the specified UUID exists in the system.
	 * 
	 * <strong>Corresponding REST API:</strong> Internally uses {@code GET /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Verifies if a context with the given UUID exists in the system</li>
	 * <li>Returns {@code true} if the context exists and is accessible to the current user</li>
	 * <li>Returns {@code false} if the context does not exist or is not accessible</li>
	 * <li>Does not return the context data, only confirms existence.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context exists and is accessible</li>
	 * <li><strong>404 Not Found</strong>: Context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access context information</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>UUID must be a valid UUID format</li>
	 * <li>Context accessibility is verified based on user authorization level</li>
	 * <li>Contexts that exist but are not accessible to the user will return {@code false}.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can check existence of any context</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can check existence of any context</li>
	 * <li><strong>Context-Manager</strong>: Can check contexts within their management scope</li>
	 * <li><strong>Other Users</strong>: Can only check contexts they have access to read.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * UUID contextUuid = UUID.fromString("c0f314e7-2807-4241-a792-2a6c79ed4fd0");
	 * 
	 * // Check if context exists
	 * if (client.exist(contextUuid)) {
	 *     System.out.println("Context exists and is accessible");
	 *     Context context = client.read(contextUuid);
	 *     // Process context...
	 * } else {
	 *     System.out.println("Context does not exist or is not accessible");
	 * }
	 * </pre>
	 * 
	 * @param uuid the UUID of the context to check for existence
	 * @return {@code true} if the context exists and is accessible to the current user, {@code false} otherwise
	 * @throws ResourceRegistryException if there are communication errors with the Resource Registry service
	 */
	public boolean exist(UUID uuid) throws ResourceRegistryException;

	/**
	 * Checks if a context with the specified UUID (as string) exists in the system.
	 * 
	 * <strong>Corresponding REST API:</strong> Internally uses {@code GET /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Converts the string UUID to a UUID object and delegates to {@code exist(UUID)}</li>
	 * <li>Returns {@code true} if the context exists and is accessible to the current user</li>
	 * <li>Returns {@code false} if the context does not exist or is not accessible</li>
	 * <li>Does not return the context data, only confirms existence.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context exists and is accessible</li>
	 * <li><strong>404 Not Found</strong>: Context does not exist or is not accessible</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access context information</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>String must be a valid UUID format</li>
	 * <li>Context accessibility is verified based on user authorization level</li>
	 * <li>Contexts that exist but are not accessible to the user will return {@code false}.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can check existence of any context</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can check existence of any context</li>
	 * <li><strong>Context-Manager</strong>: Can check contexts within their management scope</li>
	 * <li><strong>Other Users</strong>: Can only check contexts they have access to read.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * String contextUuidString = "c0f314e7-2807-4241-a792-2a6c79ed4fd0";
	 * 
	 * // Check if context exists
	 * if (client.exist(contextUuidString)) {
	 *     System.out.println("Context exists and is accessible");
	 *     Context context = client.read(contextUuidString);
	 *     // Process context...
	 * } else {
	 *     System.out.println("Context does not exist or is not accessible");
	 * }
	 * 
	 * // Alternative: UUID validation
	 * try {
	 *     boolean exists = client.exist(contextUuidString);
	 *     // Process result...
	 * } catch (IllegalArgumentException e) {
	 *     System.err.println("Invalid UUID format: " + contextUuidString);
	 * }
	 * </pre>
	 * 
	 * @param uuid the UUID of the context (as string) to check for existence
	 * @return {@code true} if the context exists and is accessible to the current user, {@code false} otherwise
	 * @throws ResourceRegistryException if there are communication errors with the Resource Registry service
	 * @throws IllegalArgumentException if the provided string is not a valid UUID format
	 */
	public boolean exist(String uuid) throws ResourceRegistryException;

	/**
	 * Retrieves a context by extracting its UUID from the provided context object.
	 * This is a convenience method that delegates to {@code read(UUID)} using the context's ID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /contexts/{context-uuid}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Extracts the UUID from the provided context object</li>
	 * <li>Delegates to {@code read(UUID)} for the actual retrieval operation</li>
	 * <li>Returns the full context object with all accessible properties</li>
	 * <li>Response detail level varies based on user authorization.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Context with the specified UUID is not found</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access context information</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response (for non-admin users)</li>
	 * <li>Default value: false (basic context information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.ContextPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Context object must contain a valid UUID</li>
	 * <li>Only the UUID is used from the provided context object, all other properties are ignored.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can retrieve any context with complete metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can retrieve any context with complete metadata</li>
	 * <li><strong>Context-Manager</strong>: Can retrieve contexts within their management scope</li>
	 * <li><strong>Other Users</strong>: Can retrieve contexts they have access to with filtered metadata.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * Context contextStub = new ContextImpl();
	 * contextStub.setID(UUID.fromString("c0f314e7-2807-4241-a792-2a6c79ed4fd0"));
	 * 
	 * // Read the full context using the stub
	 * Context fullContext = client.read(contextStub);
	 * </pre>
	 * 
	 * @param context the context object containing the UUID to retrieve
	 * @return the complete Context object with role-appropriate detail level
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if there are communication errors or authorization issues
	 */
	public Context read(Context context) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Retrieves a specific context by its UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /contexts/{context-uuid}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Retrieves the context with the specified UUID from the system</li>
	 * <li>Returns the complete context object with all accessible properties</li>
	 * <li>Response detail level varies based on user authorization</li>
	 * <li>Metadata inclusion is controlled by client configuration and user authorization.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Context with the specified UUID is not found</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access context information</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response (for non-admin users)</li>
	 * <li>Default value: false (basic context information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.ContextPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>UUID must be a valid UUID format</li>
	 * <li>Context accessibility is verified based on user authorization level.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can retrieve any context with complete metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can retrieve any context with complete metadata</li>
	 * <li><strong>Context-Manager</strong>: Can retrieve contexts within their management scope with full information when operating in their own context</li>
	 * <li><strong>Other Users</strong>: Can retrieve contexts they have access to with filtered metadata.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * UUID contextUuid = UUID.fromString("c0f314e7-2807-4241-a792-2a6c79ed4fd0");
	 * 
	 * // Basic usage - read context
	 * Context context = client.read(contextUuid);
	 * System.out.println("Context name: " + context.getName());
	 * 
	 * // With metadata for non-admin users
	 * client.setIncludeMeta(true);
	 * Context contextWithMeta = client.read(contextUuid);
	 * </pre>
	 * 
	 * @param uuid the UUID of the context to retrieve
	 * @return the Context object with role-appropriate detail level
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if there are communication errors or authorization issues
	 */
	public Context read(UUID uuid) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Retrieves a specific context by its UUID (as string) and returns it as JSON.
	 * This method delegates to {@code read(UUID)} and marshals the result to JSON format.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code GET /contexts/{context-uuid}[?includeMeta={true|false}]}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Converts the string UUID to a UUID object</li>
	 * <li>Delegates to {@code read(UUID)} for the actual retrieval operation</li>
	 * <li>Marshals the resulting Context object to JSON format</li>
	 * <li>Returns the complete context as JSON string with all accessible properties.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Context with the specified UUID is not found</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access context information</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response (for non-admin users)</li>
	 * <li>Default value: false (basic context information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.ContextPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>String must be a valid UUID format</li>
	 * <li>Context accessibility is verified based on user authorization level.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can retrieve any context with complete metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can retrieve any context with complete metadata</li>
	 * <li><strong>Context-Manager</strong>: Can retrieve contexts within their management scope</li>
	 * <li><strong>Other Users</strong>: Can retrieve contexts they have access to with filtered metadata.</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * String contextUuidString = "c0f314e7-2807-4241-a792-2a6c79ed4fd0";
	 * 
	 * // Basic usage - read context as JSON
	 * String contextJson = client.read(contextUuidString);
	 * System.out.println("Context JSON: " + contextJson);
	 * 
	 * // With metadata for non-admin users
	 * client.setIncludeMeta(true);
	 * String contextWithMetaJson = client.read(contextUuidString);
	 * 
	 * // Error handling for invalid UUID
	 * try {
	 *     String result = client.read("invalid-uuid-format");
	 * } catch (IllegalArgumentException e) {
	 *     System.err.println("Invalid UUID format");
	 * }
	 * </pre>
	 * 
	 * @param uuid the UUID of the context (as string) to retrieve
	 * @return the Context as JSON string with role-appropriate detail level
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if there are communication errors or authorization issues
	 * @throws IllegalArgumentException if the provided string is not a valid UUID format
	 */
	public String read(String uuid) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Retrieves the current context associated with the client's authorization token.
	 * This method extracts the context information from the security context and retrieves the full context object.
	 * 
	 * <strong>Corresponding REST API:</strong> Equivalent to {@code GET /contexts/CURRENT_CONTEXT} but implemented client-side
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Extracts the context full name from the current authorization token via {@code SecretManagerProvider}</li>
	 * <li>Resolves the context full name to its UUID</li>
	 * <li>Delegates to {@code read(UUID)} for the actual context retrieval</li>
	 * <li>Returns the complete context object associated with the current authorization token.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Current context successfully retrieved</li>
	 * <li><strong>404 Not Found</strong>: Current context cannot be found or resolved</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to access their current context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response (for non-admin users)</li>
	 * <li>Default value: false (basic context information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.ContextPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Context information is automatically extracted from the JWT token</li>
	 * <li>No explicit input parameters required.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can retrieve current context with complete metadata</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can retrieve current context with complete metadata</li>
	 * <li><strong>Context-Manager</strong>: Can retrieve their current context with full information</li>
	 * <li><strong>Other Users</strong>: Can retrieve their current context (this is typically always granted).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * // Get current context
	 * Context currentContext = client.readCurrentContext();
	 * System.out.println("Current context: " + currentContext.getName());
	 * System.out.println("Current context UUID: " + currentContext.getID());
	 * 
	 * // With metadata
	 * client.setIncludeMeta(true);
	 * Context currentContextWithMeta = client.readCurrentContext();
	 * 
	 * // Use current context for relative operations
	 * if (currentContext.getParent() != null) {
	 *     Context parentContext = client.read(currentContext.getParent().getID());
	 *     System.out.println("Parent context: " + parentContext.getName());
	 * }
	 * </pre>
	 * 
	 * @return the Context object associated with the current authorization token
	 * @throws ContextNotFoundException if the current context cannot be found or resolved
	 * @throws ResourceRegistryException if there are communication errors or authorization issues
	 */
	public Context readCurrentContext() throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Updates an existing context in the system.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Updates the context with the specified UUID using the provided context object</li>
	 * <li><strong>Name Change:</strong> Updates the context name while preserving all other properties</li>
	 * <li><strong>Parent Change (Move Operation):</strong> Moves the context to a different parent in the context hierarchy</li>
	 * <li><strong>Combined Changes:</strong> Both name and parent can be changed in the same update request</li>
	 * <li>Context UUIDs remain stable regardless of name or parent changes.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context successfully updated</li>
	 * <li><strong>404 Not Found</strong>: Context with the specified UUID is not found</li>
	 * <li><strong>400 Bad Request</strong>: Invalid context data or validation failure</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to update the context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response</li>
	 * <li>Default value: false (basic context information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.ContextPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Context object must contain a valid UUID</li>
	 * <li>All context properties are validated against system constraints</li>
	 * <li>Name and parent changes are validated for conflicts and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can update any context without restrictions</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can update any context without restrictions</li>
	 * <li><strong>Context-Manager</strong>: Can update contexts only if they are Context-Manager of the parent context</li>
	 * <li><strong>Other Users</strong>: Cannot update contexts (will receive authorization errors).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * // Get context to update
	 * Context context = client.read(contextUuid);
	 * 
	 * // Update context name
	 * context.setName("NewContextName");
	 * Context updatedContext = client.update(context);
	 * 
	 * // Update with metadata for non-admin users
	 * client.setIncludeMeta(true);
	 * Context contextWithMeta = client.update(context);
	 * </pre>
	 * 
	 * @param context the Context object with updated properties
	 * @return the updated Context object with complete information
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if the context cannot be updated due to authorization issues or other errors
	 */
	public Context update(Context context) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Updates an existing context in the system from its JSON representation.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PUT /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Updates the context using the provided JSON representation</li>
	 * <li><strong>Name Change:</strong> Updates the context name while preserving all other properties</li>
	 * <li><strong>Parent Change (Move Operation):</strong> Moves the context to a different parent in the context hierarchy</li>
	 * <li><strong>Combined Changes:</strong> Both name and parent can be changed in the same update request</li>
	 * <li>Returns the updated context as JSON string with complete information.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context successfully updated</li>
	 * <li><strong>404 Not Found</strong>: Context with the specified UUID is not found</li>
	 * <li><strong>400 Bad Request</strong>: Invalid context data, JSON format, or validation failure</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to update the context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>includeMeta</strong> (configurable via client configuration):
	 * <ul>
	 * <li>Whether to include metadata in the response</li>
	 * <li>Default value: false (basic context information only)</li>
	 * <li>Usage: {@code client.setIncludeMeta(true)}</li>
	 * <li>Values: true (includes metadata with role-based filtering) | false (basic information only)</li>
	 * <li>Query parameter: {@link org.gcube.informationsystem.resourceregistry.api.rest.ContextPath#INCLUDE_META_QUERY_PARAMETER}</li>
	 * <li><strong>Restriction:</strong> IS-Manager and Infrastructure-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>JSON must be a valid Context representation</li>
	 * <li>JSON must contain a valid UUID field to identify the target context</li>
	 * <li>All context properties are validated against system constraints</li>
	 * <li>Name and parent changes are validated for conflicts and authorization.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can update any context without restrictions</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can update any context without restrictions</li>
	 * <li><strong>Context-Manager</strong>: Can update contexts only if they are Context-Manager of the parent context</li>
	 * <li><strong>Other Users</strong>: Cannot update contexts (will receive authorization errors).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * // Update context from JSON string
	 * String contextJson = "{\"@class\":\"Context\",\"id\":\"uuid\",\"name\":\"UpdatedName\"}";
	 * 
	 * // Basic usage - update context from JSON
	 * String updatedContextJson = client.update(contextJson);
	 * 
	 * // With metadata inclusion for non-admin users
	 * client.setIncludeMeta(true);
	 * String contextWithMeta = client.update(contextJson);
	 * </pre>
	 * 
	 * @param context the JSON string representation of the context with updated properties
	 * @return the updated context as JSON string with complete information
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if the context cannot be updated due to authorization issues, invalid JSON, or other errors
	 */
	public String update(String context) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Deletes an existing context permanently from the system by extracting its UUID.
	 * This is a convenience method that delegates to {@code delete(UUID)} using the context's ID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Extracts the UUID from the provided context object</li>
	 * <li>Delegates to {@code delete(UUID)} for the actual deletion operation</li>
	 * <li>Context is permanently deleted from the system and moved to a "cemetery" for audit trails and historical tracking</li>
	 * <li>The UUID becomes permanently unavailable for reuse</li>
	 * <li>Once deleted, the context cannot be restored through normal operations.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: Context with the specified UUID is not found</li>
	 * <li><strong>409 Conflict</strong>: Context has child contexts or other dependencies preventing deletion</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to delete the context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Context object must contain a valid UUID</li>
	 * <li>Only the UUID is used from the provided context object, all other properties are ignored</li>
	 * <li><strong>Child Contexts:</strong> Cannot delete a context that has child contexts (all child contexts must be deleted first or moved to a different parent)</li>
	 * <li><strong>Instance Impact:</strong> Instances belonging to the deleted context will no longer belong to any context (if instances exist only in the deleted context, they will be permanently deleted from the system).</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can delete any context without restrictions</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can delete any context without restrictions</li>
	 * <li><strong>Context-Manager</strong>: Can delete contexts only if they are Context-Manager of the parent context</li>
	 * <li><strong>Other Users</strong>: Cannot delete contexts (will receive authorization errors).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * Context contextToDelete = client.read(contextUuid);
	 * 
	 * // Delete the context using the context object
	 * boolean deleted = client.delete(contextToDelete);
	 * if (deleted) {
	 *     System.out.println("Context successfully deleted");
	 * }
	 * </pre>
	 * 
	 * @param context the context object to delete
	 * @return {@code true} if the context was successfully deleted, {@code false} otherwise
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if the context cannot be deleted due to authorization issues, child contexts, or other errors
	 */
	public boolean delete(Context context) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Deletes an existing context permanently from the system using its UUID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Permanently deletes the context with the specified UUID from the system</li>
	 * <li>Context is moved to a "cemetery" for audit trails and historical tracking</li>
	 * <li>The UUID becomes permanently unavailable for reuse</li>
	 * <li>Once deleted, the context cannot be restored through normal operations.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: Context with the specified UUID is not found</li>
	 * <li><strong>409 Conflict</strong>: Context has child contexts or other dependencies preventing deletion</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to delete the context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>UUID must be a valid UUID format</li>
	 * <li>Context must exist and be accessible to the user</li>
	 * <li><strong>Child Contexts:</strong> Cannot delete a context that has child contexts (all child contexts must be deleted first or moved to a different parent)</li>
	 * <li><strong>Instance Impact:</strong> Instances belonging to the deleted context will no longer belong to any context (if instances exist only in the deleted context, they will be permanently deleted from the system).</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can delete any context without restrictions</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can delete any context without restrictions</li>
	 * <li><strong>Context-Manager</strong>: Can delete contexts only if they are Context-Manager of the parent context</li>
	 * <li><strong>Other Users</strong>: Cannot delete contexts (will receive authorization errors).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * UUID contextUuid = UUID.fromString("c0f314e7-2807-4241-a792-2a6c79ed4fd0");
	 * 
	 * // Delete the context using UUID
	 * boolean deleted = client.delete(contextUuid);
	 * if (deleted) {
	 *     System.out.println("Context successfully deleted");
	 * }
	 * </pre>
	 * 
	 * @param uuid the UUID of the context to delete
	 * @return {@code true} if the context was successfully deleted, {@code false} otherwise
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if the context cannot be deleted due to authorization issues, child contexts, or other errors
	 */
	public boolean delete(UUID uuid) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Deletes an existing context permanently from the system using its UUID (as string).
	 * 
	 * <strong>Corresponding REST API:</strong> {@code DELETE /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li>Converts the string UUID to a UUID object and delegates to {@code delete(UUID)}</li>
	 * <li>Permanently deletes the context with the specified UUID from the system</li>
	 * <li>Context is moved to a "cemetery" for audit trails and historical tracking</li>
	 * <li>The UUID becomes permanently unavailable for reuse.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context successfully deleted</li>
	 * <li><strong>404 Not Found</strong>: Context with the specified UUID is not found</li>
	 * <li><strong>409 Conflict</strong>: Context has child contexts or other dependencies preventing deletion</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to delete the context</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>String must be a valid UUID format</li>
	 * <li>Context must exist and be accessible to the user</li>
	 * <li><strong>Child Contexts:</strong> Cannot delete a context that has child contexts (all child contexts must be deleted first or moved to a different parent)</li>
	 * <li><strong>Instance Impact:</strong> Instances belonging to the deleted context will no longer belong to any context (if instances exist only in the deleted context, they will be permanently deleted from the system).</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can delete any context without restrictions</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can delete any context without restrictions</li>
	 * <li><strong>Context-Manager</strong>: Can delete contexts only if they are Context-Manager of the parent context</li>
	 * <li><strong>Other Users</strong>: Cannot delete contexts (will receive authorization errors).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * String contextUuidString = "c0f314e7-2807-4241-a792-2a6c79ed4fd0";
	 * 
	 * // Delete the context using string UUID
	 * try {
	 *     boolean deleted = client.delete(contextUuidString);
	 *     if (deleted) {
	 *         System.out.println("Context successfully deleted");
	 *     }
	 * } catch (IllegalArgumentException e) {
	 *     System.err.println("Invalid UUID format: " + contextUuidString);
	 * }
	 * </pre>
	 * 
	 * @param uuid the UUID of the context (as string) to delete
	 * @return {@code true} if the context was successfully deleted, {@code false} otherwise
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if the context cannot be deleted due to authorization issues, child contexts, or other errors
	 * @throws IllegalArgumentException if the provided string is not a valid UUID format
	 */
	public boolean delete(String uuid) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Changes the state of an existing context by extracting its UUID from the provided context object.
	 * This is a convenience method that delegates to the internal change state operation using the context's ID.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PATCH /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>State-Only Operation:</strong> Only the state property from the context object is evaluated and applied</li>
	 * <li><strong>Property Isolation:</strong> All other properties in the context object are ignored during state change</li>
	 * <li><strong>Complete Response:</strong> Returns the complete updated context with the new state and all other properties</li>
	 * <li>Extracts the UUID from the provided context object for identification.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context state successfully changed</li>
	 * <li><strong>404 Not Found</strong>: Context with the specified UUID is not found</li>
	 * <li><strong>400 Bad Request</strong>: Invalid state value or validation failure</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to change the context state</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>Context object must contain a valid UUID for identification</li>
	 * <li>Context object must contain a valid state value</li>
	 * <li>State values must be valid according to the ContextState class</li>
	 * <li>All other properties in the context object are ignored and will not be modified.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can change state of any context without restrictions</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can change state of any context without restrictions</li>
	 * <li><strong>Context-Manager</strong>: Can change state of contexts only if they are Context-Manager of the parent context</li>
	 * <li><strong>Other Users</strong>: Cannot change context states (will receive authorization errors).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * // Get context and change its state
	 * Context context = client.read(contextUuid);
	 * context.setState(ContextState.ACTIVE); // Set new state
	 * 
	 * // Change the context state
	 * Context updatedContext = client.changeState(context);
	 * System.out.println("New state: " + updatedContext.getState());
	 * 
	 * // All other properties are preserved
	 * System.out.println("Context name unchanged: " + updatedContext.getName());
	 * </pre>
	 * 
	 * @param context the context object containing the UUID and new state to apply
	 * @return the complete Context object with the updated state and full metadata
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if the state cannot be changed due to authorization issues, invalid state, or other errors
	 */
	public Context changeState(Context context) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Changes the state of an existing context from its JSON representation.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PATCH /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>State-Only Operation:</strong> Only the state property from the JSON is evaluated and applied</li>
	 * <li><strong>Property Isolation:</strong> All other properties in the JSON are ignored during state change</li>
	 * <li><strong>Complete Response:</strong> Returns the complete updated context as JSON with the new state and all other properties</li>
	 * <li>Parses the JSON to extract UUID and state information.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context state successfully changed</li>
	 * <li><strong>404 Not Found</strong>: Context with the specified UUID is not found</li>
	 * <li><strong>400 Bad Request</strong>: Invalid JSON format, state value, or validation failure</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to change the context state</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>JSON must contain a valid Context representation</li>
	 * <li>JSON must include a valid UUID field to identify the target context</li>
	 * <li>JSON must include a valid state field with the new state value</li>
	 * <li>State values must be valid according to the ContextState class</li>
	 * <li>All other JSON properties are ignored and will not be modified.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can change state of any context without restrictions</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can change state of any context without restrictions</li>
	 * <li><strong>Context-Manager</strong>: Can change state of contexts only if they are Context-Manager of the parent context</li>
	 * <li><strong>Other Users</strong>: Cannot change context states (will receive authorization errors).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * // JSON with UUID and new state (other properties ignored)
	 * String contextJson = """
	 *     {
	 *         "@class": "Context",
	 *         "id": "c0f314e7-2807-4241-a792-2a6c79ed4fd0",
	 *         "state": "ACTIVE",
	 *         "name": "This will be ignored"
	 *     }
	 *     """;
	 * 
	 * // Change the context state from JSON
	 * String updatedContextJson = client.changeState(contextJson);
	 * 
	 * // Parse response to verify state change
	 * Context updatedContext = ElementMapper.unmarshal(Context.class, updatedContextJson);
	 * System.out.println("New state: " + updatedContext.getState());
	 * </pre>
	 * 
	 * @param context the JSON string representation containing the context UUID and new state
	 * @return the complete updated context as JSON string with the new state and full metadata
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if the state cannot be changed due to authorization issues, invalid state, invalid JSON, or other errors
	 */
	public String changeState(String context) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Changes the state of an existing context using its UUID and the desired state.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PATCH /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>State-Only Operation:</strong> Only the state property is evaluated and applied by the server</li>
	 * <li><strong>Complete Response:</strong> Returns the complete updated context with the new state and all other properties</li>
	 * <li><strong>Type Safety:</strong> Uses ContextState enum instead of string, preventing invalid state values at compile time</li>
	 * <li><strong>Simplicity:</strong> No need to construct a full context object or JSON string.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context state successfully changed</li>
	 * <li><strong>404 Not Found</strong>: Context with the specified UUID is not found</li>
	 * <li><strong>400 Bad Request</strong>: Invalid state value or validation failure</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to change the context state</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>UUID must be a valid UUID format</li>
	 * <li>State values must be valid according to the ContextState enum</li>
	 * <li>Context must exist and be accessible to the user.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can change state of any context without restrictions</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can change state of any context without restrictions</li>
	 * <li><strong>Context-Manager</strong>: Can change state of contexts only if they are Context-Manager of the parent context</li>
	 * <li><strong>Other Users</strong>: Cannot change context states (will receive authorization errors).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * UUID contextUuid = UUID.fromString("c0f314e7-2807-4241-a792-2a6c79ed4fd0");
	 * 
	 * // Change context state directly
	 * Context updatedContext = client.changeState(contextUuid, ContextState.ACTIVE);
	 * System.out.println("New state: " + updatedContext.getState());
	 * 
	 * // All other properties are preserved
	 * System.out.println("Context name unchanged: " + updatedContext.getName());
	 * 
	 * // Example with different states
	 * client.changeState(contextUuid, ContextState.INACTIVE);
	 * client.changeState(contextUuid, ContextState.READY);
	 * </pre>
	 * 
	 * @param uuid the UUID of the context whose state should be changed
	 * @param state the new state to apply to the context
	 * @return the complete Context object with the updated state and full metadata
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if the state cannot be changed due to authorization issues, invalid state, or other errors
	 */
	public Context changeState(UUID uuid, ContextState state) throws ContextNotFoundException, ResourceRegistryException;

	/**
	 * Changes the state of an existing context using its UUID (as string) and the desired state (as string).
	 * This method provides string-based parameters for scenarios where enum types are not available or preferred.
	 * 
	 * <strong>Corresponding REST API:</strong> {@code PATCH /contexts/{context-uuid}}
	 * 
	 * <strong>Operation Behavior:</strong>
	 * <ul>
	 * <li><strong>State-Only Operation:</strong> Only the state property is evaluated and applied by the server</li>
	 * <li><strong>Complete Response:</strong> Returns the complete updated context as JSON with the new state and all other properties</li>
	 * <li>Converts string parameters to appropriate types and delegates to internal change state operation.</li>
	 * </ul>
	 * 
	 * <strong>HTTP Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Context state successfully changed</li>
	 * <li><strong>404 Not Found</strong>: Context with the specified UUID is not found</li>
	 * <li><strong>400 Bad Request</strong>: Invalid UUID format, state value, or validation failure</li>
	 * <li><strong>403 Forbidden</strong>: User lacks authorization to change the context state</li>
	 * <li><strong>401 Unauthorized</strong>: Invalid or missing authentication credentials.</li>
	 * </ul>
	 * 
	 * <strong>Input Processing:</strong>
	 * <ul>
	 * <li>UUID string must be a valid UUID format</li>
	 * <li>State strings must correspond to valid ContextState enum values</li>
	 * <li>Context must exist and be accessible to the user.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * <ul>
	 * <li><strong>IS-Manager</strong>: Can change state of any context without restrictions</li>
	 * <li><strong>Infrastructure-Manager</strong>: Can change state of any context without restrictions</li>
	 * <li><strong>Context-Manager</strong>: Can change state of contexts only if they are Context-Manager of the parent context</li>
	 * <li><strong>Other Users</strong>: Cannot change context states (will receive authorization errors).</li>
	 * </ul>
	 * 
	 * <strong>Example Usage:</strong>
	 * <pre>
	 * ResourceRegistryContextClient client = ResourceRegistryContextClientFactory.create();
	 * 
	 * String contextUuidString = "c0f314e7-2807-4241-a792-2a6c79ed4fd0";
	 * 
	 * // Change context state using strings
	 * String updatedContextJson = client.changeState(contextUuidString, "ACTIVE");
	 * 
	 * // Parse response to verify state change
	 * Context updatedContext = ElementMapper.unmarshal(Context.class, updatedContextJson);
	 * System.out.println("New state: " + updatedContext.getState());
	 * 
	 * // Example with different states
	 * client.changeState(contextUuidString, "INACTIVE");
	 * client.changeState(contextUuidString, "READY");
	 * 
	 * // Error handling for invalid inputs
	 * try {
	 *     client.changeState("invalid-uuid", "ACTIVE");
	 * } catch (IllegalArgumentException e) {
	 *     System.err.println("Invalid UUID format");
	 * }
	 * 
	 * try {
	 *     client.changeState(contextUuidString, "INVALID_STATE");
	 * } catch (ResourceRegistryException e) {
	 *     System.err.println("Invalid state value");
	 * }
	 * </pre>
	 * 
	 * @param uuid the UUID of the context (as string) whose state should be changed
	 * @param state the new state (as string) to apply to the context
	 * @return the complete updated context as JSON string with the new state and full metadata
	 * @throws ContextNotFoundException if the context with the specified UUID is not found
	 * @throws ResourceRegistryException if the state cannot be changed due to authorization issues, invalid state, or other errors
	 * @throws IllegalArgumentException if the provided UUID string is not a valid UUID format or if the state string is not a valid ContextState value
	 */
	public String changeState(String uuid, String state) throws ContextNotFoundException, ResourceRegistryException;

}

