package org.gcube.informationsystem.resourceregistry.query;

import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.query.InvalidQueryException;
import org.gcube.informationsystem.resourceregistry.api.rest.AccessPath;
import org.gcube.informationsystem.resourceregistry.context.ContextUtility;
import org.gcube.informationsystem.resourceregistry.context.security.SecurityContext;
import org.gcube.informationsystem.resourceregistry.context.security.SecurityContext.PermissionMode;
import org.gcube.informationsystem.resourceregistry.er.ERManagement;
import org.gcube.informationsystem.resourceregistry.er.ERManagementUtility;
import org.gcube.informationsystem.resourceregistry.utils.Utility;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.orientechnologies.orient.core.record.impl.ODocument;
import com.orientechnologies.orient.core.sql.query.OSQLSynchQuery;
import com.tinkerpop.blueprints.Element;
import com.tinkerpop.blueprints.impls.orient.OrientGraph;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class QueryImpl implements Query {
	
	private static Logger logger = LoggerFactory.getLogger(QueryImpl.class);
	
	@Override
	public String query(String query, Integer limit, String fetchPlan, boolean raw) throws InvalidQueryException {
		if(limit == null) {
			limit = AccessPath.DEFAULT_LIMIT;
		}
		limit = (limit <= 0) ? AccessPath.UNBOUNDED : limit;
		
		OrientGraph orientGraph = null;
		
		try {
			SecurityContext securityContext = ContextUtility.getCurrentSecurityContext();
			
			orientGraph = securityContext.getGraph(PermissionMode.READER);
			orientGraph.setAutoStartTx(false);
			orientGraph.begin();
			
			OSQLSynchQuery<ODocument> osqlSynchQuery = new OSQLSynchQuery<>(query, limit);
			osqlSynchQuery.setFetchPlan(fetchPlan);
			osqlSynchQuery.setCacheableResult(true);
			
			logger.debug("Going to execute query : \"{}\", fetchPlan : \"{}\", limit : {}", osqlSynchQuery.getText(),
					osqlSynchQuery.getFetchPlan(), osqlSynchQuery.getLimit());
			
			Iterable<Element> elements = orientGraph.command(osqlSynchQuery).execute();
			
			ObjectMapper objectMapper = new ObjectMapper();
			ArrayNode arrayNode = objectMapper.createArrayNode();
			
			
			for(Element element : elements) {
				try {
					JsonNode jsonNode = null;
					if(raw) {
						jsonNode = Utility.toJsonNode(element, false);
					} else {
						@SuppressWarnings("rawtypes")
						ERManagement erManagement = ERManagementUtility.getERManagement(securityContext, orientGraph,
								element);
						jsonNode = erManagement.serializeAsJson();
					}
					arrayNode.add(jsonNode);
					
				} catch(ResourceRegistryException e) {
					logger.error("Unable to correctly serialize {}. It will be excluded from results. {}",
							element.toString(), Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
				}
			}
			
			return objectMapper.writeValueAsString(arrayNode);
			
		} catch(Exception e) {
			throw new InvalidQueryException(e.getMessage());
		} finally {
			if(orientGraph != null) {
				orientGraph.shutdown();
			}
		}
		
	}
	
	@Override
	public String gremlinQuery(String query) throws InvalidQueryException {
		throw new UnsupportedOperationException();
		
		/*
		OGremlinHelper.global().create();
		
		ODatabaseDocumentTx oDatabaseDocumentTx = null;
		try {
			oDatabaseDocumentTx = ContextUtility.getActualSecurityContextDatabaseTx(PermissionMode.READER);
			
			String finalQuery = String.format("select gremlin('%s')", query);
			OCommandSQL OCommandSQL = new OCommandSQL(finalQuery);
			OCommandRequest oCommandRequest = oDatabaseDocumentTx.command(OCommandSQL);
			OBasicResultSet res = oCommandRequest.execute();
			
			Iterator iterator = res.iterator();
			
			while(iterator.hasNext()) {
				ODocument oDocument = (ODocument) iterator.next();
				logger.debug("{}", oDocument);
			}
			
			return res.toString();
			
		} catch(Exception e) {
			throw new InvalidQueryException(e.getMessage());
		} finally {
			if(oDatabaseDocumentTx != null) {
				oDatabaseDocumentTx.close();
			}
		}
		*/
	}
	
}
