package org.gcube.informationsystem.resourceregistry.er.entity;

import java.util.Iterator;

import org.gcube.informationsystem.model.reference.AccessType;
import org.gcube.informationsystem.model.reference.entities.Resource;
import org.gcube.informationsystem.model.reference.relations.ConsistsOf;
import org.gcube.informationsystem.model.reference.relations.IsRelatedTo;
import org.gcube.informationsystem.resourceregistry.api.exceptions.NotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceAlreadyPresentException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceAvailableInAnotherContextException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entity.resource.ResourceNotFoundException;
import org.gcube.informationsystem.resourceregistry.context.security.SecurityContext;
import org.gcube.informationsystem.resourceregistry.context.security.SecurityContext.PermissionMode;
import org.gcube.informationsystem.resourceregistry.er.relation.ConsistsOfManagement;
import org.gcube.informationsystem.resourceregistry.er.relation.IsRelatedToManagement;
import org.gcube.informationsystem.resourceregistry.er.relation.RelationManagement;
import org.gcube.informationsystem.resourceregistry.utils.Utility;

import com.fasterxml.jackson.databind.JsonNode;
import com.tinkerpop.blueprints.Direction;
import com.tinkerpop.blueprints.Edge;
import com.tinkerpop.blueprints.Vertex;
import com.tinkerpop.blueprints.impls.orient.OrientEdge;
import com.tinkerpop.blueprints.impls.orient.OrientEdgeType;
import com.tinkerpop.blueprints.impls.orient.OrientGraph;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
public class ResourceManagement extends EntityManagement<Resource> {
	
	public ResourceManagement() {
		super(AccessType.RESOURCE);
	}
	
	public ResourceManagement(SecurityContext workingContext, OrientGraph orientGraph) {
		super(AccessType.RESOURCE, workingContext, orientGraph);
	}
	
	@Override
	protected ResourceNotFoundException getSpecificElementNotFoundException(NotFoundException e) {
		return new ResourceNotFoundException(e.getMessage(), e.getCause());
	}
	
	@Override
	protected ResourceAvailableInAnotherContextException getSpecificERAvailableInAnotherContextException(
			String message) {
		return new ResourceAvailableInAnotherContextException(message);
	}
	
	@Override
	protected ResourceAlreadyPresentException getSpecificERAlreadyPresentException(String message) {
		return new ResourceAlreadyPresentException(message);
	}
	
	@Override
	public String serialize() throws ResourceRegistryException {
		return serializeAsJson().toString();
	}
	
	@SuppressWarnings("unchecked")
	@Override
	public JsonNode serializeAsJson() throws ResourceRegistryException {
		
		JsonNode sourceResource = serializeSelfOnly();
		
		/*
		 * Cannot get ConsistsOf edge only because is not polymorphic for a
		 * com.tinkerpop.blueprints.Vertex vertex.getEdges(Direction.OUT,
		 * ConsistsOf.NAME); TODO Looks for a different query
		 */
		
		Iterable<Edge> edges = getElement().getEdges(Direction.OUT);
		for(Edge edge : edges) {
			
			@SuppressWarnings("rawtypes")
			RelationManagement relationManagement = getRelationManagement(edge);
			relationManagement.setReload(reload);
			
			if(relationManagement.giveMeSourceEntityManagementAsIs() == null) {
				relationManagement.setSourceEntityManagement(this);
			}
			
			if(relationManagement.giveMeSourceEntityManagementAsIs() != this) {
				StringBuilder errorMessage = new StringBuilder();
				errorMessage.append("SourceEntityManagement for ");
				errorMessage.append(relationManagement.getClass().getSimpleName());
				errorMessage.append(" is not the one expected. ");
				errorMessage.append(Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
				throw new ResourceRegistryException(errorMessage.toString());
			}
			
			if(relationManagement instanceof ConsistsOfManagement) {
				try {
					JsonNode consistsOf = relationManagement.serializeAsJson(true, true);
					sourceResource = addConsistsOf(sourceResource, consistsOf);
				} catch(ResourceRegistryException e) {
					logger.error("Unable to correctly serialize {}. {}", edge, Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
					throw e;
				} catch(Exception e) {
					logger.error("Unable to correctly serialize {}. {}", edge, Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
					throw new ResourceRegistryException(e);
				}
				
			}
			/*
			 * This comment is just to show that IsRelatedTo is not serialized by default as
			 * design choice and not because forget
			 * 
			 * else if(orientEdgeType.isSubClassOf(IsRelatedTo.NAME)){ JsonNode
			 * isRelatedTo = relationManagement.serializeAsJson(true, true); sourceResource
			 * = addIsRelatedTo(sourceResource, isRelatedTo); }
			 */
		}
		
		return sourceResource;
	}
	
	public static JsonNode addConsistsOf(JsonNode sourceResource, JsonNode consistsOf)
			throws ResourceRegistryException {
		return addRelation(sourceResource, consistsOf, AccessType.CONSISTS_OF.lowerCaseFirstCharacter());
	}
	
	public static JsonNode addIsRelatedTo(JsonNode sourceResource, JsonNode isRelatedTo)
			throws ResourceRegistryException {
		return addRelation(sourceResource, isRelatedTo, AccessType.IS_RELATED_TO.lowerCaseFirstCharacter());
	}
	
	@Override
	protected Vertex reallyCreate() throws ResourceAlreadyPresentException, ResourceRegistryException {
		
		createVertex();
		
		String property = AccessType.CONSISTS_OF.lowerCaseFirstCharacter();
		if(jsonNode.has(property)) {
			JsonNode jsonNodeArray = jsonNode.get(property);
			for(JsonNode consistOfJsonNode : jsonNodeArray) {
				ConsistsOfManagement com = new ConsistsOfManagement(getWorkingContext(), orientGraph);
				com.setJsonNode(consistOfJsonNode);
				com.setSourceEntityManagement(this);
				com.internalCreate();
				addToRelationManagement(com);
			}
		}
		
		property = AccessType.IS_RELATED_TO.lowerCaseFirstCharacter();
		if(jsonNode.has(property)) {
			JsonNode jsonNodeArray = jsonNode.get(property);
			for(JsonNode relationJsonNode : jsonNodeArray) {
				IsRelatedToManagement irtm = new IsRelatedToManagement(getWorkingContext(), orientGraph);
				irtm.setJsonNode(relationJsonNode);
				irtm.setSourceEntityManagement(this);
				irtm.internalCreate();
				addToRelationManagement(irtm);
			}
		}
		
		return element;
	}
	
	@Override
	protected Vertex reallyUpdate() throws ResourceNotFoundException, ResourceRegistryException {
		
		getElement();
		
		String property = AccessType.CONSISTS_OF.lowerCaseFirstCharacter();
		if(jsonNode.has(property)) {
			JsonNode jsonNodeArray = jsonNode.get(property);
			for(JsonNode relationJsonNode : jsonNodeArray) {
				ConsistsOfManagement com = new ConsistsOfManagement(getWorkingContext(), orientGraph);
				com.setJsonNode(relationJsonNode);
				com.internalCreateOrUdate();
				addToRelationManagement(com);
			}
		}
		
		property = AccessType.IS_RELATED_TO.lowerCaseFirstCharacter();
		if(jsonNode.has(property)) {
			JsonNode jsonNodeArray = jsonNode.get(property);
			for(JsonNode relationJsonNode : jsonNodeArray) {
				IsRelatedToManagement irtm = new IsRelatedToManagement(getWorkingContext(), orientGraph);
				irtm.setJsonNode(relationJsonNode);
				irtm.internalUpdate();
				addToRelationManagement(irtm);
			}
		}
		
		return element;
	}
	
	@SuppressWarnings("unchecked")
	@Override
	protected boolean reallyDelete() throws ResourceNotFoundException, ResourceRegistryException {
		// internalDeleteResource(orientGraph, uuid, null);
		
		getElement();
		
		Iterable<Edge> iterable = element.getEdges(Direction.OUT);
		Iterator<Edge> iterator = iterable.iterator();
		while(iterator.hasNext()) {
			
			Edge edge = iterator.next();
			OrientEdgeType orientEdgeType = ((OrientEdge) edge).getType();
			@SuppressWarnings("rawtypes")
			RelationManagement relationManagement = null;
			if(orientEdgeType.isSubClassOf(IsRelatedTo.NAME)) {
				relationManagement = new IsRelatedToManagement(getWorkingContext(), orientGraph);
			} else if(orientEdgeType.isSubClassOf(ConsistsOf.NAME)) {
				relationManagement = new ConsistsOfManagement(getWorkingContext(), orientGraph);
			} else {
				logger.warn("{} is not a {} nor a {}. {}", Utility.toJsonString(edge, true), IsRelatedTo.NAME,
						ConsistsOf.NAME, Utility.SHOULD_NOT_OCCUR_ERROR_MESSAGE);
			}
			if(relationManagement != null) {
				relationManagement.setElement(edge);
				relationManagement.internalDelete();
			}
			
		}
		
		element.remove();
		
		return true;
	}
	
	public String all(boolean polymorphic) throws ResourceRegistryException {
		try {
			orientGraph = getWorkingContext().getGraph(PermissionMode.READER);
			
			return reallyGetAll(polymorphic);
		} catch(ResourceRegistryException e) {
			throw e;
		} catch(Exception e) {
			throw new ResourceRegistryException(e);
		} finally {
			if(orientGraph != null) {
				orientGraph.shutdown();
			}
		}
	}
	
}
