package org.gcube.informationsystem.resourceregistry.rest;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

import org.eclipse.microprofile.openapi.annotations.Operation;
import org.eclipse.microprofile.openapi.annotations.media.Content;
import org.eclipse.microprofile.openapi.annotations.responses.APIResponse;
import org.eclipse.microprofile.openapi.annotations.tags.Tag;
import org.gcube.com.fasterxml.jackson.core.JsonProcessingException;
import org.gcube.com.fasterxml.jackson.databind.JsonNode;
import org.gcube.com.fasterxml.jackson.databind.ObjectMapper;
import org.gcube.com.fasterxml.jackson.databind.node.ArrayNode;
import org.gcube.informationsystem.contexts.reference.entities.Context;
import org.gcube.informationsystem.resourceregistry.ResourceInitializer;
import org.gcube.informationsystem.resourceregistry.api.exceptions.ResourceRegistryException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.contexts.ContextNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.entities.resource.ResourceNotFoundException;
import org.gcube.informationsystem.resourceregistry.api.exceptions.types.SchemaViolationException;
import org.gcube.informationsystem.resourceregistry.api.rest.SharingPath;
import org.gcube.informationsystem.resourceregistry.api.rest.SharingPath.SharingOperation;
import org.gcube.informationsystem.resourceregistry.base.ElementManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.ERManagement;
import org.gcube.informationsystem.resourceregistry.instances.model.ERManagementUtility;
import org.gcube.informationsystem.resourceregistry.rest.requests.ServerRequestInfo;
import org.gcube.informationsystem.resourceregistry.utils.UUIDUtility;
import org.gcube.informationsystem.utils.TypeUtility;

import jakarta.ws.rs.DefaultValue;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import jakarta.ws.rs.PathParam;
import jakarta.ws.rs.Produces;
import jakarta.ws.rs.QueryParam;

/**
 * @author Luca Frosini (ISTI - CNR)
 */
@Path(SharingPath.SHARING_PATH_PART)
@Tag(name = "Sharing", description = "Operations for sharing and unsharing instances between contexts.")
public class SharingManager extends BaseRest {

	public SharingManager() {
		super();
	}
	
	protected String serializeAffectedInstaces(ObjectMapper objectMapper, Map<UUID,JsonNode> affectedInstances) throws ResourceRegistryException {
		ArrayNode arrayNode = objectMapper.createArrayNode();
		for(JsonNode jsonNode : affectedInstances.values()) {
			arrayNode.add(jsonNode);
		}
		
		try {
			return objectMapper.writeValueAsString(arrayNode);
		} catch (JsonProcessingException e) {
			throw new ResourceRegistryException(e);
		}
	}
	
	/**
	 * Add/Remove to/from the context identified by context-uuid path parameter the list of instances contained in the body of the request.
	 * The the body is the following 
	 * 
	 * [
	 *  {"type" : "HostingNode", "id" : "16032d09-3823-444e-a1ff-a67de4f350a8" },
	 *  {"type" : "Hosts", "id" : "97ab8a6b-6b1b-4868-b8fc-ba48d0439ba9"},
	 *  {"type" : "EService", "id" : "d3b1a29b-aa70-4a5a-be83-361a4209dd3e"}
	 * ]
	 * 
	 * 
	 * Each instance is managed without considering the propagation constraint of relations. 
	 * 
	 * POST /sharing/contexts/{context-uuid}?operation=ADD|REMOVE&amp;[dryRun=true]
	 * 
	 * e.g 
	 * POST /resource-registry/contexts/67062c11-9c3a-4906-870d-7df6a43408b0?operation=ADD&amp;dryRun=true;
	 * POST /resource-registry/contexts/67062c11-9c3a-4906-870d-7df6a43408b0?operation=REMOVE.
	 * 
	 * where
	 * 67062c11-9c3a-4906-870d-7df6a43408b0/ is the Context UUID
	 * 
	 * The body contains the list of instances to add/remove to/from the context identified by <strong>context-uuid</strong>
	 * 
	 */
	// @POST
	// @Path("/{" + SharingPath.CONTEXTS_PATH_PART + "/{" + AccessPath.CONTEXT_UUID_PATH_PARAM + "}")
	public String addRemoveNoPropagationConstraint(
			@PathParam(ContextManager.CONTEXT_UUID_PATH_PARAMETER) String contextId,  
			@QueryParam(SharingPath.OPERATION_QUERY_PARAMETER) SharingOperation operation,
			@QueryParam(SharingPath.DRY_RUN_QUERY_QUERY_PARAMETER) @DefaultValue("false") Boolean dryRun,
			@QueryParam(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER) @DefaultValue("false") Boolean forceAddToContext,
			String body)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {

		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkAllIncludeQueryParameters();
		
		try {
			StringBuffer calledMethod = new StringBuffer(); 
			if(dryRun==null) {
				dryRun = false;
			}
			
			if(dryRun) {
				calledMethod.append("dryRun");
			}
			
			if(operation == SharingOperation.ADD) {
				logger.info("Requested {} {} to {} with UUID {}", dryRun? "a dry run for adding": "to add", body, Context.NAME, contextId);
				calledMethod.append("AddToContext");
			}else {
				logger.info("Requested {} {} from {} with UUID {}", dryRun? "a dry run for removing": "to remove", body, Context.NAME, contextId);
				calledMethod.append("RemoveFromContext");
			}
			calledMethod.append("NoPropagationConstraint");
			setAccountingMethod(calledMethod.toString());
	
			ObjectMapper objectMapper = new ObjectMapper();
			ArrayNode arrayNode = (ArrayNode) objectMapper.readTree(body);
			
			Map<UUID,JsonNode> expectedInstances = new HashMap<>();
			
			for(JsonNode node : arrayNode) {
				@SuppressWarnings("unused")
				String type = TypeUtility.getTypeName(node);
				UUID uuid = UUIDUtility.getUUID(node);
				expectedInstances.put(uuid, node);
			}
			
			@SuppressWarnings("unused")
			UUID contextUUID = UUID.fromString(contextId);
			Map<UUID,JsonNode> affectedInstances = null;
			if(operation == SharingOperation.ADD) {
				// affectedInstances = ERManagementUtility.addToContextNoPropagationConstraint(expectedInstances, contextUUID, dryRun);
			}else {
				// affectedInstances = ERManagementUtility.removeFromContextNoPropagationConstraint(expectedInstances, contextUUID, dryRun);
			}
			
			return serializeAffectedInstaces(objectMapper, affectedInstances);
		}catch (ResourceRegistryException e) {
			throw e;
		}catch (Exception e) {
			throw new ResourceRegistryException(e);
		}
		
	}
	
	/** 
	 * Add or remove an instance from a context. The operation can have a cascade effect due to propagation constraints.
	 * Returns the list of instances affected by the add/remove operation on the target instance identified by UUID path parameter.
	 * 
	 * <strong>REST Endpoint:</strong> {@code POST /sharing/contexts/{context-uuid}/{type-name}/{instance-uuid}?operation={ADD|REMOVE}[&dryRun={true|false}][&forceAddToContext={true|false}][&includeMeta={true|false}][&allMeta={true|false}][&includeContexts={true|false}]}
	 * 
	 * <strong>Request Examples:</strong>
	 * <ul>
	 * <li>POST /sharing/contexts/67062c11-9c3a-4906-870d-7df6a43408b0/HostingNode/16032d09-3823-444e-a1ff-a67de4f350a8?operation=ADD (add HostingNode to context)</li>
	 * <li>POST /sharing/contexts/67062c11-9c3a-4906-870d-7df6a43408b0/HostingNode/16032d09-3823-444e-a1ff-a67de4f350a8?operation=REMOVE (remove HostingNode from context)</li>
	 * <li>POST /sharing/contexts/67062c11-9c3a-4906-870d-7df6a43408b0/HostingNode/16032d09-3823-444e-a1ff-a67de4f350a8?operation=ADD&amp;dryRun=true (preview what would be affected by adding HostingNode)</li>
	 * <li>POST /sharing/contexts/67062c11-9c3a-4906-870d-7df6a43408b0/HostingNode/16032d09-3823-444e-a1ff-a67de4f350a8?operation=ADD&amp;forceAddToContext=true (force add even if instance is not in current context)</li>
	 * <li>POST /sharing/contexts/67062c11-9c3a-4906-870d-7df6a43408b0/HostingNode/16032d09-3823-444e-a1ff-a67de4f350a8?operation=ADD&amp;includeMeta=true&amp;includeContexts=true (add with metadata and context information in response).</li>
	 * </ul>
	 * 
	 * Where:
	 * <ul>
	 * <li>67062c11-9c3a-4906-870d-7df6a43408b0 is the Context UUID</li>
	 * <li>16032d09-3823-444e-a1ff-a67de4f350a8 is the HostingNode UUID.</li>
	 * </ul>
	 * 
	 * <strong>Path Parameters:</strong>
	 * <ul>
	 * <li><strong>context-uuid</strong>: The UUID of the target context where the instance should be added or removed</li>
	 * <li><strong>type-name</strong>: The name of the Information System type (e.g., "EService", "ContactFacet")</li>
	 * <li><strong>instance-uuid</strong>: The unique identifier of the instance to add or remove.</li>
	 * </ul>
	 * 
	 * <strong>Query Parameters:</strong>
	 * 
	 * <strong>operation</strong> (required):
	 * <ul>
	 * <li>Specifies the operation to perform on the instance</li>
	 * <li>Values: ADD (add instance to context) | REMOVE (remove instance from context)</li>
	 * <li>ADD: Adds the specified instance to the target context, respecting propagation constraints</li>
	 * <li>REMOVE: Removes the specified instance from the target context, respecting propagation constraints.</li>
	 * </ul>
	 * 
	 * <strong>dryRun</strong> (optional):
	 * <ul>
	 * <li>Whether to perform a preview of the operation without actually executing it</li>
	 * <li>Default value: false (execute the operation)</li>
	 * <li>Values: true | false</li>
	 * <li>When true: Returns the list of instances that would be affected by the operation, but does not perform any actual changes</li>
	 * <li>When false: Performs the actual add/remove operation and returns the list of instances that were affected</li>
	 * <li>Useful for previewing the cascade effects before committing to the operation.</li>
	 * </ul>
	 * 
	 * <strong>forceAddToContext</strong> (optional):
	 * <ul>
	 * <li>Forces the addition of an instance to a context even if the instance is not present in the current context</li>
	 * <li>Default value: false (standard validation applies)</li>
	 * <li>Values: true | false</li>
	 * <li><strong>Authorization Required</strong>: Only available to IS-Manager and Infrastructure-Manager</li>
	 * <li>When true: Allows adding an instance to a context even if the instance exists only in other contexts</li>
	 * <li>When false: Standard validation applies - instance must be accessible in the current context for ADD operations</li>
	 * <li>Only applies to ADD operations; ignored for REMOVE operations.</li>
	 * </ul>
	 * 
	 * <strong>includeMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata in the response for affected instances</li>
	 * <li>Default value: false (basic information only)</li>
	 * <li>Values: true | false</li>
	 * <li>Example: ?includeMeta=true (includes metadata with appropriate filtering based on user role)</li>
	 * <li>Metadata includes the created/updated instance and nested elements based on polymorphic setting</li>
	 * <li><strong>Restriction:</strong> IS-Manager, Infrastructure-Manager, and Context-Manager see complete metadata including sensitive information (createdBy, lastUpdatedBy); other users see filtered metadata with sensitive fields obfuscated.</li>
	 * </ul>
	 * 
	 * <strong>allMeta</strong> (optional):
	 * <ul>
	 * <li>Whether to include metadata for all nested instances (ConsistsOf relations, Facets, etc.) in the response</li>
	 * <li>Must be used in conjunction with includeMeta=true</li>
	 * <li>Default value: false (metadata only for main instance, more human-readable)</li>
	 * <li>Values: true | false</li>
	 * <li>Example: ?includeMeta=true&amp;allMeta=true (includes metadata for all nested elements)</li>
	 * <li>Purpose: When false, produces more human-readable responses with less JSON to process</li>
	 * <li><strong>Available to all users</strong> (not restricted to administrators).</li>
	 * </ul>
	 * 
	 * <strong>includeContexts</strong> (optional):
	 * <ul>
	 * <li>Whether to include the list of contexts where the instance and its nested elements are available in the response</li>
	 * <li>Default value: false (context information not included)</li>
	 * <li>Values: true | false</li>
	 * <li>Example: ?includeContexts=true (shows context availability for the instance and nested elements)</li>
	 * <li><strong>Note:</strong> A Resource is present in all contexts that form the union of contexts of all its Facets</li>
	 * <li><strong>Note:</strong> ConsistsOf relations are present in all contexts where their target Facets are present</li>
	 * <li><strong>Note:</strong> Unlike read operations, this parameter does not require hierarchical=true.</li>
	 * </ul>
	 * 
	 * <strong>Authorization Requirements:</strong>
	 * 
	 * <strong>IS-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD/REMOVE operations on any instance in any context</li>
	 * <li>Can use forceAddToContext=true to add instances from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Infrastructure-Manager:</strong>
	 * <ul>
	 * <li>Can perform ADD/REMOVE operations on any instance in any context</li>
	 * <li>Can use forceAddToContext=true to add instances from other contexts</li>
	 * <li>Full administrative privileges across all contexts</li>
	 * <li>See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Other Users:</strong>
	 * <ul>
	 * <li>Can perform ADD/REMOVE operations only on instances accessible in their current context</li>
	 * <li>Cannot use forceAddToContext=true (parameter is ignored if provided)</li>
	 * <li>Operations are limited to instances they have access to in their authorized context</li>
	 * <li>See filtered metadata with sensitive fields obfuscated when includeMeta=true.</li>
	 * </ul>
	 * 
	 * <strong>Operation Behavior:</strong>
	 * 
	 * <strong>Propagation Constraints and Cascade Effects:</strong>
	 * 
	 * <strong>ADD Operation Cascading:</strong>
	 * <ul>
	 * <li>When an instance is added to a context, related instances may also be automatically added due to propagation constraints</li>
	 * <li>The response includes all instances that were added as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>REMOVE Operation Cascading:</strong>
	 * <ul>
	 * <li>When an instance is removed from a context, related instances may also be automatically removed due to propagation constraints</li>
	 * <li>The response includes all instances that were removed as part of the cascade effect</li>
	 * <li>Propagation follows the schema-defined relationships and their constraints.</li>
	 * </ul>
	 * 
	 * <strong>Schema Validation for ConsistsOf Relations and Facets:</strong>
	 * <ul>
	 * <li>When operations are performed on ConsistsOf relations or Facet instances, the parent Resource is automatically validated against its schema definition</li>
	 * <li>If the Resource is no longer compliant with its schema after the add/remove operation, a SchemaViolationException is thrown</li>
	 * <li>This results in an HTTP 400 Bad Request response with details about the schema violation</li>
	 * <li><strong>Example scenarios that violate schema constraints:</strong></li>
	 * </ul>
	 *   • Removing a required/mandatory Facet or ConsistsOf relation from a Resource;
	 *   • Adding/removing instances that violate cardinality constraints (min/max occurrences) defined in the schema.
	 * <ul>
	 * <li>All these scenarios make the Resource incomplete or invalid according to its schema definition</li>
	 * <li><strong>Constraint Sources</strong>: Constraints are defined by the specific type definitions that have been installed on the system</li>
	 * <li>Schema validation ensures the integrity and consistency by enforcing the constraints defined in each type's schema during operations.</li>
	 * </ul>
	 * 
	 * <strong>Response Codes:</strong>
	 * <ul>
	 * <li><strong>200 OK</strong>: Operation completed successfully (instances added/removed)</li>
	 * <li><strong>400 Bad Request</strong>: Schema validation failed (SchemaViolationException) - occurs when operation would violate schema constraints</li>
	 * <li><strong>404 Not Found</strong>: Instance, type, or context not found</li>
	 * <li><strong>403 Forbidden</strong>: Insufficient permissions to perform the operation.</li>
	 * </ul>
	 * 
	 * <strong>Response Format:</strong>
	 * 
	 * <strong>Success Response:</strong>
	 * <ul>
	 * <li>Returns a JSON array containing all instances affected by the operation</li>
	 * <li>Each instance object includes its UUID, type, and relevant metadata</li>
	 * <li>For dryRun=true: Shows what would be affected without making changes</li>
	 * <li>For dryRun=false: Shows what was actually affected by the completed operation.</li>
	 * </ul>
	 * 
	 * <strong>Example Response:</strong>
	 * ``{@code json
	 * [
	 *   {
	 *     "id": "16032d09-3823-444e-a1ff-a67de4f350a8",
	 *     "type": "HostingNode",
	 *     // ... additional instance properties
	 *   },
	 *   {
	 *     "id": "97ab8a6b-6b1b-4868-b8fc-ba48d0439ba9",
	 *     "type": "Hosts",
	 *     // ... additional instance properties
	 *   }
	 * ]
	 * }``
	 * 
	 * @param contextId the UUID of the target context for the add/remove operation
	 * @param type the type name of the instance to add/remove (e.g., "HostingNode", "EService")
	 * @param instanceId the UUID of the instance to add/remove
	 * @param operation the operation to perform: ADD (add to context) or REMOVE (remove from context)
	 * @param dryRun whether to preview the operation without executing it (default: false)
	 * @param forceAddToContext whether to force addition even if instance is not in current context - admin only (default: false)
	 * @return JSON array containing all instances affected by the operation (or that would be affected if dryRun=true)
	 * @throws SchemaViolationException if the operation violates schema constraints
	 * @throws ResourceNotFoundException if the specified instance is not found
	 * @throws ContextNotFoundException if the specified context is not found
	 * @throws ResourceRegistryException for other operation errors or authorization failures
	 */
	@POST
	@Path("/" + SharingPath.CONTEXTS_PATH_PART + "/{" + ContextManager.CONTEXT_UUID_PATH_PARAMETER + "}/"
		+ "{" + TypeManager.TYPE_PATH_PARAMETER + "}" + "/{" + InstancesManager.INSTANCE_UUID_PATH_PARAMETER + "}")
	@Produces(ResourceInitializer.APPLICATION_JSON_CHARSET_UTF_8)
	@Operation(
		summary = "Add or Remove Instance from Context",
		description = """
		Add or remove an instance from a context. The operation can have a cascade effect due to propagation constraints.
		Returns the list of instances affected by the add/remove operation on the target instance identified by UUID path parameter.


		**Request Examples:**
		- POST /sharing/contexts/67062c11-9c3a-4906-870d-7df6a43408b0/HostingNode/16032d09-3823-444e-a1ff-a67de4f350a8?operation=ADD (add HostingNode to context);
		- POST /sharing/contexts/67062c11-9c3a-4906-870d-7df6a43408b0/HostingNode/16032d09-3823-444e-a1ff-a67de4f350a8?operation=REMOVE (remove HostingNode from context);
		- POST /sharing/contexts/67062c11-9c3a-4906-870d-7df6a43408b0/HostingNode/16032d09-3823-444e-a1ff-a67de4f350a8?operation=ADD&dryRun=true (preview what would be affected by adding HostingNode);
		- POST /sharing/contexts/67062c11-9c3a-4906-870d-7df6a43408b0/HostingNode/16032d09-3823-444e-a1ff-a67de4f350a8?operation=ADD&forceAddToContext=true (force add even if instance is not in current context);
		- POST /sharing/contexts/67062c11-9c3a-4906-870d-7df6a43408b0/HostingNode/16032d09-3823-444e-a1ff-a67de4f350a8?operation=ADD&includeMeta=true&includeContexts=true (add with metadata and context information in response).

		Where:
		- 67062c11-9c3a-4906-870d-7df6a43408b0 is the Context UUID;
		- 16032d09-3823-444e-a1ff-a67de4f350a8 is the HostingNode UUID.


		**Authorization Requirements:**
		- **IS-Manager:**
			- Can perform ADD/REMOVE operations on any instance in any context;
			- Can use forceAddToContext=true to add instances from other contexts;
			- Full administrative privileges across all contexts;
			- See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.

		- **Infrastructure-Manager:**
			- Can perform ADD/REMOVE operations on any instance in any context;
			- Can use forceAddToContext=true to add instances from other contexts;
			- Full administrative privileges across all contexts;
			- See complete metadata including sensitive information (createdBy, lastUpdatedBy) when includeMeta=true.

		- **Other Users:**
			- Can perform ADD/REMOVE operations only on instances accessible in their current context;
			- Cannot use forceAddToContext=true (parameter is ignored if provided);
			- Operations are limited to instances they have access to in their authorized context;
			- See filtered metadata with sensitive fields obfuscated when includeMeta=true.


		**Operation Behavior:**
		- **Propagation Constraints and Cascade Effects:**
			- **ADD Operation Cascading:** When an instance is added to a context, related instances may also be automatically added due to propagation constraints;
			- **REMOVE Operation Cascading:** When an instance is removed from a context, related instances may also be automatically removed due to propagation constraints;
			- The response includes all instances that were added/removed as part of the cascade effect;
			- Propagation follows the schema-defined relationships and their constraints.

		- **Schema Validation for ConsistsOf Relations and Facets:**
			- When operations are performed on ConsistsOf relations or Facet instances, the parent Resource is automatically validated against its schema definition;
			- If the Resource is no longer compliant with its schema after the add/remove operation, a SchemaViolationException is thrown;
			- This results in an HTTP 400 Bad Request response with details about the schema violation;
			- **Example scenarios that violate schema constraints:**
				- Removing a required/mandatory Facet or ConsistsOf relation from a Resource;
				- Adding/removing instances that violate cardinality constraints (min/max occurrences) defined in the schema.
			- All these scenarios make the Resource incomplete or invalid according to its schema definition;
			- **Constraint Sources**: Constraints are defined by the specific type definitions that have been installed on the system;
			- Schema validation ensures the integrity and consistency by enforcing the constraints defined in each type's schema during operations.
		"""
	)
	@APIResponse(
		responseCode = "200",
		description = "Operation completed successfully (instances added/removed)",
		content = @Content(mediaType = "application/json")
	)
	@APIResponse(
		responseCode = "400",
		description = "Schema validation failed (SchemaViolationException) - occurs when operation would violate schema constraints"
	)
	@APIResponse(
		responseCode = "404",
		description = "Instance, type, or context not found"
	)
	@APIResponse(
		responseCode = "403",
		description = "Insufficient permissions to perform the operation"
	)
	public String addRemove(
			@PathParam(ContextManager.CONTEXT_UUID_PATH_PARAMETER) String contextId,
			@PathParam(TypeManager.TYPE_PATH_PARAMETER) String type,
			@PathParam(InstancesManager.INSTANCE_UUID_PATH_PARAMETER) String instanceId,
			@QueryParam(SharingPath.OPERATION_QUERY_PARAMETER) SharingOperation operation,
			@QueryParam(SharingPath.DRY_RUN_QUERY_QUERY_PARAMETER) @DefaultValue("false") Boolean dryRun,
			@QueryParam(SharingPath.FORCE_ADD_TO_CONTEXT_QUERY_PARAMETER) @DefaultValue("false") Boolean forceAddToContext)
			throws SchemaViolationException, ResourceNotFoundException, ContextNotFoundException, ResourceRegistryException {
		
		ServerRequestInfo serverRequestInfo = initRequestInfo();
		serverRequestInfo.checkAllIncludeQueryParameters();
		
		StringBuffer calledMethod = new StringBuffer(); 
		if(dryRun==null) {
			dryRun = false;
		}
		
		if(dryRun) {
			calledMethod.append("dryRun");
		}
		
		if(operation == SharingOperation.ADD) {
			logger.info("Requested {} {} with UUID {} to {} with UUID {}", dryRun? "a dry run for adding": "to add", type, instanceId, Context.NAME, contextId);
			calledMethod.append("AddToContext");
		}else {
			logger.info("Requested {} {} with UUID {} from {} with UUID {}", dryRun? "a dry run for removing": "to remove", type, instanceId, Context.NAME, contextId);
			calledMethod.append("RemoveFromContext");
		}
		setAccountingMethod(calledMethod.toString());
		

		ElementManagement<?,?> erManagement = ERManagementUtility.getERManagement(type);
		erManagement.setUUID(UUID.fromString(instanceId));
		erManagement.setDryRun(dryRun);
		UUID contextUUID = UUID.fromString(contextId);
		
		
		if(operation == SharingOperation.ADD) {
			((ERManagement) erManagement).setForceAddToContext(forceAddToContext);
			((ERManagement) erManagement).addToContext(contextUUID);
		}else {
			((ERManagement) erManagement).removeFromContext(contextUUID);
		}
		
		try {
			ObjectMapper objectMapper = new ObjectMapper();
			return serializeAffectedInstaces(objectMapper, erManagement.getAffectedInstances());
		} catch (Exception e) {
			throw new ResourceRegistryException(e);
		}
	}

}
