package org.gcube.keycloak.event;

import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.Map;

import org.gcube.event.publisher.Event;
import org.keycloak.events.admin.AdminEvent;

public class KeycloakEvent extends Event {

    private static final long serialVersionUID = 4072256389444123291L;

    public static final String TYPE = "keycloak";
    public static final String REALM = "realm";
    public static final String USER = "user";
    public static final String CLIENT = "client";
    public static final String RESOURCE = "resource";
    public static final String RESOURCE_TYPE = "resource-type";
    public static final String OPERATION = "operation";

    public static final String ADMIN_NAME = "ADMIN";

    public KeycloakEvent(String name, String sender, String realm) {
        this(name, sender, realm, null);
    }

    public KeycloakEvent(String name, String sender, String realm, Map<String, String> data) {
        super(name, TYPE, sender, data);
        setRealm(realm);
    }

    public static KeycloakEvent newKeycloakEvent(org.keycloak.events.Event event) {
        KeycloakEvent keycloakEvent = new KeycloakEvent(event.getType().name(), event.getIpAddress(),
                event.getRealmId(), event.getDetails());

        // Overriding the timestamp setting it equal to the event time
        keycloakEvent.setTimestamp(convertEventDate(event.getTime()));

        keycloakEvent.setClient(event.getClientId());
        keycloakEvent.setUser(event.getUserId());
        return keycloakEvent;
    }

    public static KeycloakEvent newKeycloakAdminEvent(AdminEvent adminEvent) {
        KeycloakEvent keycloakEvent = new KeycloakEvent(ADMIN_NAME, adminEvent.getResourcePath(),
                adminEvent.getRealmId());

        // Overriding the timestamp setting it equal to the event time
        keycloakEvent.setTimestamp(convertEventDate(adminEvent.getTime()));
        keycloakEvent.setResource(adminEvent.getResourcePath());
        keycloakEvent.setResourceType(adminEvent.getResourceTypeAsString());
        keycloakEvent.setOperation(adminEvent.getOperationType().name());
        return keycloakEvent;
    }

    private static OffsetDateTime convertEventDate(long millis) {
        OrchestratorEventPublisherProvider.logger.infof("Creating offset date time from millis: %l", millis);
        return Instant.ofEpochMilli(millis).atZone(ZoneOffset.systemDefault()).toOffsetDateTime();
    }

    public void setRealm(String realm) {
        set(REALM, realm);
    }

    public String getRealm() {
        return (String) get(REALM);
    }

    public void setUser(String user) {
        set(USER, user);
    }

    public String getUser() {
        return (String) get(USER);
    }

    public void setClient(String client) {
        set(CLIENT, client);
    }

    public String getClient() {
        return (String) get(CLIENT);
    }

    public void setResource(String resource) {
        set(RESOURCE, resource);
    }

    public String getResource() {
        return (String) get(RESOURCE);
    }

    public void setResourceType(String resourceType) {
        set(RESOURCE_TYPE, resourceType);
    }

    public String getResourceType() {
        return (String) get(RESOURCE_TYPE);
    }

    public void setOperation(String operation) {
        set(OPERATION, operation);
    }

    public String getOperation() {
        return (String) get(OPERATION);
    }
}
