/**
 * 
 */
package org.gcube.portlets.user.warmanagementwidget.server.management;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.StringWriter;
import java.util.Collections;
import java.util.zip.GZIPOutputStream;

import org.apache.commons.io.IOUtils;
import org.apache.log4j.Logger;
import org.apache.tools.tar.TarEntry;
import org.apache.tools.tar.TarOutputStream;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.resources.GCUBEService;
import org.gcube.common.core.resources.common.PlatformDescription;
import org.gcube.common.core.resources.service.Software;
import org.gcube.common.core.resources.service.Package.GHNRequirement;
import org.gcube.common.core.resources.service.Package.ScopeLevel;
import org.gcube.common.core.resources.service.Package.GHNRequirement.Category;
import org.gcube.common.core.resources.service.Package.GHNRequirement.OpType;
import org.gcube.common.core.resources.service.Software.Type;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.security.GCUBESecurityManager;
import org.gcube.portlets.user.warmanagementwidget.client.data.WarProfile;
import org.gcube.portlets.user.warmanagementwidget.client.progress.OperationProgress;
import org.gcube.portlets.user.warmanagementwidget.client.progress.OperationState;
import org.gcube.portlets.user.warmanagementwidget.client.upload.progress.CreationStates;
import org.gcube.portlets.user.warmanagementwidget.server.accesslog.AccessLogUtil;
import org.gcube.portlets.user.warmanagementwidget.server.management.SoftwareGatewayRegistrationResult.RegistrationStatus;
import org.gcube.portlets.user.warmanagementwidget.server.management.maven.ConsoleMavenDeployer;
import org.gcube.portlets.user.warmanagementwidget.server.management.maven.MavenCoordinates;
import org.gcube.portlets.user.warmanagementwidget.server.management.maven.MavenDeployer;
import org.gcube.portlets.user.warmanagementwidget.server.util.Util;
import org.gcube.vremanagement.softwaregateway.client.RegisterProfileClient;

/**
 * @author Federico De Faveri defaveri@isti.cnr.it
 *
 */
public class GCubeWarUploader implements Runnable{

	protected static final String SVN_PATH_TXT = "https://svn.research-infrastructures.eu/d4science/gcube/trunk/";

	protected static final String SERVICE_VERSION = "1.0.0";

	protected static final String SERVICE_CLASS = "WebApp";
	
	protected static final String GROUP_ID = "org.gcube.webapp";
	protected static final boolean SNAPSHOT = false; 

	protected Logger logger = Logger.getLogger(GCubeWarUploader.class);

	protected OperationProgress operationProgress;

	protected String user;
	protected String sessionId;
	protected File warFile;
	protected WarProfile profile;
	protected String baseUrl;
	protected GCUBEScope scope;
	protected GCUBESecurityManager securityManager;

	/**
	 * @param operationProgress
	 * @param sessionId
	 * @param warFile
	 * @param profile
	 * @param baseUrl
	 */
	public GCubeWarUploader(OperationProgress operationProgress,
			String user, GCUBEScope scope, GCUBESecurityManager securityManager,
			String sessionId, File warFile, WarProfile profile, String baseUrl) {

		this.operationProgress = operationProgress;
		this.user = user;
		this.sessionId = sessionId;
		this.warFile = warFile;
		this.profile = profile;
		this.baseUrl = baseUrl;
		this.scope = scope;
		this.securityManager = securityManager;
	}

	public GCubeWarUploader() {
	}

	@Override
	public void run() {
		
		operationProgress.setElaboratedLenght(CreationStates.CREATE_STATE_PROFILE_CREATION);
		
		logger.trace("generating Maven Coordinates");
		MavenCoordinates mavenCoordinates = calculateMavenCoordinates(profile);
		logger.trace("mavenCoordinates: "+mavenCoordinates);
		
		logger.trace("generating the XML profile");
		String xmlProfile = null;
		try {
			xmlProfile = generateXMLProfile(profile, mavenCoordinates);
		} catch (Exception e) {
			logger.error("An error occured creating the XML profile", e);
			operationProgress.setFailed("An error occured creating the XML profile", Util.exceptionDetailMessage(e));
			return ;
		}
		
		
		logger.trace("creating SA");
		File saFile;
		try {
			saFile = createServiceArchive(warFile, xmlProfile);
		} catch (Exception e) {
			logger.error("An error occured creating the SA", e);
			operationProgress.setFailed("An error occured creating the Service Archive", Util.exceptionDetailMessage(e));
			return ;
		}
		
		logger.trace("Deploying Jar on Maven");
		operationProgress.setElaboratedLenght(CreationStates.CREATE_STATE_DEPLOYING_WAR);
		
		try {
			MavenDeployer mavenDeployer = new ConsoleMavenDeployer();
			mavenDeployer.deploy(scope, mavenCoordinates, warFile, "war", null, true);
			mavenDeployer.deploy(scope, mavenCoordinates, saFile, "tar.gz", "servicearchive", false);

		} catch (Exception e) {
			logger.error("Maven deploy failed", e);
			operationProgress.setFailed("Maven deploy failed", Util.exceptionDetailMessage(e));
			return;
		}		
		
		logger.trace("storing the Profile in the Software Gateway");
		operationProgress.setElaboratedLenght(CreationStates.CREATE_STATE_STORING_PROFILE);
		
		String storeResult;
		try {
			storeResult = RegisterProfileClient.registerProfile(scope, xmlProfile);
		} catch (Exception e) {
			logger.error("An error occured storing the profile in the Software Gateway", e);
			operationProgress.setFailed("An error occured storing the Software Gateway", Util.exceptionDetailMessage(e));
			return;
		}
		
		logger.trace("parsing the registry operation result: "+storeResult);
		SoftwareGatewayRegistrationResult result;
		try {
			result = parseSoftwareGatewayResult(storeResult);
			logger.trace("result: "+result);
		} catch (Exception e) {
			logger.error("An error occured parsing the SG result", e);
			operationProgress.setFailed("An error occured parsing the SG report result", Util.reportDetailMessage(e, storeResult));
			return;
		}
		
		if (result.getStatus() == RegistrationStatus.ERROR) {
			logger.error("Store failed. Report: "+result.getReport());
			operationProgress.setFailed("Store failed", Util.reportDetailMessage(storeResult));
			return;
		}
		
		operationProgress.setState(OperationState.COMPLETED);
		AccessLogUtil.logWarUpload(user, scope, profile);
	}

	public SoftwareGatewayRegistrationResult parseSoftwareGatewayResult(String result) 
	{
		//TODO: workaround, the SG generate an invalid XML
		if (result.toLowerCase().contains(("<status>warn</status>")) || result.toLowerCase().contains(("<status>success</status>"))) return new SoftwareGatewayRegistrationResult(RegistrationStatus.OK, result);
		return new SoftwareGatewayRegistrationResult(RegistrationStatus.ERROR, result);
	}
	
	public MavenCoordinates calculateMavenCoordinates(WarProfile profile)
	{
		String groupId = GROUP_ID;
		String artifactId = profile.getApplicationName();
		String version = Util.getVersion(profile);
		if (SNAPSHOT) version += "-SNAPSHOT";
		return new MavenCoordinates(groupId, artifactId, version);
	}

	public String generateXMLProfile(WarProfile profile, MavenCoordinates mavenCoordinates) throws Exception
	{

		GCUBEService gcubeService = GHNContext.getImplementation(GCUBEService.class);

		/*	<Description>**DESCRIPTION**</Description>
		<Class>WebApp</Class>
		<Name>**APP CATEGORY**</Name>
		<Version>1.0.0</Version>
		 */
		gcubeService.setServiceName(profile.getCategoryName());
		gcubeService.setDescription(profile.getCategoryDescription());
		gcubeService.setServiceClass(SERVICE_CLASS);
		gcubeService.setVersion(SERVICE_VERSION);

		/*	
		<Packages>
			<Software>
				<Description>**DESCRIPTION**</Description>
				<Name>**APP NAME**</Name>
				<Version>**VERSION**</Version>
		 */
		Software warPackage = new Software();
		warPackage.setName(profile.getApplicationName());
		warPackage.setDescription(profile.getApplicationDescription());

		String version = Util.getVersion(profile);
		if (SNAPSHOT) version += "-SNAPSHOT";
		warPackage.setVersion(version);
		
      /*  <Main>
		        <Description>...</Description>
		        <Name>myapp</Name>
		        <Version>1.0.0</Version>
		        <MavenCoordinates>
		            <groupId>org.gcube.customwars</groupId>
		            <artifactId>myapp</artifactId>
		            <version>1.0.0</version>
		        </MavenCoordinates>
		    </Main>
       */
		
		warPackage.setMavenCoordinates(mavenCoordinates.getGroupId(), mavenCoordinates.getArtifactId(), mavenCoordinates.getVersion());
		
		/*	
		<TargetPlatform>
			<Name>Tomcat</Name>
			<Version>6</Version>
			<MinorVersion>0</MinorVersion>
		</TargetPlatform>
		 */
		PlatformDescription targetPlatform = new PlatformDescription();
		targetPlatform.setName("Tomcat");
		targetPlatform.setVersion((short) 6);
		targetPlatform.setMinorVersion((short) 0);
		warPackage.setTargetPlatform(targetPlatform);

		/*
		 *					<MultiVersion value="true" />
					<Mandatory level="NONE" />
					<Shareable level="NONE" />
					<GHNRequirements>
						<Requirement category="Site" operator="ge" requirement="string" value="java1.6" />
					</GHNRequirements>
					<SpecificData/>
					<Type>webapplication</Type>
		 */

		warPackage.setMultiVersion(true);
		warPackage.setMandatoryLevel(ScopeLevel.NONE);
		warPackage.setSharingLevel(ScopeLevel.NONE);

		GHNRequirement requirement = new GHNRequirement();
		requirement.setCategory(Category.SITE_LOCATION);
		requirement.setOperator(OpType.GE);
		requirement.setRequirement("string");
		requirement.setValue("java1.6");
		warPackage.setGHNRequirements(Collections.singletonList(requirement));
		warPackage.setType(Type.webapplication);

		/*
		<EntryPoint>**APPLICATION ENTRY POINT 1**</EntryPoint>	
		<EntryPoint>**APPLICATION ENTRY POINT N**</EntryPoint>
					<Files>
							<File>**NAME OF THE WAR**</File>
						</Files>
					</Software>
				</Packages>
			</Profile>
		</Resource>*/

		warPackage.getEntrypoints().addAll(profile.getEntryPoints());
		warPackage.getFiles().add(profile.getWarFileName());
		gcubeService.getPackages().add(warPackage);

		StringWriter xml = new StringWriter();
		gcubeService.store(xml);

		logger.trace("generated profile: "+xml.toString());
		return xml.toString();
	}
	
	public File createServiceArchive(File warFile, String xmlProfile) throws Exception
	{
		logger.trace("generating the Service Archive");

		File saFile = File.createTempFile("serviceArchive", "tmp.tar.gz");
		logger.trace("saving SA in "+saFile.getAbsolutePath());

		BufferedOutputStream bos = new BufferedOutputStream(new FileOutputStream(saFile));
		TarOutputStream tarOutput = new TarOutputStream(new GZIPOutputStream(bos));
		tarOutput.setLongFileMode(TarOutputStream.LONGFILE_GNU);

		//profile.xml
		logger.trace("adding profile...");
		TarEntry profileEntry = new TarEntry("profile.xml");
		profileEntry.setSize(xmlProfile.length());
		tarOutput.putNextEntry(profileEntry);
		tarOutput.write(xmlProfile.getBytes());
		tarOutput.closeEntry();

		//war file in application directory
		logger.trace("adding war file...");
		String warDir = profile.getApplicationName()+"/";
		TarEntry warFolderEntry = new TarEntry(warDir);
		tarOutput.putNextEntry(warFolderEntry);
		tarOutput.closeEntry();

		TarEntry warEntry = new TarEntry(warDir+profile.getWarFileName());
		warEntry.setSize(warFile.length());
		tarOutput.putNextEntry(warEntry);
		IOUtils.copy(new FileInputStream(warFile), tarOutput);
		tarOutput.closeEntry();

		//svnpath.txt
		TarEntry svnPathEntry = new TarEntry(warDir+"svnpath.txt");
		svnPathEntry.setSize(SVN_PATH_TXT.length());
		tarOutput.putNextEntry(svnPathEntry);
		tarOutput.write(SVN_PATH_TXT.getBytes());
		tarOutput.closeEntry();

		tarOutput.close();
		logger.trace("SA complete");

		return saFile;
	}
}
