/****************************************************************************
 *  This software is part of the gCube Project.
 *  Site: http://www.gcube-system.org/
 ****************************************************************************
 * The gCube/gCore software is licensed as Free Open Source software
 * conveying to the EUPL (http://ec.europa.eu/idabc/eupl).
 * The software and documentation is provided by its authors/distributors
 * "as is" and no expressed or
 * implied warranty is given for its use, quality or fitness for a
 * particular case.
 ****************************************************************************
 * Filename: GenericResourceTreeManagerForm.java
 ****************************************************************************
 * @author <a href="mailto:assante@isti.cnr.it">Massimiliano Assante</a>
 ***************************************************************************/

package org.gcube.portlets.admin.resourcemanagement.client.forms.genericresources;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import org.gcube.portlets.admin.resourcemanagement.client.remote.ProxyRegistry;
import org.gcube.portlets.admin.resourcemanagement.client.utils.OpCommands;
import org.gcube.resourcemanagement.support.client.views.validators.SelectValidator;
import org.gcube.resourcemanagement.support.shared.plugins.TMPluginFormField;
import org.gcube.resourcemanagement.support.shared.types.Tuple;
import org.gcube.resourcemanagement.support.shared.types.datamodel.AtomicTreeNode;

import com.extjs.gxt.ui.client.Style.Scroll;
import com.extjs.gxt.ui.client.data.ModelData;
import com.extjs.gxt.ui.client.event.ComponentEvent;
import com.extjs.gxt.ui.client.event.SelectionChangedEvent;
import com.extjs.gxt.ui.client.event.SelectionChangedListener;
import com.extjs.gxt.ui.client.store.ListStore;
import com.extjs.gxt.ui.client.widget.Component;
import com.extjs.gxt.ui.client.widget.Dialog;
import com.extjs.gxt.ui.client.widget.MessageBox;
import com.extjs.gxt.ui.client.widget.button.Button;
import com.extjs.gxt.ui.client.widget.form.ComboBox;
import com.extjs.gxt.ui.client.widget.form.ComboBox.TriggerAction;
import com.extjs.gxt.ui.client.widget.form.Field;
import com.extjs.gxt.ui.client.widget.form.FieldSet;
import com.extjs.gxt.ui.client.widget.form.FormButtonBinding;
import com.extjs.gxt.ui.client.widget.form.FormPanel;
import com.extjs.gxt.ui.client.widget.form.TextArea;
import com.extjs.gxt.ui.client.widget.form.TextField;
import com.extjs.gxt.ui.client.widget.layout.FitLayout;
import com.extjs.gxt.ui.client.widget.layout.FlowLayout;
import com.extjs.gxt.ui.client.widget.layout.FormData;
import com.extjs.gxt.ui.client.widget.layout.FormLayout;
import com.google.gwt.user.client.Element;
import com.google.gwt.user.client.rpc.AsyncCallback;
/**
 * 
 * @author Massimiliano Assante, ISTI-CNR
 * @version 1.0 Oct 2012
 *
 */
public class GenericResourceTreeManagerForm extends Dialog {
	private FormData formData = null;
	private FormPanel form = null;
	private FieldSet pluginFieldSet = null;
	private String selectedPlugin = null;
	private HashMap<String, ArrayList<TMPluginFormField>> loadedPlugins = null;
	private TextField<String> resName = null;

	public GenericResourceTreeManagerForm() {
		this.setLayout(new FitLayout());
		this.setHeading("Generic Resource Creation (TM Plugin)");
		this.setModal(true);
		this.setWidth(700);
		this.setHeight(500);
		this.setResizable(false);
		this.getButtonBar().removeAll();
		this.setHideOnButtonClick(true);
	}

	@Override
	protected final void onRender(final Element parent, final int index) {
		super.onRender(parent, index);
		formData = new FormData("-20");
		createForm();
		createPluginForm();
		this.add(form);

		this.initButtons();

		this.recalculate();
		this.doLayout(true);

		this.form.mask("Loading plugins...");
		// Gets the plugins
		ProxyRegistry.getProxyInstance().getGenericResourceTreeMagnagerPlugins(new AsyncCallback<HashMap<String, ArrayList<TMPluginFormField>>>() {
			@Override
			public void onSuccess(HashMap<String, ArrayList<TMPluginFormField>> result) {
				addPluginSelect(result);
				form.unmask();
			}
			@Override
			public void onFailure(final Throwable caught) {
				MessageBox.alert("Create Generic Resource (Tree Manager Plugin)",
						"Received an exception: " + caught.getMessage(), null);
				form.unmask();
			}
		});
	}

	public final void closeDialog() {
		this.hide();
	}

	private void setPluginFields(final ArrayList<TMPluginFormField> fields) {
		if (fields.size() == 0) {
			return;
		}
		Component select = pluginFieldSet.getItem(0);
		pluginFieldSet.removeAll();
		pluginFieldSet.add(select);

		for (TMPluginFormField field : fields) {

			TextField<String> toAdd = new TextField<String>();
			toAdd.setFieldLabel(field.getLabel() + (field.isRequired() ? "*" : ""));
			toAdd.setName(field.getLabel());
			//toAdd.setValidator(new StringValidator(128, !field.isRequired()));
			toAdd.setAllowBlank(!field.isRequired());  // is required
			toAdd.setValue(field.getDefaultValue() + (field.isRepeatable() ? "," + field.getDefaultValue() + "2 (Repeatable, use comma to separate " + field.getLabel() + ")" : ""));
			pluginFieldSet.add(toAdd, formData);
			pluginFieldSet.recalculate();
			this.doLayout(true);

		}
	}

	private void addPluginSelect(final HashMap<String, ArrayList<TMPluginFormField>> plugins) {
		ListStore<ModelData> store = new ListStore<ModelData>();
		for (String pluginName : plugins.keySet()) {
			store.add(new AtomicTreeNode(pluginName));
		}

		ComboBox<ModelData> combo = new ComboBox<ModelData>();
		combo.setFieldLabel("Plugin");
		combo.addSelectionChangedListener(new SelectionChangedListener<ModelData>() {
			@Override
			public void selectionChanged(final SelectionChangedEvent<ModelData> se) {
				if (se != null &&
						se.getSelection() != null
						&& se.getSelection() instanceof ArrayList) {
					@SuppressWarnings("rawtypes")
					String selection = ((ArrayList) se.getSelection()).get(0).toString();
					selectedPlugin = selection;
					setPluginFields(loadedPlugins.get(selection));
				}
			}
		});
		combo.setEmptyText("Select a plugin...");
		combo.setDisplayField("name");
		combo.setStore(store);
		combo.setTypeAhead(true);
		combo.setValidator(new SelectValidator(false));
		combo.setAllowBlank(false);
		combo.setForceSelection(true);
		combo.setTriggerAction(TriggerAction.ALL);
		combo.recalculate();


		pluginFieldSet.recalculate();
		pluginFieldSet.add(combo, formData);
		this.doLayout(true);
		this.loadedPlugins = plugins;
	}

	private void createForm() {
		form = new FormPanel();
		form.setFrame(true);
		form.setAutoWidth(true);

		form.setHeaderVisible(false);
		form.getHeader().setStyleName("x-hide-panel-header");
		form.setLayout(new FlowLayout());

		form.setScrollMode(Scroll.AUTO);

		FieldSet formHead = new FieldSet();
		formHead.setHeading("Basic parameters");
		formHead.setCollapsible(true);

		FormLayout layout = new FormLayout();
		layout.setLabelWidth(90);
		formHead.setLayout(layout);

		TextField<String> resID = new TextField<String>();
		resID.setFieldLabel("Resource ID");
		// resID.setValidator(new StringValidator(64, false));
		resID.setAllowBlank(false);  // is required
		// If in editing mode

		resID.setEmptyText("<generated by Resource Manager>");
		resID.setEnabled(false);
		formHead.add(resID, formData);

		this.resName = new TextField<String>();
		resName.setFieldLabel("Resource Name");
		// resName.setValidator(new StringValidator(64, false));
		resName.setAllowBlank(false);  // is required
		resName.setValue("TMRecord");
		resName.setEnabled(false);
		formHead.add(resName, formData);

		ComboBox<AtomicTreeNode> resType = new ComboBox<AtomicTreeNode>();
		ListStore<AtomicTreeNode> storePlugin = new ListStore<AtomicTreeNode>();
		storePlugin.add(new AtomicTreeNode("ActivationRecord"));
		resType.setStore(storePlugin);
		resType.setFieldLabel("Secondary Type");
		//resType.setEmptyText("Select a subtype...");
		resType.setValue(storePlugin.getAt(0));
		resType.setDisplayField("name");
		resType.setAllowBlank(false);
		resType.setEditable(false);
		resType.setTypeAhead(false);
		resType.setTriggerAction(TriggerAction.ALL);
		resType.setForceSelection(true);
		resType.addSelectionChangedListener(new SelectionChangedListener<AtomicTreeNode>() {
			@Override
			public void selectionChanged(final SelectionChangedEvent<AtomicTreeNode> se) {
				// The usual casting trick for fixing invalid class specification
				// of generic return in models.
				if (se.getSelection().get(0).getName().equals("VirtualCollection")) {
					resName.setFieldLabel("Collection Name");
					resName.setValue(null);
					resName.setEnabled(true);
				} else {
					resName.setFieldLabel("Resource Name");
					resName.setValue("TMRecord");
					resName.setEnabled(false);
				}
			}
		});
		formHead.add(resType, formData);

		TextArea description = new TextArea();
		description.setFieldLabel("Description");
		description.setPreventScrollbars(false);
		//description.setHeight(50);
		formHead.add(description, formData);

		form.add(formHead);
	}

	private void createPluginForm() {
		this.pluginFieldSet = new FieldSet();
		pluginFieldSet.setHeading("Plugin parameters");
		pluginFieldSet.setCollapsible(true);

		FormLayout layout = new FormLayout();
		layout.setLabelWidth(110);
		pluginFieldSet.setLayout(layout);
		form.add(pluginFieldSet);
	}

	public final void initButtons() {
		Button submitBtn = new Button("Submit") {
			@SuppressWarnings("deprecation")
			@Override
			protected void onClick(final ComponentEvent ce) {
				super.onClick(ce);
				if (!form.isValid()) {
					return;
				}
				Tuple<String> values = new Tuple<String>();
				for (Field<?> field  : form.getFields()) {
					values.append(field.getRawValue());
				}

				String resName = values.get(1);
				String resType = values.get(2);
				String description = values.get(3);

				StringBuilder body = new StringBuilder();
				body.append("<ns3:Record xmlns:ns3=\"http://gcube-system.org/namespaces/common\" xmlns:ns2=\"http://gcube-system.org/namespaces/data/tm\">");
				body.append("<createdBy xmlns:ns3=\"http://gcube-system.org/namespaces/common\" xmlns:ns2=\"http://gcube-system.org/namespaces/data/tm\">ResourceManagementPortlet</createdBy>");
				body.append("<parameters xmlns:ns3=\"http://gcube-system.org/namespaces/common\" xmlns:ns2=\"http://gcube-system.org/namespaces/data/tm\">");
				body.append("<broadcast xmlns:ns3=\"http://gcube-system.org/namespaces/common\" xmlns:ns2=\"http://gcube-system.org/namespaces/data/tm\">true</broadcast>");

				if (selectedPlugin == null) {
					return;
				}
				ArrayList<TMPluginFormField> fields = loadedPlugins.get(selectedPlugin);
				body.append("<speciesRequest>");
				Map<String, Object> submitted = new HashMap<String, Object>();
				for (Field<?> field  : form.getFields()) {
					submitted.put(field.getName(), field.getRawValue());
				}
				if (fields != null && fields.size() > 0) {
					for (TMPluginFormField field : fields) {
						try {
							String tag = field.getLabel();
							String value = submitted.get(tag).toString();
							if (value != null && value.trim().length() > 0) {
								if (value.split(",").length > 0) {
									String[] toAdd = value.split(",");
									for (int i = 0; i < toAdd.length; i++) {
										body.append("<" + tag + ">" + toAdd[i] + "</" + tag + ">");
									}
								}
								else
									body.append("<" + tag + ">" + value + "</" + tag + ">");
							}
						} catch (RuntimeException e) {
							// parsing errors will be skipped
						}
					}
				}

				body.append("</speciesRequest>");
				body.append("<plugin xmlns:ns3=\"http://gcube-system.org/namespaces/common\" xmlns:ns2=\"http://gcube-system.org/namespaces/data/tm\">" + selectedPlugin + "</plugin>");
				body.append("</parameters>");
				body.append("</ns3:Record>");


				OpCommands.doCreateGenericResource(null, resName, description, body.toString(), resType);


				closeDialog();
			}
		};
		this.form.addButton(submitBtn);

		this.form.addButton(new Button("Cancel") {
			@Override
			protected void onClick(final ComponentEvent ce) {
				super.onClick(ce);
				closeDialog();
			}
		});

		///form.setButtonAlign(HorizontalAlignment.CENTER);

		FormButtonBinding binding = new FormButtonBinding(form);
		binding.addButton(submitBtn);
	}
}
