package org.gcube.portlets.user.homelibrary.jcr.workspace.folder.items;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import javax.jcr.Binary;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.commons.io.IOUtils;
import org.apache.jackrabbit.util.Text;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;
import org.gcube.portlets.user.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.portlets.user.homelibrary.home.workspace.Properties;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.portlets.user.homelibrary.home.workspace.WorkspaceItemType;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.FolderItem;
import org.gcube.portlets.user.homelibrary.home.workspace.folder.FolderItemType;
import org.gcube.portlets.user.homelibrary.jcr.repository.JCRRepository;
import org.gcube.portlets.user.homelibrary.jcr.repository.external.GCUBEStorage;
import org.gcube.portlets.user.homelibrary.jcr.workspace.JCRProperties;
import org.gcube.portlets.user.homelibrary.jcr.workspace.JCRWorkspace;
import org.gcube.portlets.user.homelibrary.jcr.workspace.JCRWorkspaceItem;

public abstract class JCRWorkspaceFolderItem extends JCRWorkspaceItem implements
		FolderItem {
	
	
	protected static final String WORKFLOW_ID 		= "hl:workflowId";
	protected static final String WORKFLOW_STATUS 	= "hl:workflowStatus";
	protected static final String WORKFLOW_DATA 	= "hl:workflowData"; 

	
	protected static final String CONTENT			= "jcr:content";
	protected static final String FOLDER_ITEM_TYPE 	= "hl:workspaceItemType";
	
	public JCRWorkspaceFolderItem(JCRWorkspace workspace, Node node) throws RepositoryException {
		super(workspace, node);
	}
	
	public JCRWorkspaceFolderItem(JCRWorkspace workspace, Node node,
			String name, String description) throws RepositoryException {
		super(workspace, node, name, description);
		
		node.setProperty(WORKFLOW_ID, "");
		node.setProperty(WORKFLOW_STATUS, "");
		node.setProperty(WORKFLOW_DATA, "");
		
	}
	
	@Override
	public void save(Node node) throws RepositoryException {
		super.save(node);

		Node contentNode = node.getNode(CONTENT);
		saveContent(contentNode);
	}

	abstract public void saveContent(Node node) throws RepositoryException;
	
	@Override
	public WorkspaceItemType getType() {
		
		return WorkspaceItemType.FOLDER_ITEM;
	}
	
	@Override
	public abstract FolderItemType getFolderItemType();
	
	@Override
	public abstract long getLength() throws InternalErrorException;
	
	@Override
	public abstract String getMimeType() throws InternalErrorException;
	
	
	@Override
	public  List<? extends WorkspaceItem> getChildren() throws InternalErrorException {
		return new ArrayList<WorkspaceItem>();
	}
	
	
	@Override
	public void removeChild(WorkspaceItem child) {
		return;
	}

	@Override
	public String getWorkflowId() throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return node.getProperty(WORKFLOW_ID).getString();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void setWorkflowId(String id) throws InternalErrorException {	
		setWorkflowProperty(WORKFLOW_ID, id);
	}
		
	@Override
	public String getWorkflowStatus() throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return node.getProperty(WORKFLOW_STATUS).getString();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void setWorkflowStatus(String status) throws InternalErrorException {
		setWorkflowProperty(WORKFLOW_STATUS, status);
	}

	@Override
	public String getWorkflowData() throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return node.getProperty(WORKFLOW_DATA).getString();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}

	@Override
	public void setWorkflowData(String data) throws InternalErrorException {
		setWorkflowProperty(WORKFLOW_DATA, data);
	}
	
	private final void setWorkflowProperty(String property, String value) throws InternalErrorException {
		
		Session session = JCRRepository.getSession();
		try {
			session.getNodeByIdentifier(identifier).setProperty(property, value);
			session.save();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}
	
	@Override
	public Properties getProperties() throws InternalErrorException {
		Session session = JCRRepository.getSession();
		try {
			Node node = session.getNodeByIdentifier(identifier);
			return new JCRProperties(node);
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} finally {
			session.logout();
		}
	}
	
	
	public void removeRemoteContent(Node node) throws RepositoryException, RemoteBackendException {
		try {
			Node contentNode = node.getNode(CONTENT);
			if (contentNode.hasProperty(JCRFile.REMOTE_STORAGE_PATH)) {	

				String remotePath = contentNode.getProperty(JCRFile.REMOTE_STORAGE_PATH).getString();
				GCUBEStorage.removeRemoteFile(remotePath);
			}	
		} catch (PathNotFoundException e) {
			logger.error("Content node "+ CONTENT + " not found",e);
		} 
	}
	
	public void copyRemoteContent(Node node) throws RepositoryException, RemoteBackendException {

		try {
			Node contentNode = node.getNode(CONTENT);
			if (contentNode.hasProperty(JCRFile.REMOTE_STORAGE_PATH)) {	
	
				String remotePath = contentNode.getProperty(JCRFile.REMOTE_STORAGE_PATH).getString();
				InputStream is = GCUBEStorage.getRemoteFile(remotePath);
				
				
				remotePath = Text.getRelativeParent(contentNode.getPath(), 2)
						+ UUID.randomUUID().toString();
				String url = GCUBEStorage.putStream(is, remotePath);
				contentNode.setProperty(JCRFile.REMOTE_STORAGE_PATH, remotePath);
				
				// Store url as byte stream in jcr:data binary property
				ByteArrayInputStream  binaryUrl = new ByteArrayInputStream(url.getBytes());
				Binary binary = contentNode.getSession().getValueFactory().createBinary(binaryUrl);
				contentNode.setProperty(JCRFile.DATA, binary);
			}	
		} catch (PathNotFoundException e) {
			logger.error("Content node "+ CONTENT + " not found",e);
		} 

	}
	
}
