package org.gcube.vremanagement.executor.stubs;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.axis.message.addressing.AttributedURI;
import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.contexts.GCUBERemotePortTypeContext;
import org.gcube.common.core.faults.GCUBEException;
import org.gcube.common.core.faults.GCUBEUnrecoverableException;
import org.gcube.common.core.informationsystem.client.RPDocument;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.security.GCUBESecurityManager;
import org.gcube.vremanagement.executor.stubs.service.EngineServiceAddressingLocator;

/**
 * A high-level call for launching a task in a given scope.
 * <p>
 * The call can work with known Executor instances.
 * Alternatively, it may do its best to find one that hosts the task. 
 * It may be used repeatedly and it will remember the Executor instance previously used. 
 *  
 * @author Fabio Simeoni (University of Strathclyde)
 *
 */
public class ExecutorCall extends BaseCall {
	
	/**
	 * Creates an instance to launch a given task in a scope and, optionally, using a security manager.
	 * @param taskName the name of the task.
	 * @param scope the scope.
	 * @param securityManager (optional) the security manager.
	 * @throws Exception if the instance could not be created.
	 */
	public ExecutorCall(String taskName, GCUBEScope scope, GCUBESecurityManager ... securityManager) throws Exception {super(taskName,scope,securityManager);}
	 
	/**{@inheritDoc}*/ 
	protected String getPortTypeName() {return Utils.ENGINE_PT_NAME;}

	/**
	 * Convenience method to set the target endpoint from host name and port.  
	 * @param hostname the host name.
	 * @param port the port.
	 * @throws Exception if a reference to the endpoint could not be built from its hostname and port.
	 */
	public void setEndpoint(String hostname,String port) throws Exception {
		setEndpointReference(new EndpointReferenceType(new AttributedURI("http://"+hostname+":"+port+"/wsrf/services/"+getPortTypeName())));
	}
	
	/**
	 * Launches a task and returns a call to it.
	 * @return the task call.
	 * @throws Exception if the call failed due to a local error.
	 * @throws GCUBEException if call failed due to remote error.
	 */
	public TaskCall launch() throws Exception, GCUBEException {
		return launch(null);
	}
	
	/**
	 * Launches a task with inputs and returns a call to it.
	 * @param inputs the inputs.
	 * @return the call.
	 * @throws Exception if the call failed due to a local error.
	 * @throws GCUBEException if call failed due to remote error.
	 */
	public synchronized TaskCall launch(final Map<String,Object> inputs) throws Exception, GCUBEException{
		
		final ResultHolder<EndpointReferenceType> result = new ResultHolder<EndpointReferenceType>();		
		new WSCallHandler() {
			public void interact(EndpointReferenceType epr) throws Exception {	
				try {
				EnginePortType engine = new EngineServiceAddressingLocator().getEnginePortTypePort(epr);
				engine = GCUBERemotePortTypeContext.getProxy(engine, getScopeManager(), getSecurityManager());
				result.value=engine.launch(new LaunchParameters(Utils.extern(inputs),taskName));
				}
				catch(Exception e) {
					if (ExecutorCall.this.getEndpointReference()==null) throw e; //query mode
					else throw new GCUBEUnrecoverableException(e); //known-epr mode, no best-effort
				}
			}
			/**{@inheritDoc}*/ //@TODO remove at next gcore release
			protected List<RPDocument> findWSResources() throws Exception {
				List<RPDocument> rpds = new ArrayList<RPDocument>();
				//filter by port-type name in case there are multiple ws-resource types per service
				for (RPDocument d : super.findWSResources())
					if (d.getEndpoint().getAddress().getPath().endsWith(ExecutorCall.this.getPortTypeName())) rpds.add(d);
				return rpds;
			}
		}.run();
		
		//create a task call configured for the launched task.
		TaskCall taskCall = new TaskCall(taskName, getScopeManager().getScope(),getSecurityManager());
		taskCall.setEndpointReference(result.value);
		return taskCall;
	}


}
