package org.gcube.vremanagement.resourcemanager.impl.resources;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.gcube.common.core.resources.GCUBEService;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.vremanagement.deployer.stubs.common.PackageInfo;
import org.gcube.vremanagement.resourcemanager.impl.operators.Operator.ACTION;
import org.gcube.vremanagement.resourcemanager.impl.resources.software.Dependency;
import org.gcube.vremanagement.resourcemanager.impl.resources.software.GCUBEPackage;
import org.gcube.vremanagement.resourcemanager.impl.resources.software.SoftwareGatewayRequest;
import org.gcube.vremanagement.resourcemanager.impl.state.VirtualNode;
import org.gcube.vremanagement.resourcemanager.impl.state.VirtualNode.NoGHNFoundException;

/**
 * 
 * Models any kind of scoped deployed software
 *
 * @author Manuele Simi (ISTI-CNR)
 *
 */
public class ScopedDeployedSoftware extends ScopedResource {

	public static final String TYPE = GCUBEService.TYPE;			
		
	/** resolved dependencies */
	private List<Dependency> resolved = new ArrayList<Dependency>();
	
	/** missing dependency */
	private List<Dependency> missing = new ArrayList<Dependency>();
	
	/** the packages deployed for this service*/	
	private Map <String, List<Dependency>> node2packages = new HashMap<String, List<Dependency>>();
	
	/** the packages missing for this service*/	
	private Map <String, List<Dependency>> node2missing = new HashMap<String, List<Dependency>>();
		
	private GCUBEPackage sourcePackage;
	
	/** maps node with the scope used when deploying there */
	//private Map<String, String> node2scope = new HashMap<String, String>();//for backward compatibility
	
	/** maps RI id with the ghn name (where the instance is deployed on)*/
	//private Map<String, String> ri2ghn = new HashMap<String, String>();
	
	//the RI generated by the software, if any
	private String RIID;
	
	//the node on which the software is deployed
	private String virtualNodeName;
		
	private String lastActivityOnGHN;
		
	private transient VirtualNode targetVirtualNode;
 	
	protected ScopedDeployedSoftware(String id, GCUBEScope scope) {
		super(id, TYPE, scope);
	}

	public void setSourcePackage(final GCUBEPackage sourcePackage) {
		this.sourcePackage = sourcePackage; 
	}

	public GCUBEPackage getSourcePackage() {
		return this.sourcePackage;
	}
	
	/**
	 * Gets the list of resolved dependencies when the deployment was performed 
	 * on a specific node
	 * 
	 * @param ghnName the ID of the node
	 * @return the resolved dependencies
	 */
	public List<Dependency> getResolvedDependencies(String ghnName) {
		if (node2packages.get(ghnName) == null)
			return Collections.emptyList();
		return Collections.unmodifiableList(node2packages.get(ghnName));
	}
	
	/** 
	 * Gets the list of resolved dependencies to be used in the next deployment
	 * 
	 * @return the resolved dependencies
	 */
	public List<Dependency> getLastResolvedDependencies() {		
		return Collections.unmodifiableList(this.resolved);
	}
	
	/**
	 * Sets the list of resolved dependencies to be used in the next deployment
	 * 
	 * @param dependencies the resolved dependencies
	 */
	public void setResolvedDependencies(List<Dependency> dependencies) {		
		this.resolved = dependencies;
	}

	/**
	 * Gets the list of missing dependencies when the deployment was performed 
	 * on a specific node
	 * 
	 * @param ghnName the name of the node
	 * @return the missing dependencies
	 */
	public List<Dependency> getMissingDependencies(String ghnName) {
		if (node2missing.get(ghnName) == null)
			return Collections.emptyList();
		return Collections.unmodifiableList(node2missing.get(ghnName));
	}
	
	/**
	 * Gets the list of missing dependencies for the last (failed) deployment
	 * 
	 * @return the missing dependencies
	 */
	public List<Dependency> getLastMissingDependencies() {
		return this.missing;
	}

	/**
	 * Sets the list of missing dependencies for the next deployment 
	 * 
	 * @param dependencies the missign dependencies
	 */
	public void setMissingDependencies(List<Dependency> dependencies) {
		///this.getLogger().trace("Setting missing deps " + dependencies.size());
		missing= dependencies;
	}
	
	@Override
	protected void find()  throws Exception {
		//looks for the service and its deps in the Software Repository
		SoftwareGatewayRequest request = new SoftwareGatewayRequest();
		request.addSoftware(this);
		this.setErrorMessage(""); //empty any previous message
		try {
			request.send(GCUBEScope.getScope(this.scope));
			if (this.resolved.size() == 0) {
				this.success = false;
				this.setErrorMessage("Invalid list of dependecies retrieved from the Software Repository");
			} else 	if (this.missing.size() > 0) {//ops, some deps are missing!				
				this.success = false;
				this.setErrorMessage("The service cannot be deployed due to one or more missing dependencies: " + this.missing);				
			} else {
				this.success = true;
			}
		} catch (Exception e) {
			this.success= false;
			this.setErrorMessage("Unable to resolve the list of dependencies for this service " + e.getMessage());
			getLogger().error("Unable to resolve the list of deps for " + this.sourcePackage, e);
			throw new Exception("Unable to resolve the list of dependencies for " + this.sourcePackage, e);			
		}
	}


	@Override
	protected void addToScope() throws ResourceNotFound, Exception {
		logger.trace("This service belongs to " + this.scope);
		if (!this.isSuccess()) {
			this.noHopeForMe(this.getErrorMessage(),new Exception());
		}		
		if (this.targetVirtualNode == null) {
			this.noHopeForMe("Unable to find a suitable target Node where to deploy the service",new Exception());
		}
		this.action = ACTION.ADD;
		this.setErrorMessage(""); //empty any previous message
		//deploy the service
		try {
			this.targetVirtualNode.setWorkingScope(GCUBEScope.getScope(this.scope));
			this.targetVirtualNode.deploy();
			//store packages for future undeployment and reporting purposes
			List<Dependency> resolvedPackages = new ArrayList<Dependency>();				
			for (Dependency dep : this.resolved)
				resolvedPackages.add(dep); 				
			//store missing packages for reporting purposes
			List<Dependency> missingPackages = new ArrayList<Dependency>();
			for (Dependency dep : this.missing)
				missingPackages.add(dep); 								
		} catch (Exception e) {
			logger.error("Failed to deploy on " + this.targetVirtualNode.getName(), e);
			this.noHopeForMe("Unable to contact the target gHN ("+ this.targetVirtualNode.getName() +") where to deploy the service",e);
		}		
	}
	

	@Override
	protected void removeFromScope() throws ResourceNotFound, Exception {
		logger.trace("Removing the software from " + this.virtualNodeName);
		if (!this.isSuccess()) {
			//TODO: could we undeploy static packages here?
			this.noHopeForMe(this.getErrorMessage(),new Exception());
		}
		this.action = ACTION.REMOVE;
		try {
			this.targetVirtualNode.setWorkingScope(GCUBEScope.getScope(this.scope));
			this.targetVirtualNode.undeploy();
		} catch (Exception e) {
			this.noHopeForMe("Unable to find the gHN with id " + this.lastActivityOnGHN + 
			" to undeploy the instance of " + this.sourcePackage,  new VirtualNode.NoGHNFoundException(""));	
		}
	}

	
	/**
	 * Schedules the undeployment of an instance of this service from the node
	 * 
	 * @param node the node from where to undeploy the service
	 * @throws NoGHNFoundException 
	 * @throws Exception if the operation fails
	 */
	public void scheduleUndeploy(VirtualNode node) throws NoGHNFoundException {
		getLogger().info("Scheduling undeployment of service "+ this.sourcePackage+ " from GHN " + this.lastActivityOnGHN);
		if (this.virtualNodeName.equals(node.getName())) {
			this.targetVirtualNode  = node;
			this.removePackagesFromTargetNode();		
			this.lastActivityOnGHN = node.getID();
		} else
			 throw new VirtualNode.NoGHNFoundException("Invalid Virtual Node for undeployment: " + node.getName());
	}
	
	
	/**
	 * Sets the GHN where the service will be deployed
	 * 
	 * @param node the target GHN
	 */
	public void scheduleDeploy(VirtualNode node) {
		getLogger().info("Using GHN " + node.getID() + " for " + this);	
		//update the ID with the node information
		this.id +="-NODE-" + node.getName();
		this.targetVirtualNode  = node;
		this.virtualNodeName = node.getName();
		this.lastActivityOnGHN  = node.getID();
		sourcePackage.setGHNName(this.getTargetNodeName());
		this.addPackagesToTargetNode();
	}

	/**
	 * Gets the ID of the target gHN
	 * @return the ID of the target gHN
	 */
	public String getTargetNodeID() {
		return lastActivityOnGHN; //this.virtualNodes.size() > 0 ? this.virtualNodes.get(0).getID() : "";
	
	}
	
	/**
	 * Sets the RI generated by this software
	 * @param ri the RI identifier
	 */
	public void setRI(String ri) {
		this.RIID = ri;
	}
	
	/**
	 * Gets the RI generated by this software or null
	 */
	public String getRI() {
		return this.RIID;
	}

	/**
	 * Sets the callback ID from now on for all the service's operations
	 * @param id the callback ID
	 */
	public void setCallbackID(String id) throws VirtualNode.NoGHNFoundException {
		if (this.targetVirtualNode == null)
			throw new VirtualNode.NoGHNFoundException("No virtual node has been set for this software");
		this.targetVirtualNode.setCallbackID(id);
	}
	
	/**
	 * Gets the name of {@link VirtualNode} where the service is actually deployed within this scope
	 * @return the  name of the {@link VirtualNode}
	 */
	public String getTargetNodeName() {		
		return this.virtualNodeName;
	}
	
	/**
	 * Number of times the resource is searched in the IS before to declare it lost
	 * @return the number of attempts to do 
	 */
	@Override
	protected int getMaxFindAttempts(){
		return 1;
	}	
	
	private void addPackagesToTargetNode() {
		Set<PackageInfo> packages = new HashSet<PackageInfo>();
		//prepare the input list of packagesToAdd for that GHN
		List<Dependency> deps = this.resolved;
		for (int i = 0; i < deps.size(); i++) {			
			PackageInfo p = new PackageInfo();
			p.setServiceName(deps.get(i).getService().getName());
			p.setServiceClass(deps.get(i).getService().getClazz());
			p.setServiceVersion(deps.get(i).getService().getVersion());
			p.setVersion(deps.get(i).getVersion());			
			p.setName(deps.get(i).getName()); //packageName
			getLogger().trace("Adding Package to target Node: " + deps.get(i));
			packages.add(p);
		}	
		this.targetVirtualNode.setPackagesToAdd(packages);
	}

	private void removePackagesFromTargetNode() {
		Set<PackageInfo> packages = new HashSet<PackageInfo>();
		//prepare the input list of packagesToRemove for that GHN
		List<Dependency> deps = this.resolved;
		for (int i = 0; i < deps.size(); i++) {			
			PackageInfo p = new PackageInfo();
			p.setServiceName(deps.get(i).getService().getName());
			p.setServiceClass(deps.get(i).getService().getClazz());
			p.setServiceVersion(deps.get(i).getService().getVersion());
			p.setVersion(deps.get(i).getVersion());			
			p.setName(deps.get(i).getName()); //packageName
			getLogger().trace("Remove Package from target Node: " + deps.get(i));
			packages.add(p);
		}	
		this.targetVirtualNode.setPackagesToRemove(packages);
	}
}
