package org.gcube.informationsystem.publisher;

import java.io.StringWriter;
import java.net.URI;
import java.util.List;

import javax.xml.ws.soap.SOAPFaultException;

import org.gcube.common.clients.stubs.jaxws.JAXWSUtils;
import org.gcube.common.resources.gcore.GCoreEndpoint;
import org.gcube.common.resources.gcore.Resource;
import org.gcube.common.resources.gcore.Resource.Type;
import org.gcube.common.resources.gcore.Resources;
import org.gcube.informationsystem.publisher.stubs.registry.RegistryConstants;
import org.gcube.informationsystem.publisher.stubs.registry.RegistryStub;
import org.gcube.informationsystem.publisher.stubs.registry.faults.InvalidResourceException;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.api.ResultParser;
import org.gcube.resources.discovery.client.impl.DelegateClient;
import org.gcube.resources.discovery.client.queries.api.SimpleQuery;
import org.gcube.resources.discovery.icclient.ICClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static org.gcube.common.clients.stubs.jaxws.StubFactory.*;
import static org.gcube.resources.discovery.icclient.ICFactory.*;

public class RegistryPublisherImpl implements RegistryPublisher {
	
	private static final Logger log = LoggerFactory.getLogger(RegistryPublisher.class);

	protected RegistryPublisherImpl(){};
	
	public String create(Resource resource) {
		try {
			Resources.validate(resource);
		}  catch (Exception e) {
			log.error("the resource is not valid", e);
			throw new InvalidResourceException(e);
		}
		
		try{
			getStubs().create(toXml(resource), resource.type().toString());
		}catch(SOAPFaultException e) {
			throw new RuntimeException(JAXWSUtils.remoteCause(e));
		}
		
		return resource.id();
	}

	
	public void update(Resource resource) {
		try {
			Resources.validate(resource);
		}  catch (Exception e) {
			log.error("the resource is not valid", e);
			throw new InvalidResourceException(e);
		}
		try{
			getStubs().update(resource.id(),toXml(resource), resource.type().toString());
		}catch(SOAPFaultException e) {
			throw new RuntimeException(JAXWSUtils.remoteCause(e));
		}
	}

	public void remove(String id, Type type){
		getStubs().remove(id, type.toString());
	}

	
	private List<URI> getEndPoints(){
		SimpleQuery query = queryFor(GCoreEndpoint.class);
		
		ResultParser<URI> uriParser =new ResultParser<URI>() {

			@Override
			public URI parse(String result) throws Exception {
				return new URI(result.replaceAll("\n", ""));
			}
			
		};
		
		DiscoveryClient<URI> client = new DelegateClient<URI>(uriParser, new ICClient());
		query.addCondition("$resource/Profile/ServiceClass/text() eq '"+RegistryConstants.service_class+"'")
			.addCondition("$resource/Profile/ServiceName/text() eq '"+RegistryConstants.service_name+"'")
			.setResult("$resource/Profile/AccessPoint/RunningInstanceInterfaces/Endpoint[string(@EntryName) eq '"+RegistryConstants.service_entrypoint+"']/string()");
		
		log.trace("query is {}", query.expression());
		
		List<URI> services = client.submit(query);
		log.trace("endpoints found {} ",services);
		return services;
	}
	
	private RegistryStub getStubs(){
		//use another method to cache epr
		URI endpoint = getEndPoints().get(0);
		log.trace("endpoint found is {}",endpoint);
		return stubFor(RegistryConstants.registry).at(endpoint);
	}
	
	private String toXml(Resource resource){
		StringWriter writer = new StringWriter();
		Resources.marshal(resource, writer);
		return writer.toString();
	}
	
}
