/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.snapshots;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import org.apache.lucene.util.CollectionUtil;
import org.elasticsearch.ExceptionsHelper;
import org.elasticsearch.action.search.ShardSearchFailure;
import org.elasticsearch.action.support.IndicesOptions;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterService;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateUpdateTask;
import org.elasticsearch.cluster.SnapshotsInProgress;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.IndexNameExpressionResolver;
import org.elasticsearch.cluster.metadata.MetaData;
import org.elasticsearch.cluster.metadata.RepositoriesMetaData;
import org.elasticsearch.cluster.metadata.SnapshotId;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodes;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import org.elasticsearch.cluster.routing.RoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.index.snapshots.IndexShardRepository;
import org.elasticsearch.index.snapshots.IndexShardSnapshotStatus;
import org.elasticsearch.repositories.RepositoriesService;
import org.elasticsearch.repositories.Repository;
import org.elasticsearch.repositories.RepositoryMissingException;
import org.elasticsearch.search.SearchShardTarget;
import org.elasticsearch.snapshots.ConcurrentSnapshotExecutionException;
import org.elasticsearch.snapshots.InvalidSnapshotNameException;
import org.elasticsearch.snapshots.Snapshot;
import org.elasticsearch.snapshots.SnapshotException;
import org.elasticsearch.snapshots.SnapshotInfo;
import org.elasticsearch.snapshots.SnapshotShardFailure;
import org.elasticsearch.threadpool.ThreadPool;

public class SnapshotsService
extends AbstractLifecycleComponent<SnapshotsService>
implements ClusterStateListener {
    private final ClusterService clusterService;
    private final IndexNameExpressionResolver indexNameExpressionResolver;
    private final RepositoriesService repositoriesService;
    private final ThreadPool threadPool;
    private final CopyOnWriteArrayList<SnapshotCompletionListener> snapshotCompletionListeners = new CopyOnWriteArrayList();

    @Inject
    public SnapshotsService(Settings settings, ClusterService clusterService, IndexNameExpressionResolver indexNameExpressionResolver, RepositoriesService repositoriesService, ThreadPool threadPool) {
        super(settings);
        this.clusterService = clusterService;
        this.indexNameExpressionResolver = indexNameExpressionResolver;
        this.repositoriesService = repositoriesService;
        this.threadPool = threadPool;
        if (DiscoveryNode.masterNode(settings)) {
            clusterService.addLast(this);
        }
    }

    public Snapshot snapshot(SnapshotId snapshotId) {
        SnapshotsService.validate(snapshotId);
        List<SnapshotsInProgress.Entry> entries = this.currentSnapshots(snapshotId.getRepository(), new String[]{snapshotId.getSnapshot()});
        if (!entries.isEmpty()) {
            return this.inProgressSnapshot(entries.iterator().next());
        }
        return this.repositoriesService.repository(snapshotId.getRepository()).readSnapshot(snapshotId);
    }

    public List<Snapshot> snapshots(String repositoryName, boolean ignoreUnavailable) {
        HashSet snapshotSet = Sets.newHashSet();
        List<SnapshotsInProgress.Entry> entries = this.currentSnapshots(repositoryName, null);
        for (SnapshotsInProgress.Entry entry : entries) {
            snapshotSet.add(this.inProgressSnapshot(entry));
        }
        Repository repository = this.repositoriesService.repository(repositoryName);
        List<SnapshotId> snapshotIds = repository.snapshots();
        for (SnapshotId snapshotId : snapshotIds) {
            try {
                snapshotSet.add(repository.readSnapshot(snapshotId));
            }
            catch (Exception ex) {
                if (ignoreUnavailable) {
                    this.logger.warn("failed to get snapshot [{}]", ex, snapshotId);
                    continue;
                }
                throw new SnapshotException(snapshotId, "Snapshot could not be read", ex);
            }
        }
        ArrayList snapshotList = new ArrayList(snapshotSet);
        CollectionUtil.timSort(snapshotList);
        return Collections.unmodifiableList(snapshotList);
    }

    public List<Snapshot> currentSnapshots(String repositoryName) {
        ArrayList<Snapshot> snapshotList = new ArrayList<Snapshot>();
        List<SnapshotsInProgress.Entry> entries = this.currentSnapshots(repositoryName, null);
        for (SnapshotsInProgress.Entry entry : entries) {
            snapshotList.add(this.inProgressSnapshot(entry));
        }
        CollectionUtil.timSort(snapshotList);
        return Collections.unmodifiableList(snapshotList);
    }

    public void createSnapshot(final SnapshotRequest request, final CreateSnapshotListener listener) {
        final SnapshotId snapshotId = new SnapshotId(request.repository(), request.name());
        SnapshotsService.validate(snapshotId);
        this.clusterService.submitStateUpdateTask(request.cause(), new ClusterStateUpdateTask(){
            private SnapshotsInProgress.Entry newSnapshot = null;

            @Override
            public ClusterState execute(ClusterState currentState) {
                SnapshotsService.this.validate(request, currentState);
                SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
                if (snapshots != null && !snapshots.entries().isEmpty()) {
                    throw new ConcurrentSnapshotExecutionException(snapshotId, "a snapshot is already running");
                }
                List<String> indices = Arrays.asList(SnapshotsService.this.indexNameExpressionResolver.concreteIndices(currentState, request.indicesOptions(), request.indices()));
                SnapshotsService.this.logger.trace("[{}][{}] creating snapshot for indices [{}]", request.repository(), request.name(), indices);
                this.newSnapshot = new SnapshotsInProgress.Entry(snapshotId, request.includeGlobalState(), SnapshotsInProgress.State.INIT, indices, System.currentTimeMillis(), null);
                snapshots = new SnapshotsInProgress(this.newSnapshot);
                return ClusterState.builder(currentState).putCustom("snapshots", snapshots).build();
            }

            @Override
            public void onFailure(String source, Throwable t) {
                SnapshotsService.this.logger.warn("[{}][{}] failed to create snapshot", t, request.repository(), request.name());
                this.newSnapshot = null;
                listener.onFailure(t);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, final ClusterState newState) {
                if (this.newSnapshot != null) {
                    SnapshotsService.this.threadPool.executor("snapshot").execute(new Runnable(){

                        @Override
                        public void run() {
                            SnapshotsService.this.beginSnapshot(newState, newSnapshot, request.partial, listener);
                        }
                    });
                }
            }

            @Override
            public TimeValue timeout() {
                return request.masterNodeTimeout();
            }
        });
    }

    private void validate(SnapshotRequest request, ClusterState state) {
        RepositoriesMetaData repositoriesMetaData = (RepositoriesMetaData)state.getMetaData().custom("repositories");
        if (repositoriesMetaData == null || repositoriesMetaData.repository(request.repository()) == null) {
            throw new RepositoryMissingException(request.repository());
        }
        SnapshotsService.validate(new SnapshotId(request.repository(), request.name()));
    }

    private static void validate(SnapshotId snapshotId) {
        String name = snapshotId.getSnapshot();
        if (!Strings.hasLength(name)) {
            throw new InvalidSnapshotNameException(snapshotId, "cannot be empty");
        }
        if (name.contains(" ")) {
            throw new InvalidSnapshotNameException(snapshotId, "must not contain whitespace");
        }
        if (name.contains(",")) {
            throw new InvalidSnapshotNameException(snapshotId, "must not contain ','");
        }
        if (name.contains("#")) {
            throw new InvalidSnapshotNameException(snapshotId, "must not contain '#'");
        }
        if (name.charAt(0) == '_') {
            throw new InvalidSnapshotNameException(snapshotId, "must not start with '_'");
        }
        if (!name.toLowerCase(Locale.ROOT).equals(name)) {
            throw new InvalidSnapshotNameException(snapshotId, "must be lowercase");
        }
        if (!Strings.validFileName(name)) {
            throw new InvalidSnapshotNameException(snapshotId, "must not contain the following characters " + Strings.INVALID_FILENAME_CHARS);
        }
    }

    private void beginSnapshot(ClusterState clusterState, final SnapshotsInProgress.Entry snapshot, final boolean partial, final CreateSnapshotListener userCreateSnapshotListener) {
        boolean snapshotCreated = false;
        try {
            Repository repository = this.repositoriesService.repository(snapshot.snapshotId().getRepository());
            MetaData metaData = clusterState.metaData();
            if (!snapshot.includeGlobalState()) {
                MetaData.Builder builder = MetaData.builder();
                for (String index : snapshot.indices()) {
                    builder.put(metaData.index(index), false);
                }
                metaData = builder.build();
            }
            repository.initializeSnapshot(snapshot.snapshotId(), snapshot.indices(), metaData);
            snapshotCreated = true;
            if (snapshot.indices().isEmpty()) {
                userCreateSnapshotListener.onResponse();
                this.endSnapshot(snapshot);
                return;
            }
            this.clusterService.submitStateUpdateTask("update_snapshot [" + snapshot.snapshotId().getSnapshot() + "]", new ClusterStateUpdateTask(){
                boolean accepted = false;
                SnapshotsInProgress.Entry updatedSnapshot;
                String failure = null;

                @Override
                public ClusterState execute(ClusterState currentState) {
                    SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
                    ArrayList<SnapshotsInProgress.Entry> entries = new ArrayList<SnapshotsInProgress.Entry>();
                    for (SnapshotsInProgress.Entry entry : snapshots.entries()) {
                        if (entry.snapshotId().equals(snapshot.snapshotId())) {
                            ImmutableMap shards = SnapshotsService.this.shards(currentState, entry.indices());
                            if (!partial) {
                                Tuple indicesWithMissingShards = SnapshotsService.this.indicesWithMissingShards((ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus>)shards, currentState.metaData());
                                Set missing = (Set)indicesWithMissingShards.v1();
                                Set closed = (Set)indicesWithMissingShards.v2();
                                if (!missing.isEmpty() || !closed.isEmpty()) {
                                    StringBuilder failureMessage = new StringBuilder();
                                    this.updatedSnapshot = new SnapshotsInProgress.Entry(entry, SnapshotsInProgress.State.FAILED, (ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus>)shards);
                                    entries.add(this.updatedSnapshot);
                                    if (!missing.isEmpty()) {
                                        failureMessage.append("Indices don't have primary shards ");
                                        failureMessage.append(missing);
                                    }
                                    if (!closed.isEmpty()) {
                                        if (failureMessage.length() > 0) {
                                            failureMessage.append("; ");
                                        }
                                        failureMessage.append("Indices are closed ");
                                        failureMessage.append(closed);
                                    }
                                    this.failure = failureMessage.toString();
                                    continue;
                                }
                            }
                            this.updatedSnapshot = new SnapshotsInProgress.Entry(entry, SnapshotsInProgress.State.STARTED, (ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus>)shards);
                            entries.add(this.updatedSnapshot);
                            if (SnapshotsInProgress.completed((Collection<SnapshotsInProgress.ShardSnapshotStatus>)shards.values())) continue;
                            this.accepted = true;
                            continue;
                        }
                        entries.add(entry);
                    }
                    return ClusterState.builder(currentState).putCustom("snapshots", new SnapshotsInProgress(Collections.unmodifiableList(entries))).build();
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    SnapshotsService.this.logger.warn("[{}] failed to create snapshot", t, snapshot.snapshotId());
                    SnapshotsService.this.removeSnapshotFromClusterState(snapshot.snapshotId(), null, t);
                    try {
                        SnapshotsService.this.repositoriesService.repository(snapshot.snapshotId().getRepository()).finalizeSnapshot(snapshot.snapshotId(), snapshot.indices(), snapshot.startTime(), ExceptionsHelper.detailedMessage(t), 0, Collections.emptyList());
                    }
                    catch (Throwable t2) {
                        SnapshotsService.this.logger.warn("[{}] failed to close snapshot in repository", snapshot.snapshotId());
                    }
                    userCreateSnapshotListener.onFailure(t);
                }

                @Override
                public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                    userCreateSnapshotListener.onResponse();
                    if (!this.accepted && this.updatedSnapshot != null) {
                        SnapshotsService.this.endSnapshot(this.updatedSnapshot, this.failure);
                    }
                }
            });
        }
        catch (Throwable t) {
            this.logger.warn("failed to create snapshot [{}]", t, snapshot.snapshotId());
            this.removeSnapshotFromClusterState(snapshot.snapshotId(), null, t);
            if (snapshotCreated) {
                try {
                    this.repositoriesService.repository(snapshot.snapshotId().getRepository()).finalizeSnapshot(snapshot.snapshotId(), snapshot.indices(), snapshot.startTime(), ExceptionsHelper.detailedMessage(t), 0, Collections.emptyList());
                }
                catch (Throwable t2) {
                    this.logger.warn("[{}] failed to close snapshot in repository", snapshot.snapshotId());
                }
            }
            userCreateSnapshotListener.onFailure(t);
        }
    }

    private Snapshot inProgressSnapshot(SnapshotsInProgress.Entry entry) {
        return new Snapshot(entry.snapshotId().getSnapshot(), entry.indices(), entry.startTime());
    }

    public List<SnapshotsInProgress.Entry> currentSnapshots(String repository, String[] snapshots) {
        SnapshotsInProgress snapshotsInProgress = (SnapshotsInProgress)this.clusterService.state().custom("snapshots");
        if (snapshotsInProgress == null || snapshotsInProgress.entries().isEmpty()) {
            return Collections.emptyList();
        }
        if ("_all".equals(repository)) {
            return snapshotsInProgress.entries();
        }
        if (snapshotsInProgress.entries().size() == 1) {
            SnapshotsInProgress.Entry entry = snapshotsInProgress.entries().get(0);
            if (!entry.snapshotId().getRepository().equals(repository)) {
                return Collections.emptyList();
            }
            if (snapshots != null && snapshots.length > 0) {
                for (String snapshot : snapshots) {
                    if (!entry.snapshotId().getSnapshot().equals(snapshot)) continue;
                    return snapshotsInProgress.entries();
                }
                return Collections.emptyList();
            }
            return snapshotsInProgress.entries();
        }
        ArrayList<SnapshotsInProgress.Entry> builder = new ArrayList<SnapshotsInProgress.Entry>();
        block1: for (SnapshotsInProgress.Entry entry : snapshotsInProgress.entries()) {
            if (!entry.snapshotId().getRepository().equals(repository)) continue;
            if (snapshots != null && snapshots.length > 0) {
                for (String snapshot : snapshots) {
                    if (!entry.snapshotId().getSnapshot().equals(snapshot)) continue;
                    builder.add(entry);
                    continue block1;
                }
                continue;
            }
            builder.add(entry);
        }
        return Collections.unmodifiableList(builder);
    }

    public ImmutableMap<ShardId, IndexShardSnapshotStatus> snapshotShards(SnapshotId snapshotId) throws IOException {
        SnapshotsService.validate(snapshotId);
        ImmutableMap.Builder shardStatusBuilder = ImmutableMap.builder();
        Repository repository = this.repositoriesService.repository(snapshotId.getRepository());
        IndexShardRepository indexShardRepository = this.repositoriesService.indexShardRepository(snapshotId.getRepository());
        Snapshot snapshot = repository.readSnapshot(snapshotId);
        MetaData metaData = repository.readSnapshotMetaData(snapshotId, snapshot, snapshot.indices());
        for (String index : snapshot.indices()) {
            IndexMetaData indexMetaData = metaData.indices().get(index);
            if (indexMetaData == null) continue;
            int numberOfShards = indexMetaData.getNumberOfShards();
            for (int i = 0; i < numberOfShards; ++i) {
                IndexShardSnapshotStatus shardSnapshotStatus;
                ShardId shardId = new ShardId(index, i);
                SnapshotShardFailure shardFailure = this.findShardFailure(snapshot.shardFailures(), shardId);
                if (shardFailure != null) {
                    shardSnapshotStatus = new IndexShardSnapshotStatus();
                    shardSnapshotStatus.updateStage(IndexShardSnapshotStatus.Stage.FAILURE);
                    shardSnapshotStatus.failure(shardFailure.reason());
                    shardStatusBuilder.put((Object)shardId, (Object)shardSnapshotStatus);
                    continue;
                }
                shardSnapshotStatus = indexShardRepository.snapshotStatus(snapshotId, snapshot.version(), shardId);
                shardStatusBuilder.put((Object)shardId, (Object)shardSnapshotStatus);
            }
        }
        return shardStatusBuilder.build();
    }

    private SnapshotShardFailure findShardFailure(List<SnapshotShardFailure> shardFailures, ShardId shardId) {
        for (SnapshotShardFailure shardFailure : shardFailures) {
            if (!shardId.getIndex().equals(shardFailure.index()) || shardId.getId() != shardFailure.shardId()) continue;
            return shardFailure;
        }
        return null;
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        try {
            if (event.localNodeMaster()) {
                if (event.nodesRemoved()) {
                    this.processSnapshotsOnRemovedNodes(event);
                }
                if (event.routingTableChanged()) {
                    this.processStartedShards(event);
                }
            }
        }
        catch (Throwable t) {
            this.logger.warn("Failed to update snapshot state ", t, new Object[0]);
        }
    }

    private void processSnapshotsOnRemovedNodes(ClusterChangedEvent event) {
        if (this.removedNodesCleanupNeeded(event)) {
            final boolean newMaster = !event.previousState().nodes().localNodeMaster();
            this.clusterService.submitStateUpdateTask("update snapshot state after node removal", new ClusterStateUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) throws Exception {
                    DiscoveryNodes nodes = currentState.nodes();
                    SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
                    if (snapshots == null) {
                        return currentState;
                    }
                    boolean changed = false;
                    ArrayList<SnapshotsInProgress.Entry> entries = new ArrayList<SnapshotsInProgress.Entry>();
                    Iterator<SnapshotsInProgress.Entry> i$ = snapshots.entries().iterator();
                    while (i$.hasNext()) {
                        SnapshotsInProgress.Entry snapshot;
                        SnapshotsInProgress.Entry updatedSnapshot = snapshot = i$.next();
                        boolean snapshotChanged = false;
                        if (snapshot.state() == SnapshotsInProgress.State.STARTED || snapshot.state() == SnapshotsInProgress.State.ABORTED) {
                            ImmutableMap.Builder shards = ImmutableMap.builder();
                            for (Map.Entry shardEntry : snapshot.shards().entrySet()) {
                                SnapshotsInProgress.ShardSnapshotStatus shardStatus = (SnapshotsInProgress.ShardSnapshotStatus)shardEntry.getValue();
                                if (shardStatus.state().completed() || shardStatus.nodeId() == null) continue;
                                if (nodes.nodeExists(shardStatus.nodeId())) {
                                    shards.put(shardEntry);
                                    continue;
                                }
                                snapshotChanged = true;
                                SnapshotsService.this.logger.warn("failing snapshot of shard [{}] on closed node [{}]", shardEntry.getKey(), shardStatus.nodeId());
                                shards.put(shardEntry.getKey(), (Object)new SnapshotsInProgress.ShardSnapshotStatus(shardStatus.nodeId(), SnapshotsInProgress.State.FAILED, "node shutdown"));
                            }
                            if (snapshotChanged) {
                                changed = true;
                                ImmutableMap shardsMap = shards.build();
                                if (!snapshot.state().completed() && SnapshotsInProgress.completed((Collection<SnapshotsInProgress.ShardSnapshotStatus>)shardsMap.values())) {
                                    updatedSnapshot = new SnapshotsInProgress.Entry(snapshot, SnapshotsInProgress.State.SUCCESS, (ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus>)shardsMap);
                                    SnapshotsService.this.endSnapshot(updatedSnapshot);
                                } else {
                                    updatedSnapshot = new SnapshotsInProgress.Entry(snapshot, snapshot.state(), (ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus>)shardsMap);
                                }
                            }
                            entries.add(updatedSnapshot);
                            continue;
                        }
                        if (snapshot.state() == SnapshotsInProgress.State.INIT && newMaster) {
                            SnapshotsService.this.deleteSnapshot(snapshot.snapshotId(), new DeleteSnapshotListener(){

                                @Override
                                public void onResponse() {
                                    SnapshotsService.this.logger.debug("cleaned up abandoned snapshot {} in INIT state", snapshot.snapshotId());
                                }

                                @Override
                                public void onFailure(Throwable t) {
                                    SnapshotsService.this.logger.warn("failed to clean up abandoned snapshot {} in INIT state", snapshot.snapshotId());
                                }
                            });
                            continue;
                        }
                        if (snapshot.state() != SnapshotsInProgress.State.SUCCESS || !newMaster) continue;
                        SnapshotsService.this.endSnapshot(snapshot);
                    }
                    if (changed) {
                        snapshots = new SnapshotsInProgress(entries.toArray(new SnapshotsInProgress.Entry[entries.size()]));
                        return ClusterState.builder(currentState).putCustom("snapshots", snapshots).build();
                    }
                    return currentState;
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    SnapshotsService.this.logger.warn("failed to update snapshot state after node removal", new Object[0]);
                }
            });
        }
    }

    private void processStartedShards(ClusterChangedEvent event) {
        if (this.waitingShardsStartedOrUnassigned(event)) {
            this.clusterService.submitStateUpdateTask("update snapshot state after shards started", new ClusterStateUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) throws Exception {
                    RoutingTable routingTable = currentState.routingTable();
                    SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
                    if (snapshots != null) {
                        boolean changed = false;
                        ArrayList<SnapshotsInProgress.Entry> entries = new ArrayList<SnapshotsInProgress.Entry>();
                        Iterator<SnapshotsInProgress.Entry> i$ = snapshots.entries().iterator();
                        while (i$.hasNext()) {
                            SnapshotsInProgress.Entry snapshot;
                            SnapshotsInProgress.Entry updatedSnapshot = snapshot = i$.next();
                            if (snapshot.state() != SnapshotsInProgress.State.STARTED) continue;
                            ImmutableMap shards = SnapshotsService.this.processWaitingShards((ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus>)snapshot.shards(), routingTable);
                            if (shards != null) {
                                changed = true;
                                if (!snapshot.state().completed() && SnapshotsInProgress.completed((Collection<SnapshotsInProgress.ShardSnapshotStatus>)shards.values())) {
                                    updatedSnapshot = new SnapshotsInProgress.Entry(snapshot, SnapshotsInProgress.State.SUCCESS, (ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus>)shards);
                                    SnapshotsService.this.endSnapshot(updatedSnapshot);
                                } else {
                                    updatedSnapshot = new SnapshotsInProgress.Entry(snapshot, (ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus>)shards);
                                }
                            }
                            entries.add(updatedSnapshot);
                        }
                        if (changed) {
                            snapshots = new SnapshotsInProgress(entries.toArray(new SnapshotsInProgress.Entry[entries.size()]));
                            return ClusterState.builder(currentState).putCustom("snapshots", snapshots).build();
                        }
                    }
                    return currentState;
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    SnapshotsService.this.logger.warn("failed to update snapshot state after shards started from [{}] ", t, source);
                }
            });
        }
    }

    private ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus> processWaitingShards(ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus> snapshotShards, RoutingTable routingTable) {
        boolean snapshotChanged = false;
        ImmutableMap.Builder shards = ImmutableMap.builder();
        for (Map.Entry shardEntry : snapshotShards.entrySet()) {
            SnapshotsInProgress.ShardSnapshotStatus shardStatus = (SnapshotsInProgress.ShardSnapshotStatus)shardEntry.getValue();
            if (shardStatus.state() == SnapshotsInProgress.State.WAITING) {
                IndexShardRoutingTable shardRouting;
                ShardId shardId = (ShardId)shardEntry.getKey();
                IndexRoutingTable indexShardRoutingTable = routingTable.index(shardId.getIndex());
                if (indexShardRoutingTable != null && (shardRouting = indexShardRoutingTable.shard(shardId.id())) != null && shardRouting.primaryShard() != null) {
                    if (shardRouting.primaryShard().started()) {
                        snapshotChanged = true;
                        this.logger.trace("starting shard that we were waiting for [{}] on node [{}]", shardEntry.getKey(), shardStatus.nodeId());
                        shards.put(shardEntry.getKey(), (Object)new SnapshotsInProgress.ShardSnapshotStatus(shardRouting.primaryShard().currentNodeId()));
                        continue;
                    }
                    if (shardRouting.primaryShard().initializing() || shardRouting.primaryShard().relocating()) {
                        shards.put(shardEntry);
                        continue;
                    }
                }
                snapshotChanged = true;
                this.logger.warn("failing snapshot of shard [{}] on unassigned shard [{}]", shardEntry.getKey(), shardStatus.nodeId());
                shards.put(shardEntry.getKey(), (Object)new SnapshotsInProgress.ShardSnapshotStatus(shardStatus.nodeId(), SnapshotsInProgress.State.FAILED, "shard is unassigned"));
                continue;
            }
            shards.put(shardEntry);
        }
        if (snapshotChanged) {
            return shards.build();
        }
        return null;
    }

    private boolean waitingShardsStartedOrUnassigned(ClusterChangedEvent event) {
        SnapshotsInProgress curr = (SnapshotsInProgress)event.state().custom("snapshots");
        if (curr != null) {
            for (SnapshotsInProgress.Entry entry : curr.entries()) {
                if (entry.state() != SnapshotsInProgress.State.STARTED || entry.waitingIndices().isEmpty()) continue;
                for (String index : entry.waitingIndices().keySet()) {
                    if (!event.indexRoutingTableChanged(index)) continue;
                    IndexRoutingTable indexShardRoutingTable = event.state().getRoutingTable().index(index);
                    for (ShardId shardId : (List)entry.waitingIndices().get((Object)index)) {
                        ShardRouting shardRouting = indexShardRoutingTable.shard(shardId.id()).primaryShard();
                        if (shardRouting == null || !shardRouting.started() && !shardRouting.unassigned()) continue;
                        return true;
                    }
                }
            }
        }
        return false;
    }

    private boolean removedNodesCleanupNeeded(ClusterChangedEvent event) {
        boolean newMaster = !event.previousState().nodes().localNodeMaster();
        SnapshotsInProgress snapshotsInProgress = (SnapshotsInProgress)event.state().custom("snapshots");
        if (snapshotsInProgress == null) {
            return false;
        }
        for (SnapshotsInProgress.Entry snapshot : snapshotsInProgress.entries()) {
            if (newMaster && (snapshot.state() == SnapshotsInProgress.State.SUCCESS || snapshot.state() == SnapshotsInProgress.State.INIT)) {
                return true;
            }
            for (DiscoveryNode node : event.nodesDelta().removedNodes()) {
                for (SnapshotsInProgress.ShardSnapshotStatus shardStatus : snapshot.shards().values()) {
                    if (shardStatus.state().completed() || !node.getId().equals(shardStatus.nodeId())) continue;
                    return true;
                }
            }
        }
        return false;
    }

    private Tuple<Set<String>, Set<String>> indicesWithMissingShards(ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus> shards, MetaData metaData) {
        HashSet missing = Sets.newHashSet();
        HashSet closed = Sets.newHashSet();
        for (Map.Entry entry : shards.entrySet()) {
            if (((SnapshotsInProgress.ShardSnapshotStatus)entry.getValue()).state() != SnapshotsInProgress.State.MISSING) continue;
            if (metaData.hasIndex(((ShardId)entry.getKey()).getIndex()) && metaData.index(((ShardId)entry.getKey()).getIndex()).getState() == IndexMetaData.State.CLOSE) {
                closed.add(((ShardId)entry.getKey()).getIndex());
                continue;
            }
            missing.add(((ShardId)entry.getKey()).getIndex());
        }
        return new Tuple<Set<String>, Set<String>>(missing, closed);
    }

    void endSnapshot(SnapshotsInProgress.Entry entry) {
        this.endSnapshot(entry, null);
    }

    private void endSnapshot(final SnapshotsInProgress.Entry entry, final String failure) {
        this.threadPool.executor("snapshot").execute(new Runnable(){

            @Override
            public void run() {
                SnapshotId snapshotId = entry.snapshotId();
                try {
                    Repository repository = SnapshotsService.this.repositoriesService.repository(snapshotId.getRepository());
                    SnapshotsService.this.logger.trace("[{}] finalizing snapshot in repository, state: [{}], failure[{}]", new Object[]{snapshotId, entry.state(), failure});
                    ArrayList<ShardSearchFailure> failures = new ArrayList<ShardSearchFailure>();
                    ArrayList<SnapshotShardFailure> shardFailures = new ArrayList<SnapshotShardFailure>();
                    for (Map.Entry shardStatus : entry.shards().entrySet()) {
                        ShardId shardId = (ShardId)shardStatus.getKey();
                        SnapshotsInProgress.ShardSnapshotStatus status = (SnapshotsInProgress.ShardSnapshotStatus)shardStatus.getValue();
                        if (!status.state().failed()) continue;
                        failures.add(new ShardSearchFailure(status.reason(), new SearchShardTarget(status.nodeId(), shardId.getIndex(), shardId.id())));
                        shardFailures.add(new SnapshotShardFailure(status.nodeId(), shardId.getIndex(), shardId.id(), status.reason()));
                    }
                    Snapshot snapshot = repository.finalizeSnapshot(snapshotId, entry.indices(), entry.startTime(), failure, entry.shards().size(), Collections.unmodifiableList(shardFailures));
                    SnapshotsService.this.removeSnapshotFromClusterState(snapshotId, new SnapshotInfo(snapshot), null);
                }
                catch (Throwable t) {
                    SnapshotsService.this.logger.warn("[{}] failed to finalize snapshot", t, snapshotId);
                    SnapshotsService.this.removeSnapshotFromClusterState(snapshotId, null, t);
                }
            }
        });
    }

    private void removeSnapshotFromClusterState(final SnapshotId snapshotId, final SnapshotInfo snapshot, final Throwable t) {
        this.clusterService.submitStateUpdateTask("remove snapshot metadata", new ClusterStateUpdateTask(){

            @Override
            public ClusterState execute(ClusterState currentState) {
                SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
                if (snapshots != null) {
                    boolean changed = false;
                    ArrayList<SnapshotsInProgress.Entry> entries = new ArrayList<SnapshotsInProgress.Entry>();
                    for (SnapshotsInProgress.Entry entry : snapshots.entries()) {
                        if (entry.snapshotId().equals(snapshotId)) {
                            changed = true;
                            continue;
                        }
                        entries.add(entry);
                    }
                    if (changed) {
                        snapshots = new SnapshotsInProgress(entries.toArray(new SnapshotsInProgress.Entry[entries.size()]));
                        return ClusterState.builder(currentState).putCustom("snapshots", snapshots).build();
                    }
                }
                return currentState;
            }

            @Override
            public void onFailure(String source, Throwable t2) {
                SnapshotsService.this.logger.warn("[{}][{}] failed to remove snapshot metadata", t2, snapshotId);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                for (SnapshotCompletionListener listener : SnapshotsService.this.snapshotCompletionListeners) {
                    try {
                        if (snapshot != null) {
                            listener.onSnapshotCompletion(snapshotId, snapshot);
                            continue;
                        }
                        listener.onSnapshotFailure(snapshotId, t);
                    }
                    catch (Throwable t2) {
                        SnapshotsService.this.logger.warn("failed to notify listener [{}]", t2, listener);
                    }
                }
            }
        });
    }

    public void deleteSnapshot(final SnapshotId snapshotId, final DeleteSnapshotListener listener) {
        SnapshotsService.validate(snapshotId);
        this.clusterService.submitStateUpdateTask("delete snapshot", new ClusterStateUpdateTask(){
            boolean waitForSnapshot = false;

            @Override
            public ClusterState execute(ClusterState currentState) throws Exception {
                ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus> shards;
                SnapshotsInProgress snapshots = (SnapshotsInProgress)currentState.custom("snapshots");
                if (snapshots == null) {
                    return currentState;
                }
                SnapshotsInProgress.Entry snapshot = snapshots.snapshot(snapshotId);
                if (snapshot == null) {
                    if (!snapshots.entries().isEmpty()) {
                        throw new ConcurrentSnapshotExecutionException(snapshotId, "another snapshot is currently running cannot delete");
                    }
                    return currentState;
                }
                this.waitForSnapshot = true;
                if (snapshot.state() == SnapshotsInProgress.State.STARTED && snapshot.shards() != null) {
                    ImmutableMap.Builder shardsBuilder = ImmutableMap.builder();
                    for (Map.Entry shardEntry : snapshot.shards().entrySet()) {
                        SnapshotsInProgress.ShardSnapshotStatus status = (SnapshotsInProgress.ShardSnapshotStatus)shardEntry.getValue();
                        if (!status.state().completed()) {
                            shardsBuilder.put(shardEntry.getKey(), (Object)new SnapshotsInProgress.ShardSnapshotStatus(status.nodeId(), SnapshotsInProgress.State.ABORTED));
                            continue;
                        }
                        shardsBuilder.put(shardEntry.getKey(), (Object)status);
                    }
                    shards = shardsBuilder.build();
                } else if (snapshot.state() == SnapshotsInProgress.State.INIT) {
                    shards = snapshot.shards();
                    SnapshotsService.this.endSnapshot(snapshot);
                } else {
                    boolean hasUncompletedShards = false;
                    for (SnapshotsInProgress.ShardSnapshotStatus shardStatus : snapshot.shards().values()) {
                        if (shardStatus.state().completed() || shardStatus.nodeId() == null || currentState.nodes().get(shardStatus.nodeId()) == null) continue;
                        hasUncompletedShards = true;
                        break;
                    }
                    if (hasUncompletedShards) {
                        SnapshotsService.this.logger.debug("trying to delete completed snapshot - should wait for shards to finalize on all nodes", new Object[0]);
                        return currentState;
                    }
                    SnapshotsService.this.logger.debug("trying to delete completed snapshot with no finalizing shards - can delete immediately", new Object[0]);
                    shards = snapshot.shards();
                    SnapshotsService.this.endSnapshot(snapshot);
                }
                SnapshotsInProgress.Entry newSnapshot = new SnapshotsInProgress.Entry(snapshot, SnapshotsInProgress.State.ABORTED, shards);
                snapshots = new SnapshotsInProgress(newSnapshot);
                return ClusterState.builder(currentState).putCustom("snapshots", snapshots).build();
            }

            @Override
            public void onFailure(String source, Throwable t) {
                listener.onFailure(t);
            }

            @Override
            public void clusterStateProcessed(String source, ClusterState oldState, ClusterState newState) {
                if (this.waitForSnapshot) {
                    SnapshotsService.this.logger.trace("adding snapshot completion listener to wait for deleted snapshot to finish", new Object[0]);
                    SnapshotsService.this.addListener(new SnapshotCompletionListener(){

                        @Override
                        public void onSnapshotCompletion(SnapshotId completedSnapshotId, SnapshotInfo snapshot) {
                            if (completedSnapshotId.equals(snapshotId)) {
                                SnapshotsService.this.logger.trace("deleted snapshot completed - deleting files", new Object[0]);
                                SnapshotsService.this.removeListener(this);
                                SnapshotsService.this.deleteSnapshotFromRepository(snapshotId, listener);
                            }
                        }

                        @Override
                        public void onSnapshotFailure(SnapshotId failedSnapshotId, Throwable t) {
                            if (failedSnapshotId.equals(snapshotId)) {
                                SnapshotsService.this.logger.trace("deleted snapshot failed - deleting files", t, new Object[0]);
                                SnapshotsService.this.removeListener(this);
                                SnapshotsService.this.deleteSnapshotFromRepository(snapshotId, listener);
                            }
                        }
                    });
                } else {
                    SnapshotsService.this.logger.trace("deleted snapshot is not running - deleting files", new Object[0]);
                    SnapshotsService.this.deleteSnapshotFromRepository(snapshotId, listener);
                }
            }
        });
    }

    public static boolean isRepositoryInUse(ClusterState clusterState, String repository) {
        SnapshotsInProgress snapshots = (SnapshotsInProgress)clusterState.custom("snapshots");
        if (snapshots != null) {
            for (SnapshotsInProgress.Entry snapshot : snapshots.entries()) {
                if (!repository.equals(snapshot.snapshotId().getRepository())) continue;
                return true;
            }
        }
        return false;
    }

    private void deleteSnapshotFromRepository(final SnapshotId snapshotId, final DeleteSnapshotListener listener) {
        this.threadPool.executor("snapshot").execute(new Runnable(){

            @Override
            public void run() {
                try {
                    Repository repository = SnapshotsService.this.repositoriesService.repository(snapshotId.getRepository());
                    repository.deleteSnapshot(snapshotId);
                    listener.onResponse();
                }
                catch (Throwable t) {
                    listener.onFailure(t);
                }
            }
        });
    }

    private ImmutableMap<ShardId, SnapshotsInProgress.ShardSnapshotStatus> shards(ClusterState clusterState, List<String> indices) {
        ImmutableMap.Builder builder = ImmutableMap.builder();
        MetaData metaData = clusterState.metaData();
        for (String index : indices) {
            IndexMetaData indexMetaData = metaData.index(index);
            if (indexMetaData == null) {
                builder.put((Object)new ShardId(index, 0), (Object)new SnapshotsInProgress.ShardSnapshotStatus(null, SnapshotsInProgress.State.MISSING, "missing index"));
                continue;
            }
            if (indexMetaData.getState() == IndexMetaData.State.CLOSE) {
                for (int i = 0; i < indexMetaData.getNumberOfShards(); ++i) {
                    ShardId shardId = new ShardId(index, i);
                    builder.put((Object)shardId, (Object)new SnapshotsInProgress.ShardSnapshotStatus(null, SnapshotsInProgress.State.MISSING, "index is closed"));
                }
                continue;
            }
            IndexRoutingTable indexRoutingTable = clusterState.getRoutingTable().index(index);
            for (int i = 0; i < indexMetaData.getNumberOfShards(); ++i) {
                ShardId shardId = new ShardId(index, i);
                if (indexRoutingTable != null) {
                    ShardRouting primary = indexRoutingTable.shard(i).primaryShard();
                    if (primary == null || !primary.assignedToNode()) {
                        builder.put((Object)shardId, (Object)new SnapshotsInProgress.ShardSnapshotStatus(null, SnapshotsInProgress.State.MISSING, "primary shard is not allocated"));
                        continue;
                    }
                    if (primary.relocating() || primary.initializing()) {
                        builder.put((Object)shardId, (Object)new SnapshotsInProgress.ShardSnapshotStatus(primary.currentNodeId(), SnapshotsInProgress.State.WAITING));
                        continue;
                    }
                    if (!primary.started()) {
                        builder.put((Object)shardId, (Object)new SnapshotsInProgress.ShardSnapshotStatus(primary.currentNodeId(), SnapshotsInProgress.State.MISSING, "primary shard hasn't been started yet"));
                        continue;
                    }
                    builder.put((Object)shardId, (Object)new SnapshotsInProgress.ShardSnapshotStatus(primary.currentNodeId()));
                    continue;
                }
                builder.put((Object)shardId, (Object)new SnapshotsInProgress.ShardSnapshotStatus(null, SnapshotsInProgress.State.MISSING, "missing routing table"));
            }
        }
        return builder.build();
    }

    public void addListener(SnapshotCompletionListener listener) {
        this.snapshotCompletionListeners.add(listener);
    }

    public void removeListener(SnapshotCompletionListener listener) {
        this.snapshotCompletionListeners.remove(listener);
    }

    @Override
    protected void doStart() {
    }

    @Override
    protected void doStop() {
    }

    @Override
    protected void doClose() {
        this.clusterService.remove(this);
    }

    public RepositoriesService getRepositoriesService() {
        return this.repositoriesService;
    }

    public static class SnapshotRequest {
        private String cause;
        private String name;
        private String repository;
        private String[] indices;
        private IndicesOptions indicesOptions = IndicesOptions.strictExpandOpen();
        private boolean partial;
        private Settings settings;
        private boolean includeGlobalState;
        private TimeValue masterNodeTimeout;

        public SnapshotRequest(String cause, String name, String repository) {
            this.cause = cause;
            this.name = name;
            this.repository = repository;
        }

        public SnapshotRequest indices(String[] indices) {
            this.indices = indices;
            return this;
        }

        public SnapshotRequest settings(Settings settings) {
            this.settings = settings;
            return this;
        }

        public SnapshotRequest includeGlobalState(boolean includeGlobalState) {
            this.includeGlobalState = includeGlobalState;
            return this;
        }

        public SnapshotRequest masterNodeTimeout(TimeValue masterNodeTimeout) {
            this.masterNodeTimeout = masterNodeTimeout;
            return this;
        }

        public SnapshotRequest indicesOptions(IndicesOptions indicesOptions) {
            this.indicesOptions = indicesOptions;
            return this;
        }

        public SnapshotRequest partial(boolean partial) {
            this.partial = partial;
            return this;
        }

        public String cause() {
            return this.cause;
        }

        public String name() {
            return this.name;
        }

        public String repository() {
            return this.repository;
        }

        public String[] indices() {
            return this.indices;
        }

        public IndicesOptions indicesOptions() {
            return this.indicesOptions;
        }

        public Settings settings() {
            return this.settings;
        }

        public boolean includeGlobalState() {
            return this.includeGlobalState;
        }

        public TimeValue masterNodeTimeout() {
            return this.masterNodeTimeout;
        }
    }

    public static interface SnapshotCompletionListener {
        public void onSnapshotCompletion(SnapshotId var1, SnapshotInfo var2);

        public void onSnapshotFailure(SnapshotId var1, Throwable var2);
    }

    public static interface DeleteSnapshotListener {
        public void onResponse();

        public void onFailure(Throwable var1);
    }

    public static interface CreateSnapshotListener {
        public void onResponse();

        public void onFailure(Throwable var1);
    }
}

