/**
 *
 */
package org.gcube.spatial.data.geoutility;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import org.gcube.spatial.data.geoutility.bean.NcWmsLayerMetadata;
import org.gcube.spatial.data.geoutility.bean.WmsParameters;
import org.gcube.spatial.data.geoutility.wms.NcWmsGetMetadata;
import org.gcube.spatial.data.geoutility.wms.NcWmsGetMetadataRequest;
import org.gcube.spatial.data.geoutility.wms.WmsGetStyles;
import org.gcube.spatial.data.geoutility.wms.WmsUrlValidator;


/**
 * The Class GeoGetStylesUtility.
 *
 * @author Francesco Mangiacrapa francesco.mangiacrapa@isti.cnr.it
 * Jan 25, 2016
 */
public class GeoGetStylesUtility {

	public static Logger logger = Logger.getLogger(GeoGetStylesUtility.class);
	public static final String COLORSCALERANGE = "COLORSCALERANGE";
	public static final String COLORSCALERANGE_DEFAULT_VALUE = "auto";
	boolean isNcWms  = false;
	private String wmsRequest;
	private List<String> geoStyles;
	private Map<String, String> mapNcWmsStyles;
	private WmsUrlValidator validator;

	/**
	 * Instantiates a new geo get styles.
	 *
	 * @param wmsRequest the wms request
	 * @throws Exception the exception
	 */
	public GeoGetStylesUtility(String wmsRequest) throws Exception{
		this.wmsRequest = wmsRequest;
		this.validator = new WmsUrlValidator(wmsRequest);
		validator.parseWmsRequest(true, false);
		loadStyles();
	}



	/**
	 * Instantiates a new geo get styles utility.
	 *
	 * @param wmsRequest the wms request
	 * @param validator the validator
	 * @throws Exception the exception
	 */
	public GeoGetStylesUtility(String wmsRequest, WmsUrlValidator validator) throws Exception{
		this.validator = validator;
		loadStyles();
	}

	/**
	 * Load styles.
	 */
	private void loadStyles(){

	    String uriWMSService = validator.getBaseWmsServiceUrl();
	    String layerName = validator.getValueOfParsedWMSParameter(WmsParameters.LAYERS);
	    String versionWms = validator.getValueOfParsedWMSParameter(WmsParameters.VERSION);
	    String crs = validator.getValueOfParsedWMSParameter(WmsParameters.CRS);

		try {

			//OVERRIDE ncWMS COLORSCALERANGE='auto'
			if(validator.getMapWmsNoStandardParams()!=null){
				mapNcWmsStyles = new HashMap<String, String>(validator.getMapWmsNoStandardParams().size());
//				mapNcWmsStyles.putAll(validator.getMapWmsNotStandardParams());

				if(validator.getMapWmsNoStandardParams().containsKey(NcWmsGetMetadataRequest.COLORSCALERANGE) || validator.getMapWmsNoStandardParams().containsKey(NcWmsGetMetadataRequest.COLORSCALERANGE.toLowerCase())){
					isNcWms = true;
					logger.debug("Ovveriding "+NcWmsGetMetadataRequest.COLORSCALERANGE +"?");
					String value = validator.getMapWmsNoStandardParams().get(NcWmsGetMetadataRequest.COLORSCALERANGE);

					if(value.compareToIgnoreCase(NcWmsGetMetadataRequest.COLORSCALERANGE_DEFAULT_VALUE)==0){ //COLORSCALERANGE == 'auto'?
						logger.trace(NcWmsGetMetadataRequest.COLORSCALERANGE +" found like 'auto'..");
						String[] minmax = NcWmsGetMetadataRequest.getColorScaleRange(uriWMSService,
							versionWms,
							validator.getValueOfParsedWMSParameter(WmsParameters.BBOX),
							layerName,
							crs,
							validator.getValueOfParsedWMSParameter(WmsParameters.WIDTH),
							validator.getValueOfParsedWMSParameter(WmsParameters.HEIGHT));

						if(minmax!=null && minmax.length==2 && minmax[0]!=null && minmax[1]!=null){
							String valueMinMax =  minmax[0]+","+minmax[1];
							mapNcWmsStyles.put(NcWmsGetMetadataRequest.COLORSCALERANGE, valueMinMax);
							logger.debug("Overrided "+NcWmsGetMetadataRequest.COLORSCALERANGE +" min,max value: "+valueMinMax);
						}
					}else{
						logger.debug(NcWmsGetMetadataRequest.COLORSCALERANGE +" is not \'auto', skipping override");
					}
				}else{ //NcWmsGetColorScaleRange.COLORSCALERANGE not found
					logger.debug(NcWmsGetMetadataRequest.COLORSCALERANGE +" not found");

					String[] minmax = NcWmsGetMetadataRequest.getColorScaleRange(uriWMSService,
							versionWms,
							validator.getValueOfParsedWMSParameter(WmsParameters.BBOX),
							layerName,
							crs,
							validator.getValueOfParsedWMSParameter(WmsParameters.WIDTH),
							validator.getValueOfParsedWMSParameter(WmsParameters.HEIGHT));

						if(minmax!=null && minmax.length==2 && minmax[0]!=null && minmax[1]!=null){
							isNcWms = true;
							logger.debug(NcWmsGetMetadataRequest.GET_METADATA +" works, adding "+NcWmsGetMetadataRequest.COLORSCALERANGE + " with minmax");
							String valueMinMax =  minmax[0]+","+minmax[1];
							mapNcWmsStyles.put(NcWmsGetMetadataRequest.COLORSCALERANGE, valueMinMax);
							logger.debug("Added "+NcWmsGetMetadataRequest.COLORSCALERANGE +" min,max value: "+valueMinMax);
						}else{
							logger.debug(NcWmsGetMetadataRequest.GET_METADATA +" failed, skip "+NcWmsGetMetadataRequest.COLORSCALERANGE +" management, is not raster?");
						}
				}
			}
		}catch (Exception e) {
			logger.error("Exception during ncWMS get style: "+e);
		}

		logger.trace("versionWms found: "+versionWms);
		logger.trace("crs found: "+crs);

		String stylesRead = validator.getValueOfParsedWMSParameter(WmsParameters.STYLES);
		logger.trace("styles found: "+stylesRead);

		List<String> styles;
		//VALIDATION STYLES
		if(stylesRead==null || stylesRead.isEmpty()){

			logger.trace("styles are empty or null - Trying to get styles by 'WMS Get Style'");
			WmsGetStyles wmsGetStyles = new WmsGetStyles();
			styles = wmsGetStyles.getStylesFromWms(uriWMSService, layerName);

			if(wmsGetStyles.getResponseCode()==200){
				logger.debug("WMS Get Style found: "+styles);
			}else{
				logger.debug("Wms GetStyles not found, Trying to get styles by 'NcWmsGetMetadata'");
				isNcWms = true;
				try{
					NcWmsLayerMetadata ncMetadata = NcWmsGetMetadata.getMetadata(uriWMSService, layerName);

					if(ncMetadata!=null && ncMetadata.getResponseCode()==200){
						if(ncMetadata.getSupportedStyles().size()>0){
							styles.add(ncMetadata.getSupportedStyles().get(0)+"/"+ncMetadata.getDefaultPalette()); //DEFAULT STYLE
							logger.debug("added ncWms default style: "+ncMetadata.getSupportedStyles().get(0)+"/"+ncMetadata.getDefaultPalette());

							for (String s : ncMetadata.getSupportedStyles()) {
								for (String p : ncMetadata.getPalettes()) {
									String as = s+"/"+p;
									if(!styles.contains(as)){
										styles.add(as);
										logger.trace("added ncWMS style: "+as);
									}
								}
							}
						}
					}
				}catch(Exception e){
					logger.error(e);
					styles = validator.getStylesAsList();
				}
			}
		}else{
			styles = validator.getStylesAsList();
		}
		logger.trace("returning style: "+styles);
		this.geoStyles = styles;
	}

	/**
	 * Gets the geo styles.
	 *
	 * @return the geoStyles
	 */
	public List<String> getGeoStyles() {

		return geoStyles;
	}

	/**
	 * Checks if is nc wms.
	 *
	 * @return the isNcWms
	 */
	public boolean isNcWms() {

		return isNcWms;
	}

	/**
	 * Gets the wms request.
	 *
	 * @return the wmsRequest
	 */
	public String getWmsRequest() {

		return wmsRequest;
	}

	/**
	 * Gets the map wms no standard parameters.
	 *
	 * @return the map wms no standard parameters
	 */
	public Map<String, String> getMapWmsNoStandardParameters(){
		return validator.getMapWmsNoStandardParams();
	}

	/**
	 * Gets the map ncWMS styles.
	 *
	 * @return an empty map if ncWMS no standard parameters don't exist
	 */
	public Map<String, String> getMapNcWmsStyles() {
		return mapNcWmsStyles == null?new HashMap<String, String>(1):mapNcWmsStyles;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {

		StringBuilder builder = new StringBuilder();
		builder.append("GeoGetStylesUtility [isNcWms=");
		builder.append(isNcWms);
		builder.append(", wmsRequest=");
		builder.append(wmsRequest);
		builder.append(", geoStyles=");
		builder.append(geoStyles);
		builder.append(", mapNcWmsStyles=");
		builder.append(mapNcWmsStyles);
		builder.append(", validator=");
		builder.append(validator);
		builder.append("]");
		return builder.toString();
	}
}
