package org.gcube.deploytest.client;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.StringBufferInputStream;
import java.util.ArrayList;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.gcube.common.core.resources.GCUBEService;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/**
 * Check the Deployment report 
 * 
 * @author Pedro Andrade (CERN) and Andrea Manzi (CERN)
 *
 */
public class QueryDeploymentInformation {

	Document dom;

	public QueryDeploymentInformation(){

	}

	/**
	 * Create a dom of the Deployment report file
	 * 
	 * @param file the Deployment report file
	 */
	public void parseXmlFile(String file){

		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();

		try {
			DocumentBuilder db = dbf.newDocumentBuilder();
			dom = db.parse(file);
		}catch(ParserConfigurationException pce) {
			pce.printStackTrace();
		}catch(SAXException se) {
			se.printStackTrace();
		}catch(IOException ioe) {
			ioe.printStackTrace();
		}
	}

	/**
	 * Create a dom of the Deployment report string
	 * 
	 * @param file the Deployment report string
	 */
	public void parseXml(String xml){


		DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
		try {
			DocumentBuilder db = dbf.newDocumentBuilder();
			dom = db.parse(new StringBufferInputStream(xml));
		}catch(ParserConfigurationException pce) {
			pce.printStackTrace();
		}catch(SAXException se) {
			se.printStackTrace();
		}catch(IOException ioe) {
			ioe.printStackTrace();
		}
	}

	/**
	 * Get the Dependency resolution element as string
	 * @return the Dependency resolution element as string
	 */
	public String getDependenciesResolutionStatus(){
		Element docEle = dom.getDocumentElement();
		NodeList nl = docEle.getElementsByTagName("DependenciesResolutionStatus");
		String textVal = null;
		if(nl != null && nl.getLength() > 0) {
			for(int i = 0 ; i < nl.getLength();i++) {
				Element el = (Element)nl.item(i);
				textVal = el.getFirstChild().getNodeValue();
			}
		}
		return textVal;
	}
	/**
	 * Get the Dependency resolution element as string
	 * @return the Dependency resolution element as string
	 */
	public boolean getDeploymentActivityStatus(){
		Element docEle = dom.getDocumentElement();
		NodeList nl = docEle.getElementsByTagName("LastReportReceived");
		
		if(nl != null && nl.getLength() > 0) {
			for(int i = 0 ; i < nl.getLength();i++) {
				Node statuses = nl.item(i);
				NodeList children = statuses.getChildNodes();
			
				String status = null;
				for(int k = 0 ; k < children.getLength();k++) {
	
					if ( children.item(k) instanceof Element)
					{
	
						if (((Element) children.item(k)).getTagName().equals("Status"))
						{
							status = ((Element) children.item(k)).getTextContent();
							Deploy.logInfo("The Deployment status is: "+ status);
							if (status.equals("FAILED")){
								//the deployment is not yet finished we should wait 
								Deploy.logInfo("\nThe undeploment has failed\n");
								this.writeFile("FAILED\n");
								System.exit(1);
							}
							else if (status.equals("OPEN")) return false;

					}
				}
			}
		}
		}else return false;
		
		return true;
	}


	/**
	 * 
	 *  Get the Statuses of the deployment as a NodeList
	 * @return the NodeList containing deployment statuses
	 */
	private NodeList getServices(){
		Element docEle = dom.getDocumentElement();
		NodeList nl = docEle.getElementsByTagName("Service");		
		return nl;

	}
	/**
	 * 
	 *  Get the Resources 
	 * @return the NodeList containing deployment statuses
	 */
	private NodeList getResources(){
		Element docEle = dom.getDocumentElement();
		NodeList nl = docEle.getElementsByTagName("Resource");		
		return nl;

	}

	/**
	 * Get the Packages Information  as a NodeList
	 * @return the Packages Information
	 */
	private NodeList getPackagesDeployed(){
		Element docEle = dom.getDocumentElement();
		NodeList nl = docEle.getElementsByTagName("Package");		
		return nl;

	}
	
	/**
	 * <DeploymentPlanCreation>
<Status>FAILED</Status>
<Message>unable to find an instance of the Resource Broker in scope</Message>
</DeploymentPlanCreation>

	 */
	private boolean getDeploymentPlanCreation(){
		Element docEle = dom.getDocumentElement();
		NodeList nl = docEle.getElementsByTagName("DeploymentPlanCreation");		
		String status = "";
		String message = "";
		if(nl != null && nl.getLength() > 0) {
			for(int i = 0 ; i < nl.getLength();i++) {

				if ( nl.item(i) instanceof Element)
				{

					if (((Element) nl.item(i)).getTagName().equals("Status"))
					{
						status = ((Element) nl.item(i)).getTextContent();
					}
					if (((Element) nl.item(i)).getTagName().equals("Message"))
					{
						message = ((Element) nl.item(i)).getTextContent();
					}

				}
			}
			
			
		}
		if (status.equals("FAILED"))
		{
			Deploy.logError("The following error has been encountered: "+ message);
			return false;
		}
		else return true;
			
		
	}

	/**
	 * Get a Package given the  GCUBEService profile and the package name
	 * @param service the  GCUBEService profile
	 * @param packageName the package name 
	 * @return the Package
	 */
	public org.gcube.common.core.resources.service.Package getPackage(GCUBEService service,String packageName) {
		for ( org.gcube.common.core.resources.service.Package pack : service.getPackages())
		{
			if (pack.getName().equals(packageName)) 
				return pack;
		}
		return null;
	}

	/**
	 * Get a Software File list
	 * 
	 * @param service the Software Package
	 * @return a list of File names
	 */
	public ArrayList<String> getFileList(org.gcube.common.core.resources.service.Package service) {

		ArrayList<String> files = new ArrayList<String>(); 

		if (service instanceof org.gcube.common.core.resources.service.Software)
		{
			for (String file : ((org.gcube.common.core.resources.service.Software) service).getFiles()){
				int index = file.lastIndexOf("/");
				if (index != -1) 
					file = file.substring(index+1);
				files.add(file);
			}
		}

		return files;
	} 

	/**
	 *
	 *  Get the Statuses of the deployment as a NodeList
	 * @return the NodeList containing deployment statuses
	 */
	private NodeList getLastReportReceived(){
		Element docEle = dom.getDocumentElement();
		NodeList nl = docEle.getElementsByTagName("Status");           
		return nl;

	}

	/**
	 * Write deployment information on a file
	 * @param status the deployment status to log
	 */
	public void writeFile(String status){
		try {
			BufferedWriter out = new BufferedWriter(new FileWriter("deployment_status"));
			out.write(status);
			out.close();
		} catch (IOException e) {
		}
	}

	/**
	 * Write the status of the VRE creation on a file
	 * 
	 * @param status the VRE Creation status
	 */
	public void writeVREFile(String status){
		try {
			BufferedWriter out = new BufferedWriter(new FileWriter("vre_status"));
			out.write(status);
			out.close();
		} catch (IOException e) {
		}
	}


	/**
	 *   <Resources>
	 * @param resources
	 */
	private boolean parseResources(NodeList resources){

		boolean finalStatus = true;

		for(int i = 0 ; i < resources.getLength();i++) {
			Node resource = resources.item(i);

			if (resource != null)
			{
				NodeList children = resource.getChildNodes();
				String ID = null;
				String Type = null;
				String status = null;

				for(int k = 0 ; k < children.getLength();k++) 
				{
					if ( children.item(k) instanceof Element)
					{

						if (((Element) children.item(k)).getTagName().equals("ID"))
						{
							ID = ((Element) children.item(k)).getTextContent();
						}
						else if (((Element) children.item(k)).getTagName().equals("Type"))
						{
							Type = ((Element) children.item(k)).getTextContent();
						}
						else if (((Element) children.item(k)).getTagName().equals("Status"))
						{
							status = ((Element) children.item(k)).getTextContent();
						}
					}
				}
				Deploy.logInfo("Resource with ID "+ ID);
				Deploy.logInfo("with Type "+ Type);
				Deploy.logInfo("has been processed with status "+ status);
				if (status.equals("FAILED"))
					finalStatus = false;
			}
		}
		return finalStatus;

	}

	private  boolean parseDependenciesResolutionStatus(NodeList services){

		boolean finalStatus = true;

		for(int i = 0 ; i < services.getLength();i++) {
			Node service = services.item(i);

			if (service != null)
			{
				NodeList children = service.getChildNodes();
				String clazz = null;
				String name = null;
				String status = null;
				String errorDesc = null;

				for(int k = 0 ; k < children.getLength();k++) 
				{
					if ( children.item(k) instanceof Element)
					{

						if (((Element) children.item(k)).getTagName().equals("Class"))
						{
							clazz = ((Element) children.item(k)).getTextContent();
						}
						else if (((Element) children.item(k)).getTagName().equals("Name"))
						{
							name = ((Element) children.item(k)).getTextContent();
						}
						else if (((Element) children.item(k)).getTagName().equals("DependenciesResolutionStatus"))
						{
							status = ((Element) children.item(k)).getTextContent();
						}
						else if (((Element) children.item(k)).getTagName().equals("ErrorDescription"))
						{
							errorDesc = ((Element) children.item(k)).getTextContent();
						}


					}
				}
				Deploy.logInfo("Dependency Resolution for Service "+ clazz +"/"+name);
				Deploy.logInfo("has been processed with status "+ status);
				if (status.equals("FAILED")) {
					Deploy.logInfo("The following error has been encountered: "+ errorDesc);
					finalStatus = false;
				}
			}
		}
		return finalStatus;


	}

	//method modified starting from gCube 2.9.0 because of Maen integration ( not all services are published on the IS)
	private void checkPackagesDeployment(NodeList packages,boolean isDeployment){

		for(int i = 0 ; i < packages.getLength();i++) {
			Node pack = packages.item(i);

			if (pack != null)
			{
				NodeList children = pack.getChildNodes();
				String serviceName = null;
				String serviceClass = null;
				String status = null;
				String packageName = null;
				GCUBEService service = null;

				 
				for(int k = 0 ; k < children.getLength();k++) 
				{
					if ( children.item(k) instanceof Element)
					{

						if (((Element) children.item(k)).getTagName().equals("ServiceClass"))
						{
							serviceClass = ((Element) children.item(k)).getTextContent();
						}
						else if (((Element) children.item(k)).getTagName().equals("ServiceName"))
						{
							serviceName = ((Element) children.item(k)).getTextContent();
						}
						else if (((Element) children.item(k)).getTagName().equals("PackageName"))
						{
							packageName = ((Element) children.item(k)).getTextContent();
						}
						else if (((Element) children.item(k)).getTagName().equals("Status"))
						{
							status = ((Element) children.item(k)).getTextContent();
						}
					}
				}
				if (isDeployment) {
					if(	!status.equals("REGISTERED") && 
						!status.equals("ACTIVATED")  && 
						!status.equals("NOTVERIFIED") && 
						!status.equals("RUNNING") && 
						!status.equals("ALREADYDEPLOYED")) {
							Deploy.logError("\nFAILED - The Software package ("+serviceClass+"/"+serviceName+"/"+packageName+") has not been corretly deployed and activated (status is:\""+status+"\")\n");
							this.writeFile("FAILED\n");
							System.exit(1);
					}
					
				} else {
					if(!status.equals("UNDEPLOYED")) {
						if(!status.equals("NOTVERIFIED")) {
							if(!status.equals("NOTUNDEPLOYABLE")) {
								this.writeFile("FAILED\n");
								Deploy.logError("\nFAILED - The  package ("+serviceClass+"/"+serviceName+"/"+packageName+") has not been undeployed \n");
								System.exit(1);
							}
						}
					}
				}
			}	
		}

	}


	/**
	 * Check the deployment report 
	 * @param depResolution the Dependency resolution file
	 * @param isClient an instance of a QueryInformationSystem
	 * @return true/false
	 */
	public boolean checkDeploymentInfo(QueryInformationSystem isClient) throws Exception{

		NodeList nl = this.getResources();	
		if (nl == null){
			//the deployment is not yet finished we should wait 
			Deploy.logInfo("\nThe deploment is still ongoing\n");
			return false;
		}
		//if one of the resource failed the whole deployment failed
		if (!(parseResources(nl))) {
			//the deployment is not yet finished we should wait 
			Deploy.logInfo("\nThe deploment has failed\n");
			this.writeFile("FAILED\n");
			System.exit(1);
		}
		
		//getDeploymentPlanCreation
		if(!getDeploymentPlanCreation()){
			//the deployment is not yet finished we should wait 
			Deploy.logInfo("\nThe deploment has failed\n");
			this.writeFile("FAILED\n");
			System.exit(1);
		}


		NodeList ns = this.getServices();

		if (ns == null){
			//the deployment is not yet finished we should wait 
			Deploy.logInfo("\nThe deploment is still ongoing\n");
			return false;
		}

		//if one of the resource failed the dependenciesResolution the whole deployment failed
		if (!(parseDependenciesResolutionStatus(ns))) {
			//the deployment is not yet finished we should wait 
			Deploy.logInfo("\nThe deploment has failed beacuse one/more dependency failed to be resolved\n");
			this.writeFile("FAILED\n");
			System.exit(1);
		}


		if (this.getDeploymentActivityStatus()){
				checkPackagesDeployment(this.getPackagesDeployed(),true);

				Deploy.logInfo("\nSUCCESS - All packages correctly deployed or activated\n");
				return true;
			}else {
				Deploy.logInfo("\nThe deploment is still ongoing\n");
				return false;
			}
		
	}


	/**
	 * Check the undeployment report 
	 * @param isClient an instance of a QueryInformationSystem
	 * @return true/false
	 */
	public boolean checkUnDeploymentInfo(QueryInformationSystem isClient) {

		NodeList nl = this.getResources();	
		if (nl == null){
			//the deployment is not yet finished we should wait 
			Deploy.logInfo("\nThe undeploment is still ongoing\n");
			return false;
		}
		//if one of the resource failed the whole deployment failed
		if (!(parseResources(nl))) {
			//the deployment is not yet finished we should wait 
			Deploy.logInfo("\nThe undeploment has failed\n");
			this.writeFile("FAILED\n");
			System.exit(1);
		}


		NodeList ns = this.getServices();

		if (ns == null){
			//the deployment is not yet finished we should wait 
			Deploy.logInfo("\nThe undeploment is still ongoing\n");
			return false;
		}


		if (this.getDeploymentActivityStatus()){
			checkPackagesDeployment(this.getPackagesDeployed(),false);

			this.writeFile("SUCCESS\n");
			Deploy.logInfo("\nSUCCESS - All packages correctly undeployed\n");
			System.exit(0);
		}else {
			Deploy.logInfo("\nThe deploment is still ongoing\n");
			return false;
		}
	return true;

	}


	/**
	 * Check if all Files belonging to a Software package have been correctly deployed
	 * 
	 * @param files array of files name
	 * @return true/false
	 */
	public boolean checkSoftwareDeployment (ArrayList<String> files) {

		boolean isSoftPresent = true;
		if (files == null)
		{
			Deploy.logInfo("No deployment files to check");
			return true;
		}

		for(String file : files) {
			if ( new File(System.getenv("GLOBUS_LOCATION")+"/lib/"+file).exists())
				Deploy.logInfo(file + " file has been correctly deployed");
			else 
			{
				Deploy.logError(file + " IS MISSING!!");
				isSoftPresent = false;
			}
		}

		return isSoftPresent;
	}

}