package org.gcube.vremanagement.executor.pluginmanager;

import java.util.HashMap;
import java.util.Map;
import java.util.ServiceLoader;

import org.gcube.vremanagement.executor.plugin.PluginDeclaration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * This is a singleton class which discover on classpath the available plugins
 * and map the plugin name to its implementation class.
 * The plugin implementation class can be retrieved using its name.
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 */
public class PluginManager {

	/**
	 * Logger
	 */
	private static Logger logger = LoggerFactory.getLogger(PluginManager.class);
	
	/** 
	 * Singleton instance
	 */
	private static PluginManager pluginManager;
	
	/**
	 * Contains mapping between plugin name and the instance of its declaration
	 * class 
	 */
	private Map<String, PluginDeclaration> availablePlugins;
	
	/**
	 * Get the singleton instance of {@link #PluginManager}.
	 * The first time this function is invoked the instance is null
	 * so it is created. Otherwise the already created instance is returned
	 * @return singleton instance of {@link #PluginManager}
	 */
	public static PluginManager getInstance(){
		if(pluginManager== null){
			pluginManager = new PluginManager();
		}
		return pluginManager;
	}
	
	/**
	 * Used by {@link #getInstance()} function check the available plugin on classpath
	 * and add them on {@link #availablePlugins}
	 */
	protected PluginManager(){
		logger.debug("Loading plugins available on classpath");
		this.availablePlugins = new HashMap<String, PluginDeclaration>();
		ServiceLoader<PluginDeclaration> serviceLoader = ServiceLoader.load(PluginDeclaration.class);
		for (PluginDeclaration pluginDeclaration : serviceLoader) {
			try {
				logger.debug(String.format("%s plugin found", pluginDeclaration.getName()));
				pluginDeclaration.init();
				String name = pluginDeclaration.getName();
				this.availablePlugins.put(name, pluginDeclaration);
			} catch (Exception e) {
				logger.debug(String.format("%s not initialized correctly. It will not be used", pluginDeclaration.getName()));
			}
		}
	}
	
	/**
	 * 
	 * @param name The name of the plugin
	 * @return The plugin declaration if available, null otherwise
	 */
	public PluginDeclaration getPlugin(String name){
		return this.availablePlugins.get(name);
	}
	
	/**
	 * @return the availablePlugins
	 */
	public Map<String, PluginDeclaration> getAvailablePlugins() {
		return availablePlugins;
	}
}
