/**
 * 
 */
package org.gcube.vremanagement.executor.configuration.jsonbased;

import java.text.ParseException;
import java.util.HashMap;
import java.util.Map;

import org.gcube.common.scope.api.ScopeProvider;
import org.gcube.vremanagement.executor.api.types.LaunchParameter;
import org.gcube.vremanagement.executor.api.types.Scheduling;
import org.gcube.vremanagement.executor.exception.ScopeNotMatchException;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Luca Frosini (ISTI - CNR) http://www.lucafrosini.com/
 *
 */
public class JSONLaunchParameter extends LaunchParameter {
	
	public static final String PLUGIN_NAME = "pluginName";
	public static final String PLUGIN_CAPABILITIES = "pluginCapabilites";
	
	public static final String INPUTS = "inputs";
	
	public static final String SCHEDULING = "scheduling";
	public static final String PERSIST = "persist";
	
	public static final String USED_BY = "usedBy";
	public static final String SCOPE = "SCOPE";
	
	/**
	 * Contains the GCOREEndpoint (aka Running Instance) ID 
	 */
	protected String usedBy; 
	
	protected String scope;
	
	@SuppressWarnings("unused")
	private JSONLaunchParameter(){}
	
	public JSONLaunchParameter(String pluginName, Map<String, Object> inputs) {
		super(pluginName, inputs);
	}
	
	public JSONLaunchParameter(String pluginName, Map<String, String> pluginCapabilities, Map<String, Object> inputs) {
		super(pluginName, pluginCapabilities, inputs);
	}

	public JSONLaunchParameter(String pluginName, Map<String, Object> inputs, Scheduling scheduling) throws ParseException {
		super(pluginName, inputs, scheduling);
	}
	
	public JSONLaunchParameter(String pluginName, Map<String, String> pluginCapabilities, Map<String, Object> inputs, Scheduling scheduling) throws ParseException {
		super(pluginName, pluginCapabilities, inputs, scheduling);
	}
	
	public JSONLaunchParameter(String pluginName, Map<String, Object> inputs, Scheduling scheduling, boolean persist) throws ParseException {
		super(pluginName, inputs, scheduling, persist);
	}
	
	public JSONLaunchParameter(String pluginName, Map<String, String> pluginCapabilities, Map<String, Object> inputs, Scheduling scheduling, boolean persist) throws ParseException {
		super(pluginName, pluginCapabilities, inputs, scheduling, persist);
		this.scheduling = new JSONScheduling(scheduling);
		this.scope = ScopeProvider.instance.get();
	}
	
	public JSONLaunchParameter(LaunchParameter parameter) throws ParseException {
		super(parameter.getPluginName(), parameter.getPluginCapabilities(), parameter.getInputs(), parameter.getScheduling(), parameter.isPersist());
		this.scheduling = new JSONScheduling(parameter.getScheduling());
		this.scope = ScopeProvider.instance.get();
	}
	
	public JSONLaunchParameter(JSONObject jsonObject) throws JSONException, ParseException, ScopeNotMatchException {
		super();
		
		this.pluginName = jsonObject.getString(PLUGIN_NAME);
		
		this.pluginCapabilities = null;
		if(jsonObject.has(PLUGIN_CAPABILITIES)){
			this.pluginCapabilities = new HashMap<String, String>();
			JSONObject capabilities = jsonObject.getJSONObject(PLUGIN_CAPABILITIES);
			JSONArray names = capabilities.names();
			for(int j=0; j<names.length(); j++){
				String key = names.getString(j);
				this.pluginCapabilities.put(key, capabilities.getString(key));
			}
		}
		
		this.inputs = new HashMap<String, Object>();
		JSONObject inputsJsonObject = jsonObject.getJSONObject(INPUTS);
		JSONArray names = inputsJsonObject.names();
		for(int j=0; j<names.length(); j++){
			String key = names.getString(j);
			this.inputs.put(key, inputsJsonObject.get(key));
		}
		
		if(jsonObject.has(SCHEDULING)){
			JSONObject schedulingJsonObject = jsonObject.getJSONObject(SCHEDULING);
			this.scheduling = new JSONScheduling(schedulingJsonObject);
		}
		
		this.persist = true;
		if(jsonObject.has(PERSIST)){
			this.persist = jsonObject.getBoolean(PERSIST);
		}
		
		if(jsonObject.has(USED_BY)){
			this.usedBy = jsonObject.getString(USED_BY);
		}
		
		if(jsonObject.has(SCOPE)){
			this.scope = jsonObject.getString(SCOPE);
		}
		
	}

	/**
	 * @return the scheduling
	 */
	public JSONScheduling getScheduling() {
		return (JSONScheduling) scheduling;
	}

	/**
	 * @param scheduling the scheduling to set
	 */
	public void setScheduling(JSONScheduling scheduling) {
		this.scheduling = scheduling;
	}
	
	public JSONObject toJSON() throws JSONException {
		JSONObject obj = new JSONObject();
		obj.put(PLUGIN_NAME, pluginName);
		
		if(pluginCapabilities!=null && !pluginCapabilities.isEmpty()){
			JSONObject capabilities = new JSONObject();
			for(String id : pluginCapabilities.keySet()){
				capabilities.put(id, pluginCapabilities.get(id));
			}
			obj.put(PLUGIN_CAPABILITIES, capabilities);
		}
		
		JSONObject inputJsonObject = new JSONObject();
		for(String id : inputs.keySet()){
			inputJsonObject.put(id, inputs.get(id));
		}
		obj.put(INPUTS, inputJsonObject);
		
		if(scheduling!=null){
			obj.put(SCHEDULING, getScheduling().toJSON());
		}
		
		obj.put(PERSIST, true);
		
		if(usedBy!=null){
			obj.put(USED_BY, usedBy);
		}
		
		return obj;
	}
	
	public String toString(){
		try {
			return toJSON().toString();
		} catch (JSONException e) {
			return String.format("{} : {}", this.getClass().getSimpleName(), 
					super.toString());
		}
	}
	
}
