/**
 * Sencha GXT 3.1.1 - Sencha for GWT
 * Copyright(c) 2007-2014, Sencha, Inc.
 * licensing@sencha.com
 *
 * http://www.sencha.com/products/gxt/license/
 */
package com.sencha.gxt.chart.client.draw;

/**
 * Represents an HSV color.
 */
public class HSV extends Hue {

  private double value = 0;

  /**
   * Creates an instance of HSV with default values.
   */
  public HSV() {
  }

  /**
   * Creates an instance of HSV with the given hue, saturation and value
   * 
   * @param hue the hue value
   * @param saturation the saturation value
   * @param value the value value
   */
  public HSV(double hue, double saturation, double value) {
    setHue(hue);
    setSaturation(saturation);
    setValue(value);
  }

  /**
   * Creates an instance of HSV with the given RGB color.
   * 
   * @param rgb the RGB color
   */
  public HSV(RGB rgb) {
    double r = rgb.getRed();
    double g = rgb.getGreen();
    double b = rgb.getBlue();
    double min = Math.min(r, Math.min(g, b));
    double max = Math.max(r, Math.max(g, b));
    setValue(max / 255.0);

    // min == max means achromatic (hue is undefined, saturation is zero)
    if (min != max) {
      double delta = max - min;
      setSaturation(delta / max);
      if (r == max) {
        setHue(60.0 * (g - b) / delta);
      } else if (g == max) {
        setHue(120.0 + 60.0 * (b - r) / delta);
      } else {
        assert b == max : "No color matched max value, shouldn't be possible";
        setHue(240.0 + 60.0 * (r - g) / delta);
      }
    }
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) return true;
    if (obj == null) return false;
    if (!(obj instanceof HSV)) return false;
    HSV other = (HSV) obj;
    //hue may not match and still be equal if saturation is zero or value is zero
    if (getHue() != other.getHue() && (getSaturation() != 0 && getValue() != 0)) return false;
    if (getSaturation() != other.getSaturation()) return false;
    if (getValue() != other.getValue()) return false;
    return true;
  }

  @Override
  public String getColor() {
    if (color == null) {
      color = (new RGB(this)).getColor();
    }
    return color;
  }

  /**
   * Returns the value.
   * 
   * @return the value
   */
  public double getValue() {
    return value;
  }

  /**
   * Sets the value, a double value between 0.0 and 1.0.
   * 
   * @param value the value
   */
  public void setValue(double value) {
    this.value = Math.min(1.0, Math.max(0.0, value));
    color = null;
  }

  @Override
  public String toString() {
    return new StringBuilder().append("hsv(").append(getHue()).append(", ").append(getSaturation()).append(", ").append(value).append(
        ")").toString();
  }

}
