package org.finconsgroup.itserr.criterion.common.exception;

import lombok.extern.slf4j.Slf4j;
import org.finconsgroup.itserr.criterion.common.dto.ErrorResponseDto;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.FieldError;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.context.request.WebRequest;

import java.util.HashMap;
import java.util.Map;

/**
 * Base exception handler providing common validation error handling.
 * <p>
 * This abstract class can be extended by microservice-specific exception handlers
 * to inherit standard validation error handling behavior.
 * </p>
 *
 * @author Fincons Group
 * @version 1.0
 * @since 1.0
 */
@Slf4j
public abstract class BaseExceptionHandler {

    /**
     * Handles validation errors from @Valid annotations on request bodies.
     *
     * @param ex      the MethodArgumentNotValidException containing validation errors
     * @param request the web request where the exception occurred
     * @return ResponseEntity with ErrorResponseDto containing all validation errors
     */
    @ExceptionHandler(MethodArgumentNotValidException.class)
    public ResponseEntity<ErrorResponseDto> handleValidationExceptions(
            MethodArgumentNotValidException ex, WebRequest request) {

        Map<String, String> errorMessages = new HashMap<>();

        ex.getBindingResult().getAllErrors().forEach((error) -> {
            String fieldName = ((FieldError) error).getField();
            String errorMessage = error.getDefaultMessage();
            errorMessages.put(fieldName, errorMessage);
        });

        String path = extractPath(request);
        log.error("Validation errors: {} | Path: {}", errorMessages, path);

        ErrorResponseDto errorResponse = ErrorResponseDto.builder()
                .errorCode(HttpStatus.BAD_REQUEST.value())
                .errorMessages(errorMessages)
                .path(path)
                .build();

        return ResponseEntity.status(HttpStatus.BAD_REQUEST).body(errorResponse);
    }

    /**
     * Handles generic exceptions that are not caught by specific handlers.
     *
     * @param ex      the generic Exception
     * @param request the web request where the exception occurred
     * @return ResponseEntity with ErrorResponseDto containing generic error message
     */
    @ExceptionHandler(Exception.class)
    public ResponseEntity<ErrorResponseDto> handleGenericException(
            Exception ex, WebRequest request) {

        String path = extractPath(request);
        log.error("Unexpected error: {} | Path: {}", ex.getMessage(), path, ex);

        ErrorResponseDto errorResponse = ErrorResponseDto.builder()
                .errorCode(HttpStatus.INTERNAL_SERVER_ERROR.value())
                .errorMessages(Map.of("error", "An unexpected error occurred. Please contact support."))
                .path(path)
                .build();

        return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body(errorResponse);
    }

    /**
     * Handles UnauthorizedException from security library.
     *
     * @param ex      the UnauthorizedException
     * @param request the web request where the exception occurred
     * @return ResponseEntity with ErrorResponseDto containing unauthorized error
     */
    @ExceptionHandler(UnauthorizedException.class)
    public ResponseEntity<ErrorResponseDto> handleUnauthorizedException(
            UnauthorizedException ex,
            WebRequest request) {

        String path = extractPath(request);
        log.warn("UnauthorizedException: {} | Path: {}", ex.getMessage(), path);

        ErrorResponseDto errorResponse = ErrorResponseDto.builder()
                .errorCode(HttpStatus.UNAUTHORIZED.value())
                .errorMessages(Map.of("error", ex.getMessage()))
                .path(path)
                .build();

        return ResponseEntity.status(HttpStatus.UNAUTHORIZED).body(errorResponse);
    }

    /**
     * Extracts the request path from WebRequest.
     *
     * @param request the web request
     * @return the request URI path
     */
    protected String extractPath(WebRequest request) {
        String description = request.getDescription(false);
        if (description != null && description.startsWith("uri=")) {
            return description.substring(4);
        }
        return description;
    }

    @ExceptionHandler(UserNotFoundException.class)
    public ResponseEntity<ErrorResponseDto> handleUserNotFound(
            UserNotFoundException ex, WebRequest request) {

        String path = extractPath(request);
        log.error("UserNotFoundException: {} | Path: {}", ex.getMessage(), path);

        ErrorResponseDto errorResponse = ErrorResponseDto.builder()
                .errorCode(HttpStatus.NOT_FOUND.value())
                .errorMessages(Map.of("error", ex.getMessage()))
                .path(path)
                .build();

        return ResponseEntity.status(HttpStatus.NOT_FOUND).body(errorResponse);
    }
}