package com.finconsgroup.itserr.marketplace.audit.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.SuperBuilder;
import org.hibernate.annotations.JdbcTypeCode;
import org.hibernate.type.SqlTypes;

import java.time.Instant;
import java.util.Map;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.audit.dm.util.DomainConstants.LONG_TEXT_LENGTH;
import static com.finconsgroup.itserr.marketplace.audit.dm.util.DomainConstants.MEDIUM_TEXT_LENGTH;
import static com.finconsgroup.itserr.marketplace.audit.dm.util.DomainConstants.SHORT_TEXT_LENGTH;

/**
 * Entity class representing an audit log in the audit system.
 * This entity is mapped to the "audit_log" table in the database and stores
 * information such as the ID, event source, event type, user, resource.
 *
 */
@SuppressWarnings("DefaultAnnotationParam")
@Entity
@Table(name = "audit_log")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class AuditLogEntity extends AbstractUUIDEntity {

    /**
     * The source of the event, usually this represents the microservice which generated the event.
     */
    @Column(name = "event_source", nullable = false, length = SHORT_TEXT_LENGTH)
    private String eventSource;

    /**
     * The type of the event.
     */
    @Column(name = "event_type", nullable = false, length = SHORT_TEXT_LENGTH)
    private String eventType;

    /**
     * The identifier for the user from User Profile whose action triggered the event.
     */
    @Column(name = "user_id")
    private UUID userId;

    /**
     * The identifier for the user from User Profile whose action triggered the event.
     */
    @Column(name = "user_username", length = SHORT_TEXT_LENGTH)
    private String userUsername;

    /**
     * The display name of the user whose action triggered the event.
     */
    @Column(name = "user_name", length = MEDIUM_TEXT_LENGTH)
    private String userName;

    /**
     * The identifier of the resource on which the event was triggered.
     */
    @Column(name = "resource_id", nullable = false, length = SHORT_TEXT_LENGTH)
    private String resourceId;

    /**
     * The name of the resource on which the event was triggered.
     */
    @Column(name = "resource_name", length = MEDIUM_TEXT_LENGTH)
    private String resourceName;

    /**
     * The name of the resource on which the event was triggered.
     */
    @Column(name = "resource_title", length = MEDIUM_TEXT_LENGTH)
    private String resourceTitle;

    /**
     * The category of the resource on which the event was triggered.
     */
    @Column(name = "resource_category", length = MEDIUM_TEXT_LENGTH)
    private String resourceCategory;

    /**
     * The status of the resource on which the event was triggered.
     */
    @Column(name = "resource_status", length = SHORT_TEXT_LENGTH)
    private String resourceStatus;

    /**
     * The message indicating the reason for an action.
     */
    @Column(name = "event_message", length = LONG_TEXT_LENGTH)
    private String eventMessage;

    /**
     * JSON structure for the additional data that was provided on the event by the source microservice.
     */
    @JdbcTypeCode(SqlTypes.JSON)
    @Column(name = "event_additional_data", columnDefinition = "json")
    private Map<String, Object> eventAdditionalData;

    /**
     * The time at which the event action was actually triggered by the user.
     */
    @Column(name = "event_action_time", nullable = false)
    private Instant eventActionTime;

    /**
     * The time at which the event was generated by the source microservice.
     */
    @Column(name = "event_creation_time", nullable = false)
    private Instant eventCreationTime;

    /**
     * The timestamp when the audit log entry was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
    }

    /**
     * @throws UnsupportedOperationException as update is not supported
     */
    @PreUpdate
    public void onUpdate() {
        throw new UnsupportedOperationException("Update not supported");
    }
}
