package com.finconsgroup.itserr.marketplace.audit.dm.api;

import com.finconsgroup.itserr.marketplace.audit.dm.dto.InputCreateAuditLogDto;
import com.finconsgroup.itserr.marketplace.audit.dm.dto.OutputAuditLogDto;
import com.finconsgroup.itserr.marketplace.audit.dm.validation.ValidInputCreateAuditLogDto;
import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related to Audit Log Management.
 *
 * <p>
 * It provides endpoints for retrieving and creating audit logs.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * GET  audit/audit-logs - Retrieve a paginated list of all audit logs.
 * GET  audit/audit-logs/{id} - Retrieve a audit by id.
 * POST audit/audit-logs - Create a new audit.
 * </pre>
 * </p>
 */
@Tag(
        name = "AuditLogApi",
        description = "The AuditLogApi API: it provides endpoints for retrieving and creating audit logs."
)
@SecurityRequirement(name = "BearerAuth")
public interface AuditLogApi {

    /**
     * Retrieves a paginated list of all audit logs.
     *
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of UserProfiles per page (default is 10)
     * @param sort       the field to sort by (default is "startDate")
     * @param direction  the direction of sorting (default is descending)
     * @return a page of {@link OutputAuditLogDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find all the audit log",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/audit/audit-logs", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    Page<OutputAuditLogDto> findAll(
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "id", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "DESC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves audit log by id.
     *
     * @param id the id of the audit log to retrieve
     * @return the found {@link OutputAuditLogDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find audit log by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            }
    )
    @GetMapping(value = "/audit/audit-logs/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputAuditLogDto findById(@PathVariable("id") UUID id);

    /**
     * Creates new audit log.
     *
     * @param inputCreateAuditLogDto the input data transfer object containing audit log details
     */
    @Operation(
            summary = "create an audit log",
            responses = {@ApiResponse(responseCode = "201", description = "Created")}
    )
    @PostMapping(
            value = "/audit/audit-logs",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    OutputAuditLogDto createAuditLog(@Valid @ValidInputCreateAuditLogDto @RequestBody InputCreateAuditLogDto inputCreateAuditLogDto);

}
