package com.finconsgroup.itserr.marketplace.discussion.bs.api;

import com.finconsgroup.itserr.marketplace.discussion.bs.dto.DiscussionDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.InputUpdateDiscussionDto;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.ThreadDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.enums.ReactionType;
import com.finconsgroup.itserr.marketplace.discussion.bs.enums.ResourceType;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.parameters.RequestBody;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.time.Instant;

/**
 * REST API contract for managing discussions, their threads, and reactions.
 *
 * <p>
 * This interface defines the HTTP contract (endpoints, parameters, and response
 * types) and is decorated with OpenAPI annotations to generate documentation.
 * It is implemented by the {@code DiscussionController}, which delegates all
 * business operations to the {@code DiscussionService}.
 * </p>
 */
@Tag(name = "Discussion Management", description = "API for managing discussions, threads, and reactions in the ITSERR marketplace")
@SecurityRequirement(name = "BearerAuth")
public interface DiscussionApi {

    @Operation(summary = "Get paginated discussions", description = "Retrieves a paginated list of discussions with sorting and filtering options")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Discussions retrieved successfully",
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = Page.class))),
            @ApiResponse(responseCode = "400", description = "Invalid pagination or sorting parameters"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @GetMapping("/discussions")
    Page<DiscussionDTO> getDiscussions(
            @Parameter(description = "Page number (0-based)", example = "0")
            @RequestParam(defaultValue = "0") int page,
            @Parameter(description = "Number of items per page", example = "10")
            @RequestParam(defaultValue = "10") int size,
            @Parameter(description = "Field to sort by", example = "createdAt")
            @RequestParam(defaultValue = "createdAt") String sortBy,
            @Parameter(description = "Sort direction (asc or desc)", example = "DESC")
            @RequestParam(defaultValue = "DESC") String direction,
            @Parameter(description = "Filter by visibility (optional)", example = "public")
            @RequestParam(required = false) String visibility,
            @Parameter(description = "Include threads in response (default: true)", example = "true")
            @RequestParam(defaultValue = "true") boolean includeThreads,
            @Parameter(description = "Filter by resource type (optional)", example = "news")
            @RequestParam(required = false) String resourceType,
            @Parameter(description = "Filter by created after (optional)", example = "2025-08-25T11:22:33.123456Z")
            @RequestParam(required = false) Instant createdAfter);



    @Operation(summary = "Create new discussion", description = "Creates a new discussion with the provided data")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "201", description = "Discussion created successfully"),
            @ApiResponse(responseCode = "400", description = "Invalid discussion data"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @PostMapping("/discussion")
    @ResponseStatus(HttpStatus.CREATED)
    DiscussionDTO createDiscussion(
            @RequestBody(description = "Discussion data to create", required = true,
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = DiscussionDTO.class)))
            @org.springframework.web.bind.annotation.RequestBody DiscussionDTO discussionDTO);


    @Operation(summary = "Update discussion", description = "Updates an existing discussion with new details")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Discussion updated successfully",
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = DiscussionDTO.class))),
            @ApiResponse(responseCode = "400", description = "Invalid discussion data"),
            @ApiResponse(responseCode = "404", description = "Discussion not found"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @PatchMapping("/discussion/{id}")
    ResponseEntity<DiscussionDTO> updateDiscussion(
            @Parameter(description = "The unique identifier of the discussion to update", required = true) @PathVariable String id,
            @RequestBody(description = "Updated discussion data", required = true,
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = InputUpdateDiscussionDto.class)))
            @org.springframework.web.bind.annotation.RequestBody InputUpdateDiscussionDto discussionDto);


    @Operation(summary = "Delete discussion", description = "Deletes a discussion by its unique identifier")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "204", description = "Discussion deleted successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion not found"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @DeleteMapping("/discussion/{id}")
    ResponseEntity<Void> deleteDiscussion(@Parameter(description = "The unique identifier of the discussion to delete", required = true)
                                          @PathVariable String id);
    /**
     * Retrieves a discussion by its unique identifier.
     *
     * @param id the unique identifier of the discussion
     * @return the {@link DiscussionDTO} representing the discussion
     */
    @Operation(summary = "Get discussion by ID", description = "Retrieves a discussion by its unique identifier")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Discussion found successfully",
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = DiscussionDTO.class))),
            @ApiResponse(responseCode = "404", description = "Discussion not found"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @GetMapping("/discussion/{id}")
    DiscussionDTO getDiscussionById(
            @Parameter(description = "The unique identifier of the discussion", required = true)
            @PathVariable String id);

    @GetMapping("/discussion/{discussionId}/threads")
    Page<ThreadDTO> getThreadsByDiscussionId(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "Page number (0-based)", example = "0") @RequestParam(defaultValue = "0") int page,
            @Parameter(description = "Number of items per page", example = "10") @RequestParam(defaultValue = "10") int size,
            @Parameter(description = "Field to sort by", example = "createdAt") @RequestParam(defaultValue = "createdAt") String sortBy,
            @Parameter(description = "Sort direction (ASC or DESC)", example = "DESC") @RequestParam(defaultValue = "DESC") Sort.Direction direction);


    @Operation(summary = "Add thread to discussion", description = "Adds a new thread to an existing discussion")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "201", description = "Thread added successfully"),
            @ApiResponse(responseCode = "400", description = "Invalid thread data"),
            @ApiResponse(responseCode = "404", description = "Discussion not found"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @PostMapping("/discussion/{discussionId}/threads")
    ResponseEntity<ThreadDTO> addThread(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @RequestBody(description = "Thread data to add", required = true,
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = ThreadDTO.class)))
            @org.springframework.web.bind.annotation.RequestBody ThreadDTO threadDTO);


    @Operation(summary = "Delete thread from discussion", description = "Deletes a specific thread from a discussion")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "204", description = "Thread deleted successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion or thread not found"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @DeleteMapping("/discussion/{discussionId}/threads/{threadId}")
    ResponseEntity<Void> deleteThread(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "The unique identifier of the thread to delete", required = true) @PathVariable String threadId);



    @Operation(summary = "Update thread in discussion", description = "Updates a specific thread within a discussion")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "204", description = "Thread updated successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion or thread not found"),
            @ApiResponse(responseCode = "400", description = "Invalid thread data"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @org.springframework.web.bind.annotation.PutMapping("/discussion/{discussionId}/threads/{threadId}")
    ResponseEntity<ThreadDTO> updateThread(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "The unique identifier of the thread to update", required = true) @PathVariable String threadId,
            @RequestBody(description = "Thread data to update", required = true,
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = ThreadDTO.class)))
            @org.springframework.web.bind.annotation.RequestBody ThreadDTO threadDTO);


    @Operation(summary = "Add reaction to discussion", description = "Adds a reaction to a discussion by a user")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "201", description = "Reaction added successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion not found"),
            @ApiResponse(responseCode = "409", description = "Reaction already exists"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @PostMapping("/discussion/{discussionId}/reactions/{userId}")
    ResponseEntity<Void> addReactionToDiscussion(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "The unique identifier of the user", required = true) @PathVariable String userId,
            @Parameter(description = "The type of reaction to add", required = true) @RequestParam ReactionType reactionType);



    @Operation(summary = "Remove reaction from discussion", description = "Removes a reaction from a discussion by a user")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "204", description = "Reaction removed successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion or reaction not found"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @DeleteMapping("/discussion/{discussionId}/reactions/{userId}")
    ResponseEntity<Void> removeReactionFromDiscussion(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "The unique identifier of the user", required = true) @PathVariable String userId);



    @Operation(summary = "Add reaction to thread", description = "Adds a reaction to a specific thread within a discussion")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "201", description = "Reaction added successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion or thread not found"),
            @ApiResponse(responseCode = "409", description = "Reaction already exists"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @PostMapping("/discussion/{discussionId}/threads/{threadId}/reactions/{userId}")
    ResponseEntity<Void> addReactionToThread(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "The unique identifier of the thread", required = true) @PathVariable String threadId,
            @Parameter(description = "The unique identifier of the user", required = true) @PathVariable String userId,
            @Parameter(description = "The type of reaction to add", required = true) @RequestParam ReactionType reactionType);


    @Operation(summary = "Remove reaction from thread", description = "Removes a reaction from a specific thread within a discussion")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "204", description = "Reaction removed successfully"),
            @ApiResponse(responseCode = "404", description = "Discussion, thread, or reaction not found"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @DeleteMapping("/discussion/{discussionId}/threads/{threadId}/reactions/{userId}")
    ResponseEntity<Void> removeReactionFromThread(
            @Parameter(description = "The unique identifier of the discussion", required = true) @PathVariable String discussionId,
            @Parameter(description = "The unique identifier of the thread", required = true) @PathVariable String threadId,
            @Parameter(description = "The unique identifier of the user", required = true) @PathVariable String userId);


    @Operation(summary = "Get discussion by resource ID and type",
            description = "Retrieves a discussion associated with the specified resource identifier and type")
    @ApiResponses(value = {
            @ApiResponse(responseCode = "200", description = "Discussion found successfully",
                    content = @Content(mediaType = "application/json", schema = @Schema(implementation = DiscussionDTO.class))),
            @ApiResponse(responseCode = "404", description = "Discussion not found"),
            @ApiResponse(responseCode = "401", description = "Unauthorized"),
            @ApiResponse(responseCode = "403", description = "Forbidden")
    })
    @GetMapping("/discussion/resource/{resourceId}")
    DiscussionDTO getDiscussionForResource(
            @Parameter(description = "The unique identifier of the resource", required = true)
            @PathVariable String resourceId,
            @Parameter(description = "The type of the resource", in = ParameterIn.QUERY, required = true)
            @RequestParam(name = "resourceType") ResourceType resourceType);

}
