package com.finconsgroup.itserr.marketplace.discussion.bs.component;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.discussion.bs.bean.DiscussionApplicationEvent;
import com.finconsgroup.itserr.marketplace.discussion.bs.dto.DiscussionDTO;
import com.finconsgroup.itserr.marketplace.discussion.bs.mapper.EventToDiscussionMapper;
import com.finconsgroup.itserr.marketplace.discussion.bs.messaging.dto.DiscussionMessagingAdditionalDataDto;
import com.finconsgroup.itserr.marketplace.discussion.bs.messaging.producer.EventProducer;
import com.finconsgroup.itserr.marketplace.discussion.bs.messaging.producer.ResourceProducer;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventUserDto;
import lombok.NonNull;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.context.ApplicationListener;
import org.springframework.stereotype.Component;

/**
 * The implementation for {@link ApplicationListener} to listen to Discussion application events
 * and produce relevant notification messages.
 */
@Component
@Slf4j
@RequiredArgsConstructor
public class MessagingDiscussionApplicationEventListener implements ApplicationListener<DiscussionApplicationEvent> {

    private final ResourceProducer resourceProducer;
    private final EventProducer eventProducer;
    private final EventToDiscussionMapper eventToDiscussionMapper;

    @Override
    public void onApplicationEvent(@NonNull DiscussionApplicationEvent event) {
        MessagingEventDto<DiscussionMessagingAdditionalDataDto> messagingEventDto =
                eventToDiscussionMapper.toMessagingEventDto(event.getDiscussion());

        // Ensure user information is populated
        MessagingEventUserDto user = MessagingEventUserDto.builder()
                .id(JwtTokenHolder.getUserId().orElse(null))
                .name(JwtTokenHolder.getName().orElse(null))
                .username(JwtTokenHolder.getPreferredUsername().orElse(null))
                .build();
        messagingEventDto.setUser(user);

        DiscussionMessagingAdditionalDataDto additionalData = getDiscussionMessagingAdditionalDataDto(event);
        messagingEventDto.setAdditionalData(additionalData);

        updateEventForType(event, messagingEventDto);
        // Publish the event based on type
        publishEventForType(event, messagingEventDto);
    }

    private void updateEventForType(DiscussionApplicationEvent applicationEvent, MessagingEventDto<DiscussionMessagingAdditionalDataDto> messagingEventDto) {
        DiscussionDTO discussionDTO = applicationEvent.getDiscussion();
        DiscussionMessagingAdditionalDataDto additionalData = getDiscussionMessagingAdditionalDataDto(applicationEvent);

        switch (applicationEvent.getEventType()) {
            case CREATED -> {
                messagingEventDto.setTimestamp(discussionDTO.getCreatedAt());
            }
            case UPDATED, DELETED -> messagingEventDto.setTimestamp(applicationEvent.getEventTimestamp());
        }

        messagingEventDto.setAdditionalData(additionalData);
    }

    @NotNull
    private static DiscussionMessagingAdditionalDataDto getDiscussionMessagingAdditionalDataDto(@NotNull DiscussionApplicationEvent event) {
        DiscussionDTO discussionDTO = event.getDiscussion();

        DiscussionMessagingAdditionalDataDto additionalData = new DiscussionMessagingAdditionalDataDto();
        additionalData.setContent(discussionDTO.getContent());
        additionalData.setVisibility(discussionDTO.getVisibility());
        additionalData.setImageUrl(discussionDTO.getUrl());
        return additionalData;
    }

    private void publishEventForType(DiscussionApplicationEvent applicationEvent,
                                     MessagingEventDto<DiscussionMessagingAdditionalDataDto> messagingEventDto) {
        switch (applicationEvent.getEventType()) {
            case CREATED -> {
                resourceProducer.publishCreatedResource(applicationEvent.getDiscussion());
                eventProducer.publishCreatedEvent(messagingEventDto);
            }
            case UPDATED -> {
                resourceProducer.publishUpdatedResource(applicationEvent.getDiscussion());
                eventProducer.publishUpdatedEvent(messagingEventDto);
            }
            case DELETED -> {
                resourceProducer.publishDeletedResource(applicationEvent.getDiscussion());
                eventProducer.publishDeletedEvent(messagingEventDto);
            }
        }
    }
}