package com.finconsgroup.itserr.marketplace.discussion.dm.entity;

import com.finconsgroup.itserr.marketplace.discussion.dm.enums.ReactionType;
import jakarta.persistence.*;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.NoArgsConstructor;

import java.time.Instant;
import java.util.UUID;

@Entity
@Table(name = "thread_reaction")
@AllArgsConstructor
@NoArgsConstructor
@Builder
@Data
public class ThreadReaction {
    /**
     * Represents the universally unique identifier (UUID) for the {@code ThreadReaction} entity.
     * This value is automatically generated using the UUID generation strategy
     * and serves as the primary key for uniquely identifying a thread reaction
     * record in the database.
     */
    @Id
    @GeneratedValue(strategy = GenerationType.UUID)
    private UUID id;

    /**
     * Represents the association between the {@code ThreadReaction} entity and the {@code Thread} entity.
     * This relationship is defined as a many-to-one association, where multiple {@code ThreadReaction} entities
     * can be associated with a single {@code Thread}.
     *
     * Key details about the relationship:
     * - The foreign key column is named {@code thread_id}.
     * - The {@code nullable = false} constraint ensures that a reaction must be associated with a thread.
     * - The {@code optional = false} property enforces that a {@code Thread} must be provided for every {@code ThreadReaction}.
     */
    @ManyToOne(optional = false)
    @JoinColumn(name = "thread_id", nullable = false)
    private Thread thread;

    /**
     * Represents the identifier of the user associated with a specific thread reaction.
     * This value is a foreign key referencing the unique identifier of the user entity.
     *
     * Key details:
     * - The value is stored in the "user_id" column in the database.
     * - Cannot be null, ensuring that every thread reaction is linked to a user.
     */
    @Column(name = "user_id", nullable = false)
    private String userId;

    /**
     * Represents the user reaction to a specific thread.
     * This field is used to capture feedback or interactions (LIKE, CELEBRATE, LOVE, INSIGHTFUL, CURIOUS)
     * expressed by users on a thread.
     *
     * The value is stored as an enumerated type in the database and is marked as non-nullable,
     * ensuring that every reaction must have an associated value.
     */
    @Enumerated(EnumType.STRING)
    @Column(nullable = false)
    private ReactionType reaction;

    /**
     * The date and time when the entity was created.
     * This field stores the timestamp for the creation
     * of a record in the system.
     */
    private Instant createdAt;

    /**
     * Represents the timestamp indicating when the {@code ThreadReaction} entity
     * was created. This field is automatically initialized with the current date
     * and time when the entity is instantiated, ensuring a record of creation.
     */

    @PrePersist
    public void onCreate() {
        createdAt = Instant.now();
    }


}
