package com.finconsgroup.itserr.marketplace.event.bs.api;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputCreateEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputUpdateEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.OutputEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.validation.annotation.ValidAssociationToLoad;
import com.finconsgroup.itserr.marketplace.event.bs.validation.annotation.ValidInputCreateOrUpdateEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputProgramSubscribedParticipantDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.Set;
import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints related to Event Management.
 *
 * <p>
 * It provides endpoints for retrieving, creating, updating, and deleting event.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * GET  event/event - Retrieve a paginated list of all event.
 * GET  event/event/{id} - Retrieve a event by id.
 * POST event/event - Create a new event.
 * PUT  event/event/{id} - Update an existing event.
 * DELETE event/event/{id} - Delete a event.
 * </pre>
 * </p>
 */
@Tag(
        name = "EventApi",
        description = "The EventApi API: it provides endpoints for " +
                "retrieving, creating, updating, and deleting event."
)
@SecurityRequirement(name = "BearerAuth")
public interface EventApi {

    /**
     * Retrieves a paginated list of all event.
     *
     * @param associationsToLoad the associations to load (default is "", indicating none)
     * @param pageNumber         the page number to retrieve (default is 0)
     * @param pageSize           the number of UserProfiles per page (default is 10)
     * @param sort               the field to sort by (default is "startDate")
     * @param direction          the direction of sorting (default is descending)
     * @return a page of {@link OutputEventDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find all the event",
            responses = {@ApiResponse(responseCode = "200", description = "OK")}
    )
    @GetMapping(value = "/event/events", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputPageDto<OutputEventDto> findAll(
            @RequestParam(name = "associationsToLoad", defaultValue = "", required = false) Set<@ValidAssociationToLoad String> associationsToLoad,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "startDate", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "DESC", required = false) SortDirection direction
    );

    /**
     * Retrieves event by id.
     *
     * @param id the id of the event to retrieve
     * @return the found {@link OutputEventDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find event by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found",
                            content = {
                                    @Content(
                                            mediaType = MediaType.APPLICATION_JSON_VALUE,
                                            schema = @Schema(implementation = ErrorResponseDto.class)
                                    )
                            }),
            }
    )
    @GetMapping(value = "/event/events/{id}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputEventDto findById(@PathVariable("id") UUID id);

    /**
     * Creates new event.
     *
     * @param inputCreateEventDto the input data transfer object containing event details
     */
    @Operation(
            summary = "create a event",
            responses = {@ApiResponse(responseCode = "201", description = "Created")}
    )
    @PostMapping(
            value = "/event/events",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    OutputEventDto createEvent(@Valid @ValidInputCreateOrUpdateEventDto @RequestBody InputCreateEventDto inputCreateEventDto);

    /**
     * Updates existing event by id.
     *
     * @param id                  the id of the event to update
     * @param inputUpdateEventDto the input data transfer object containing updated event details
     * @return the updated {@link OutputEventDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "updates event by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @PutMapping(
            value = "/event/events/{id}",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputEventDto updateEventById(
            @PathVariable("id") UUID id,
            @Valid @ValidInputCreateOrUpdateEventDto @RequestBody InputUpdateEventDto inputUpdateEventDto
    );

    /**
     * Deletes event by id.
     *
     * @param id the id of the event to delete
     */
    @Operation(
            summary = "delete a event by id",
            responses = {
                    @ApiResponse(responseCode = "204", description = "No Content"),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @ResponseStatus(HttpStatus.NO_CONTENT)
    @DeleteMapping(value = "/event/events/{id}")
    void deleteEventById(@PathVariable("id") UUID id);

    /**
     * Register a user to an event.
     */
    @Operation(
            summary = "register a user to an event",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Success"),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
                    @ApiResponse(responseCode = "409", description = "Conflict", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    })
            }
    )
    @PostMapping(
            value = "/event/events/{id}/register",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputEventDto register(@PathVariable("id") UUID id);

    /**
     * Unregister a user to an event.
     */
    @Operation(
            summary = "unregister a user to an event",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Success"),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @DeleteMapping(
            value = "/event/events/{id}/register",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputEventDto unregister(@PathVariable("id") UUID id);

    /**
     * Register a user to a program.
     */
    @Operation(
            summary = "register a user to a program",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Success"),
                    @ApiResponse(responseCode = "400", description = "Bad Request", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
                    @ApiResponse(responseCode = "409", description = "Conflict", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    })
            }
    )
    @PostMapping(
            value = "/event/events/{eventId}/programs/{programId}/register",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputEventDto registerProgram(@PathVariable("eventId") UUID eventId,
                                   @PathVariable("programId") UUID programId,
                                   @Valid @RequestBody InputProgramSubscribedParticipantDto inputProgramSubscribedParticipantDto);

    /**
     * Unregister a user to a program.
     */
    @Operation(
            summary = "unregister a user to a program",
            responses = {
                    @ApiResponse(responseCode = "200", description = "Success"),
                    @ApiResponse(responseCode = "404", description = "Not Found", content = {
                            @Content(
                                    mediaType = MediaType.APPLICATION_JSON_VALUE,
                                    schema = @Schema(implementation = ErrorResponseDto.class)
                            )
                    }),
            }
    )
    @DeleteMapping(
            value = "/event/events/{eventId}/programs/{programId}/register",
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.OK)
    OutputEventDto unregisterProgram(@PathVariable("eventId") UUID eventId, @PathVariable("programId") UUID programId);
}
