package com.finconsgroup.itserr.marketplace.event.bs.service;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.dto.PageRequestDto;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2DuplicateResourceException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ValidationException;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputCreateEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputProgramSubscribedParticipantDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.InputUpdateEventDto;
import com.finconsgroup.itserr.marketplace.event.bs.dto.OutputEventDto;
import org.springframework.lang.NonNull;

import java.util.Set;
import java.util.UUID;

/**
 * Service for handling business logic related to Event. This interface provides methods to retrieve, create, update, and delete Event,
 *
 * <p>Example usage:
 * <pre>
 * Page<OutputEventDto> Event = EventService.findAll(pageable);
 * </pre>
 * </p>
 */
public interface EventService {

    /**
     * Retrieves a paginated list of all Event.
     *
     * @param associationsToLoad the associations to be returned on the response
     * @param pageRequest        the pagination information
     * @return a page of OutputEventDto representing the Event
     */
    @NonNull
    OutputPageDto<OutputEventDto> findAll(Set<String> associationsToLoad, @NonNull PageRequestDto pageRequest);

    /**
     * Retrieves event by id.
     *
     * @param eventId the id of the event
     * @return an OutputEventDto representing the event
     * @throws WP2ResourceNotFoundException if no event is found with the given id
     */
    @NonNull
    OutputEventDto findById(@NonNull UUID eventId);

    /**
     * Creates new event for user using the provided input data.
     *
     * @param inputCreateEventDto the input DTO containing event details
     */
    @NonNull
    OutputEventDto create(@NonNull InputCreateEventDto inputCreateEventDto);

    /**
     * Updates existing event by id with the provided data.
     *
     * @param eventId             the id of the event to update
     * @param inputUpdateEventDto the input DTO containing updated  event details
     * @return an OutputEventDto eventDto representing the updated event
     * @throws WP2ResourceNotFoundException if no event is found with the given id
     */
    @NonNull
    OutputEventDto updateById(@NonNull UUID eventId, @NonNull InputUpdateEventDto inputUpdateEventDto);


    /**
     * Deletes event by id. It will move the event entity on a separated table containing deleted records.
     *
     * @param eventId the id of the event to delete
     * @throws WP2ResourceNotFoundException if no event is found with the given id
     */
    @NonNull
    void deleteById(@NonNull UUID eventId);

    /**
     * register a user to an event
     *
     * @param eventId the id of the event
     * @throws WP2ResourceNotFoundException  if no event is found with the given id
     * @throws WP2DuplicateResourceException if user is already subscribed to the event
     */
    @NonNull
    OutputEventDto register(@NonNull UUID eventId);

    /**
     * unregister a user to an event
     *
     * @param eventId the id of the event
     * @throws WP2ResourceNotFoundException if no event is found with the given id
     */
    @NonNull
    OutputEventDto unregister(@NonNull UUID eventId);

    /**
     * register a user to a program
     *
     * @param eventId   the id of the event
     * @param programId the id of the program
     * @throws WP2ResourceNotFoundException  if no program is found with the given id
     * @throws WP2DuplicateResourceException if user is already subscribed to the program
     * @throws WP2ValidationException        if the program reached its maximum capacity
     */
    @NonNull
    OutputEventDto registerProgram(@NonNull UUID eventId,
                                   @NonNull UUID programId,
                                   @NonNull InputProgramSubscribedParticipantDto inputProgramSubscribedParticipantDto);

    /**
     * unregister a user to a program
     *
     * @param eventId   the id of the event
     * @param programId the id of the program
     * @throws WP2ResourceNotFoundException if no program is found with the given id
     */
    @NonNull
    OutputEventDto unregisterProgram(@NonNull UUID eventId, @NonNull UUID programId);
}
