package com.finconsgroup.itserr.marketplace.event.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.CascadeType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.OneToMany;
import jakarta.persistence.OrderBy;
import jakarta.persistence.OrderColumn;
import jakarta.persistence.PreUpdate;
import jakarta.persistence.Table;
import jakarta.persistence.Version;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.ToString;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.time.LocalTime;
import java.util.List;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.event.dm.util.DomainConstants.LINK_TEXT_LENGTH;
import static com.finconsgroup.itserr.marketplace.event.dm.util.DomainConstants.MEDIUM_TEXT_LENGTH;
import static com.finconsgroup.itserr.marketplace.event.dm.util.DomainConstants.SHORT_TEXT_LENGTH;

/**
 * Entity class representing a program in the event system.
 */
@SuppressWarnings("DefaultAnnotationParam")
@Entity
@Table(name = "program")
@SuperBuilder
@Getter
@Setter
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ProgramEntity extends AbstractUUIDEntity {

    /**
     * Association to the related schedule
     */
    @ManyToOne(fetch = FetchType.LAZY, optional = false)
    @JoinColumn(name = "schedule_id", updatable = false)
    private ScheduleEntity schedule;

    /**
     * The ID of the related schedule.
     */
    @Column(name = "schedule_id", insertable = false, updatable = false)
    private UUID scheduleId;

    /**
     * The title.
     */
    @Column(name = "title", nullable = false, length = MEDIUM_TEXT_LENGTH)
    private String title;

    /**
     * The description.
     */
    @Column(name = "description", length = MEDIUM_TEXT_LENGTH)
    private String description;

    /**
     * The program link
     */
    @Column(name = "program_link", length = LINK_TEXT_LENGTH)
    private String programLink;

    /**
     * The place.
     */
    @Column(name = "place", length = MEDIUM_TEXT_LENGTH)
    private String place;

    /**
     * The street.
     */
    @Column(name = "street", length = MEDIUM_TEXT_LENGTH)
    private String street;

    /**
     * The house number.
     */
    @Column(name = "house_number", length = SHORT_TEXT_LENGTH)
    private String houseNumber;

    /**
     * Start Time of the schedule.
     */
    @Column(name = "start_time", nullable = false)
    private LocalTime startTime;

    /**
     * End Time of the schedule.
     */
    @Column(name = "end_time", nullable = false)
    private LocalTime endTime;

    /**
     * The maximum number of participants allowed for the program.
     */
    @Column(name = "max_participants")
    private Integer maxParticipants;

    /**
     * The number of participants subscribed for the program.
     */
    @Column(name = "subscribed_participants")
    private Integer subscribedParticipantsCount;

    /**
     * The subscribed participants related to the program.
     */
    @OneToMany(cascade = CascadeType.REMOVE, orphanRemoval = true, mappedBy = "program")
    @OrderBy("creationTime ASC")
    @ToString.Exclude
    private List<ProgramSubscribedParticipantEntity> subscribedParticipants;

    /**
     * The program conductors related to the program.
     */
    @OneToMany(cascade = CascadeType.ALL, orphanRemoval = true, mappedBy = "program")
    @OrderColumn(name = "conductor_order")
    @ToString.Exclude
    private List<ProgramConductorEntity> programConductors;

    /**
     * The timestamp when the program was created.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the program was last updated.
     */
    @Column(name = "update_time", nullable = false)
    private Instant updateTime;

    /**
     * The version field used for optimistic locking.
     * <p>
     * This value is automatically managed by JPA to detect concurrent updates.
     * Each time the entity is updated, the version is incremented.
     * If two transactions try to update the same entity simultaneously,
     * JPA will detect the conflict based on this version
     * and throw an {@link jakarta.persistence.OptimisticLockException}.
     */
    @Version
    @Column(name = "version", nullable = false)
    private long version;

    /**
     * The ordering column.
     */
    @Column(name = "program_order", nullable = false)
    private long programOrder;

    @Override
    public void prePersist() {
        super.prePersist();
        Instant now = Instant.now();
        if (creationTime == null) {
            creationTime = now;
        }
        if (updateTime == null) {
            updateTime = now;
        }
    }

    /**
     * Updates {@code updateTime} just before the entity is updated.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PreUpdate
    public void onUpdate() {
        updateTime = Instant.now();
    }

    /**
     * Increments the participant count if there is available capacity.
     */
    public void incrementParticipants() {
        if (!hasAvailableCapacity()) {
            return;
        }
        this.subscribedParticipantsCount++;
    }

    /**
     * Decrements the participant count, ensuring it doesn't go below zero.
     */
    public void decrementParticipants() {
        if (this.subscribedParticipantsCount == 0) {
            return;
        }
        this.subscribedParticipantsCount--;
    }

    /**
     * Checks if the program has available capacity for more participants.
     *
     * @return true if there is available capacity, false if the program is full
     */
    public boolean hasAvailableCapacity() {
        if (maxParticipants == null) {
            return true;
        }
        if (subscribedParticipantsCount == null) {
            subscribedParticipantsCount = 0;
        }
        return subscribedParticipantsCount < maxParticipants;
    }
}
