package com.finconsgroup.itserr.marketplace.favourite.user.bs.client;


import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.InputCreateFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.InputPatchFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.OutputFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.OutputFavouriteUserItemSubscriberDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.enums.ItemContext;
import jakarta.validation.Valid;
import org.springframework.data.domain.Sort;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PatchMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;

import java.util.List;
import java.util.UUID;

/**
 * Feign Client interface for Favourite User Domain Service
 */
public interface FavouriteUserDmClient {

    /**
     * Creates a new favourite user item
     *
     * @param dto the input data transfer object containing favourite user item details
     * @return the created {@link OutputFavouriteUserItemDto} and HTTP status 201 (CREATED)
     * @throws feign.FeignException.FeignClientException.InternalServerError if there is an error inside the dm service
     */
    @PostMapping(value = "/favourite-user/items", consumes = MediaType.APPLICATION_JSON_VALUE,
        produces = MediaType.APPLICATION_JSON_VALUE)
    OutputFavouriteUserItemDto create(@RequestBody InputCreateFavouriteUserItemDto dto);

    /**
     * Fetch favourite user item for the provided id.
     *
     * @param favouriteUserItemId the id of the favourite user item to fetch
     * @return the found {@link OutputFavouriteUserItemDto} and HTTP status 200 (Ok)
     * @throws feign.FeignException.FeignClientException.NotFound if the favourite user item does not exist
     * @throws feign.FeignException.FeignClientException.InternalServerError if there is an error inside the dm service
     */
    @GetMapping("/favourite-user/items/{favouriteUserItemId}")
    OutputFavouriteUserItemDto get(@PathVariable("favouriteUserItemId") UUID favouriteUserItemId);

    /**
     * Delete favourite user item for provided id.
     *
     * @param favouriteUserItemId the id of the favourite user item to delete
     *                Returns HTTP status 204 (No Content), if the favourite user item is deleted successfully
     * @throws feign.FeignException.FeignClientException.NotFound if the favourite user item does not exist
     * @throws feign.FeignException.FeignClientException.InternalServerError if there is an error inside the dm service
     */
    @DeleteMapping("/favourite-user/items/{favouriteUserItemId}")
    void delete(@PathVariable("favouriteUserItemId") UUID favouriteUserItemId);

    /**
     * Updates only the provided fields of favourite user item i.e. patch operation.
     *
     * @param dto the input data transfer object containing favourite user item details to update
     * @return the updated {@link OutputFavouriteUserItemDto} and HTTP status 200 (OK)
     */
    @PatchMapping("/favourite-user/items/{favouriteUserItemId}")
    OutputFavouriteUserItemDto patch(@PathVariable("favouriteUserItemId") UUID favouriteUserItemId,
                                     @Valid @RequestBody InputPatchFavouriteUserItemDto dto);

    /**
     * Retrieves a paginated list of all favourite user items matching the provided context.
     *
     * @param context    the ItemContext to search favourite user items for
     * @param subContext the subContext to search favourite user items for
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of favourite user item per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return ResponseEntity containing a page of OutputFavouriteUserItemDto and HTTP status 200 (OK)
     * @throws feign.FeignException.FeignClientException.InternalServerError if there is an error inside the dm service
     */
    @GetMapping(value = "/favourite-user/items", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPageDto<OutputFavouriteUserItemDto> findByContext(
        @RequestParam(name = "context") String context,
        @RequestParam(name = "subContext") String subContext,
        @RequestParam(name = "pageNumber") int pageNumber,
        @RequestParam(name = "pageSize") int pageSize,
        @RequestParam(name = "sort") String sort,
        @RequestParam(name = "direction") String direction
    );

    /**
     * Retrieves a paginated list of all favourite user items matching the provided context.
     *
     * @param context           the {@link ItemContext} to search favourite user items for
     * @param subContext        the subContext to search favourite user items for
     * @param itemIds           the delimiter (default ',') separated list of item ids
     * @param itemIdSeparator   the separator for item ids (default if ",")
     * @return ResponseEntity containing a page of OutputFavouriteUserItemDto and HTTP status 200 (OK)
     */
    @GetMapping(value = "/favourite-user/items-by-item-ids", produces = MediaType.APPLICATION_JSON_VALUE)
    List<OutputFavouriteUserItemDto> findByContextAndItemIds(
        @RequestParam(name = "context") String context,
        @RequestParam(name = "subContext") String subContext,
        @RequestParam(name = "itemIds") String itemIds,
        @RequestParam(name = "itemIdSeparator") String itemIdSeparator
    );

    /**
     * Deletes related favourite user items for all users.
     *
     * @param context    the {@link ItemContext} to search favourite user items for
     * @param subContext the subContext to search favourite user items for (optional)
     * @param itemId     the item id subscribed for
     * @return ResponseEntity containing a list of deleted OutputFavouriteUserItemDtos and HTTP status 200 (OK)
     */
    @DeleteMapping(value = "/favourite-user/items-by-item-id", produces = MediaType.APPLICATION_JSON_VALUE)
    List<OutputFavouriteUserItemDto> deleteByContextAndItemId(
            @RequestParam(name = "context") String context,
            @RequestParam(name = "subContext", required = false) String subContext,
            @RequestParam(name = "itemId") String itemId
    );

    /**
     * Retrieves a paginated list of favourite user items representing the subscribers that have followed the item.
     *
     * @param context    the {@link ItemContext} to search favourite user items for
     * @param subContext the subContext to search favourite user items for (optional)
     * @param itemId     the item id subscribed for
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize   the number of favourite user item per page (default is 10)
     * @param sort       the field to sort by (default is "id")
     * @param direction  the direction of sorting (default is ascending)
     * @return ResponseEntity containing a page of OutputFavouriteUserItemSubscriberDto and HTTP status 200 (OK)
     */
    @GetMapping(value = "/favourite-user/items/subscribers", produces = MediaType.APPLICATION_JSON_VALUE)
    OutputPageDto<OutputFavouriteUserItemSubscriberDto> findSubscribers(
            @RequestParam(name = "context") String context,
            @RequestParam(name = "subContext") String subContext,
            @RequestParam(name = "itemId") String itemId,
            @RequestParam(name = "pageNumber") int pageNumber,
            @RequestParam(name = "pageSize") int pageSize,
            @RequestParam(name = "sort") String sort,
            @RequestParam(name = "direction") Sort.Direction direction
    );
}
