package com.finconsgroup.itserr.marketplace.favourite.user.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.dto.PageRequestDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2BusinessException;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.bean.DetailRequest;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.client.FavouriteUserDmClient;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.client.UserProfileDmClient;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.FavouriteUserItemDetail;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.InputCreateFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.InputFindFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.InputPatchFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.OutputFavouriteUserItemDetailDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.OutputFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.OutputFavouriteUserItemSubscriberDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.people.InputFindUserProfilesByTokenInfoDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.people.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.enums.ItemContext;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.EventProducer;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.dto.FavouriteItemMessageBodyDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.dto.ResourceByFolloweeMessageBodyDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.helper.MessagingHelper;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.service.FavouriteUserItemDetailProvider;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.service.FavouriteUserItemDetailProviderRegistry;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.service.FavouriteUserItemService;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;
import com.finconsgroup.itserr.messaging.dto.MessagingEventUserDto;
import io.micrometer.common.util.StringUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.function.IntFunction;
import java.util.stream.Collectors;

/**
 * Default implementation of {@link FavouriteUserItemService} to perform operations related to favourite user item resources.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultFavouriteUserItemService implements FavouriteUserItemService {

    /**
     * Default field to be used for sorting in favourite user domain service
     */
    private static final String USER_DM_DEFAULT_SORT_FIELD = "id";

    /**
     * Fields supported for sorting in the favourite user domain service
     */
    public static final Set<String> USER_DM_SORT_FIELDS = Set.of("id", "context", "subContext", "itemId",
            "creationTime", "updateTime");

    /**
     * Page size to fetch all the favourite user items
     */
    private static final int FETCH_ALL_PAGE_SIZE = 200;

    // suppress warning as bean is injected by core library
    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    private final FavouriteUserDmClient favouriteUserDmClient;
    // suppress warning as bean is injected by core library
    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    private final UserProfileDmClient userProfileDmClient;
    private final FavouriteUserItemDetailProviderRegistry favouriteUserItemDetailProviderRegistry;

    /** An instance of {@code MessagingHelper} used for building messaging-related DTOs and handling messaging operations within the service. */
    private final MessagingHelper messagingHelper;

    /** Handles the production of event messages. */
    private final EventProducer eventProducer;

    @NonNull
    @Override
    public OutputFavouriteUserItemDto create(@NonNull InputCreateFavouriteUserItemDto dto) {

        // create favourite
        final OutputFavouriteUserItemDto favouriteItem = favouriteUserDmClient.create(dto);

        // send bus event
        final FavouriteItemMessageBodyDto message = messagingHelper.buildFavouriteItemMessage(favouriteItem);
        eventProducer.publishFavouriteItemCreatedEvent(message);

        // return favourite
        return favouriteItem;

    }

    @NonNull
    @Override
    public <T extends FavouriteUserItemDetail> OutputFavouriteUserItemDetailDto<T> findById(@NonNull UUID favouriteUserItemId) {
        OutputFavouriteUserItemDto outputFavouriteUserItemDto = favouriteUserDmClient.get(favouriteUserItemId);
        return getDetailForItem(outputFavouriteUserItemDto);
    }

    @NonNull
    private <T extends FavouriteUserItemDetail> OutputFavouriteUserItemDetailDto<T> getDetailForItem(OutputFavouriteUserItemDto outputFavouriteUserItemDto) {
        final ItemContext context = outputFavouriteUserItemDto.getContext();
        FavouriteUserItemDetailProvider<T> detailProvider = getDetailProvider(context);
        OutputFavouriteUserItemDetailDto<T> outputFavouriteUserItemDetailDto = buildFavouriteUserItemDetailDto(outputFavouriteUserItemDto);
        T itemDetail = detailProvider.getDetailById(outputFavouriteUserItemDto.getItemId());
        outputFavouriteUserItemDetailDto.setItemDetail(itemDetail);
        return outputFavouriteUserItemDetailDto;
    }

    @Override
    public void deleteById(@NonNull UUID favouriteUserItemId) {

        // Find the favourite item
        final OutputFavouriteUserItemDto favouriteItem = favouriteUserDmClient.get(favouriteUserItemId);

        // Delete the favourite item
        favouriteUserDmClient.delete(favouriteUserItemId);

        // send bus event
        final FavouriteItemMessageBodyDto message = messagingHelper.buildFavouriteItemMessage(favouriteItem);
        eventProducer.publishFavouriteItemDeletedEvent(message);

    }

    @NonNull
    @Override
    public OutputFavouriteUserItemDto patchById(@NonNull UUID favouriteUserItemId, @NonNull InputPatchFavouriteUserItemDto dto) {

        // update favourite item
        final OutputFavouriteUserItemDto updatedFavouriteItem = favouriteUserDmClient.patch(favouriteUserItemId, dto);

        // send bus event
        final FavouriteItemMessageBodyDto message = messagingHelper.buildFavouriteItemMessage(updatedFavouriteItem);
        eventProducer.publishFavouriteItemUpdatedEvent(message);

        // return updated favourite item
        return updatedFavouriteItem;

    }

    @NonNull
    @Override
    public <T extends FavouriteUserItemDetail> OutputPageDto<OutputFavouriteUserItemDetailDto<T>> findByContext(@NonNull InputFindFavouriteUserItemDto dto) {
        ItemContext context = dto.getContext();
        FavouriteUserItemDetailProvider<T> detailProvider = getDetailProvider(context);
        List<DetailRequest.Filter> detailProviderFilters = detailProvider.getApplicableFilters(dto.getFilters());

        // if the sorting/filtering is to be applied by favourite user item domain service
        if (isSupportedUserDomainServiceSort(dto.getSort()) && detailProviderFilters.isEmpty()) {
            return findByContextWithSortFiltersOnUserDmService(dto, detailProvider);
        } else {
            return findByContextWithSortFiltersOnDetailProvider(dto, detailProvider, detailProviderFilters);
        }
    }

    @NonNull
    @Override
    public List<OutputFavouriteUserItemDto> findByContextAndItemIds(@NonNull ItemContext context,
            String subContext,
            @NonNull String itemIds,
            String itemIdSeparator) {
        return favouriteUserDmClient.findByContextAndItemIds(context.getId(), subContext, itemIds, itemIdSeparator);
    }

    @Override
    public void deleteByContextAndItemId(@NonNull ItemContext context,
                                         String subContext,
                                         @NonNull String itemId,
                                         MessagingEventUserDto messagingEventUserDto) {
        log.info("Deleting all user favourites for item with context: {}, subContext: {} and id: {}",
                context, subContext, itemId);
        List<OutputFavouriteUserItemDto> deletedFavouriteItemDtos =
                favouriteUserDmClient.deleteByContextAndItemId(context.getId(), subContext, itemId);
        // send bus events
        deletedFavouriteItemDtos.forEach(dto -> {
            final FavouriteItemMessageBodyDto message = messagingHelper.buildFavouriteItemMessage(dto);
            // set user from the source message, if available and not already set on the message
            if (messagingEventUserDto != null && (message.getUser() == null || message.getUser().getId() == null)) {
                message.setUser(messagingEventUserDto);
            }
            eventProducer.publishFavouriteItemDeletedEvent(message);
        });
    }

    @Override
    public void publishResourceCreatedByFolloweeEvent(@NonNull String context,
                                                      @NonNull String followeeRef,
                                                      @NonNull MessagingEventDto<?> resourceEventDto) {

        // fetch the followee user using user profile dm client to resolve the user details
        List<OutputUserProfileDto> userProfileDtos = userProfileDmClient.findAllByTokenInfo(InputFindUserProfilesByTokenInfoDto.builder()
                .tokenInfos(List.of(followeeRef))
                .build());

        if (userProfileDtos == null || userProfileDtos.isEmpty()) {
            log.info("No user profile found for followee reference: {}", followeeRef);
            return;
        }

        OutputUserProfileDto followee = userProfileDtos.getFirst();

        // fetch all the users that are following the followee
        List<OutputFavouriteUserItemSubscriberDto> followingUsers = getAllSubscribers(ItemContext.PEOPLE, null, followee.getId().toString())
                .stream().filter(OutputFavouriteUserItemDto::isFollowed)
                .toList();

        // no users are following the creator
        if (followingUsers.isEmpty()) {
            log.info("No users following the followee: {}", followee.getId());
            return;
        }

        ResourceByFolloweeMessageBodyDto messagingEventDto = new ResourceByFolloweeMessageBodyDto();

        messagingEventDto.setId(resourceEventDto.getId());
        messagingEventDto.setUser(resourceEventDto.getUser());
        messagingEventDto.setName(resourceEventDto.getName());
        messagingEventDto.setTitle(resourceEventDto.getTitle());
        messagingEventDto.setCategory(resourceEventDto.getCategory());
        messagingEventDto.setStatus(resourceEventDto.getStatus());
        messagingEventDto.setMessage(resourceEventDto.getMessage());
        messagingEventDto.setTimestamp(resourceEventDto.getTimestamp());

        ResourceByFolloweeMessageBodyDto.AdditionalData additionalData =
                ResourceByFolloweeMessageBodyDto.AdditionalData.builder()
                        .itemContext(context)
                        .followeeId(followee.getId())
                        .followeeUsername(followee.getPreferredUsername())
                        .followeeName(followee.getName())
                        .notifyUserIds(followingUsers.stream().map(dto -> dto.getUserId().toString()).toList())
                        .build();
        messagingEventDto.setAdditionalData(additionalData);

        eventProducer.publishResourceCreatedByFolloweeEvent(messagingEventDto);
    }

    // 1. Fetch the page from Favourite User Domain Service by applying sort/filters
    // 2. Fetch the details from Detail Provider for single page of item ids
    // 3. Merge the details with favourite user item data with page information from Favourite User Domain Service
    // 4. Return the merge data
    private <T extends FavouriteUserItemDetail> OutputPageDto<OutputFavouriteUserItemDetailDto<T>> findByContextWithSortFiltersOnUserDmService(
            @NonNull InputFindFavouriteUserItemDto dto,
            @NonNull FavouriteUserItemDetailProvider<T> detailProvider
    ) {
        OutputPageDto<OutputFavouriteUserItemDto> favouriteItemsPage = favouriteUserDmClient.findByContext(
                dto.getContext().getId(), dto.getSubContext(), dto.getPageNumber(), dto.getPageSize(), dto.getSort(),
                dto.getDirection().name());

        // no need to fetch details if there are no user favourite items for provided context
        if (favouriteItemsPage.getContent().isEmpty()) {
            return OutputPageDto.emptyWithPage(favouriteItemsPage.getPage());
        }

        List<String> itemIds = favouriteItemsPage.getContent().stream().map(OutputFavouriteUserItemDto::getItemId).toList();

        OutputPageDto<T> outputDetailPageDto = detailProvider.getDetails(DetailRequest
                .builder()
                .itemContext(dto.getContext())
                .itemIds(itemIds)
                .pageRequestDto(PageRequestDto
                        .builder()
                        .pageNumber(0)
                        .pageSize(itemIds.size())
                        .build())
                .build());

        Map<String, T> itemDetailByItemId = outputDetailPageDto
                .getContent()
                .stream()
                .collect(Collectors.toMap(FavouriteUserItemDetail::getItemId, Function.identity()));

        List<OutputFavouriteUserItemDetailDto<T>> outputFavouriteUserItemDetailList = new ArrayList<>();
        favouriteItemsPage.getContent().forEach(favouriteUserItemDto -> {
            OutputFavouriteUserItemDetailDto<T> outputFavouriteUserItemDetailDto = buildFavouriteUserItemDetailDto(favouriteUserItemDto);
            outputFavouriteUserItemDetailDto.setItemDetail(itemDetailByItemId.get(favouriteUserItemDto.getItemId()));
            outputFavouriteUserItemDetailList.add(outputFavouriteUserItemDetailDto);
        });

        return OutputPageDto
                .<OutputFavouriteUserItemDetailDto<T>> builder()
                .content(outputFavouriteUserItemDetailList)
                .page(favouriteItemsPage.getPage())
                .build();
    }

    // 1. Fetch all user favourite items from Favourite User Domain Service
    // 2. Fetch the details for single page from Detail Provider by applying sort/filters, passing all user favourite item ids
    // 3. Merge the details with favourite user item data with page information from detail provider
    // 4. Return the merge data
    private <T extends FavouriteUserItemDetail> OutputPageDto<OutputFavouriteUserItemDetailDto<T>> findByContextWithSortFiltersOnDetailProvider(
            @NonNull InputFindFavouriteUserItemDto dto,
            @NonNull FavouriteUserItemDetailProvider<T> detailProvider,
            @NonNull List<DetailRequest.Filter> detailProviderFilters) {
        // if the sorting/filtering is to be applied by detail provider
        List<OutputFavouriteUserItemDto> items = getAllFavouriteItems(dto.getContext(), dto.getSubContext());

        // no need to fetch details if there are no user favourite items for provided context
        if (items.isEmpty()) {
            return OutputPageDto.emptyWithPageSize(dto.getPageSize());
        }

        List<String> itemIds = new ArrayList<>();
        Map<String, OutputFavouriteUserItemDetailDto<T>> favouriteUserItemDetailDtoByItemId = new HashMap<>();
        items.forEach(item -> {
            itemIds.add(item.getItemId());
            favouriteUserItemDetailDtoByItemId.put(item.getItemId(), buildFavouriteUserItemDetailDto(item));
        });

        // pass the sort field only if starts with 'itemDetail.' removing the 'itemDetail.' prefix
        String sort = "";
        if (StringUtils.isNotBlank(dto.getSort()) && dto.getSort().startsWith("itemDetail.")) {
            sort = dto.getSort().substring("itemDetail.".length());
        }

        OutputPageDto<T> outputDetailPageDto = detailProvider.getDetails(DetailRequest
                .builder()
                .itemContext(dto.getContext())
                .itemIds(itemIds)
                .filters(detailProviderFilters)
                .pageRequestDto(PageRequestDto
                        .builder()
                        .pageNumber(dto.getPageNumber())
                        .pageSize(dto.getPageSize())
                        .sort(sort)
                        .direction(dto.getDirection())
                        .build())
                .build());

        List<OutputFavouriteUserItemDetailDto<T>> outputFavouriteUserItemDetailList = new ArrayList<>();
        outputDetailPageDto.getContent().forEach(itemDetailDto -> {
            OutputFavouriteUserItemDetailDto<T> outputFavouriteUserItemDetailDto =
                    favouriteUserItemDetailDtoByItemId.get(itemDetailDto.getItemId());
            outputFavouriteUserItemDetailDto.setItemDetail(itemDetailDto);
            outputFavouriteUserItemDetailList.add(outputFavouriteUserItemDetailDto);
        });

        return OutputPageDto
                .<OutputFavouriteUserItemDetailDto<T>> builder()
                .content(outputFavouriteUserItemDetailList)
                .page(outputDetailPageDto.getPage())
                .build();
    }

    /**
     * Fetch all the favourite user item ids for the given context.
     *
     * @param context the {@link ItemContext}
     * @return all the found favourite items for the user
     */
    @NonNull
    private List<OutputFavouriteUserItemDto> getAllFavouriteItems(@NonNull ItemContext context, String subContext) {
        return getAllPagedData(pageNumber ->
                favouriteUserDmClient.findByContext(
                        context.getId(), subContext, pageNumber, FETCH_ALL_PAGE_SIZE, USER_DM_DEFAULT_SORT_FIELD,
                        SortDirection.ASC.name())
        );
    }

    /**
     * Fetch all the favourite user items representing the subscribers that have followed the item.
     *
     * @param context the {@link ItemContext}
     * @param subContext the sub context
     * @param itemId  the item id
     * @return all the found subscribers (followers) for the item
     */
    @NonNull
    private List<OutputFavouriteUserItemSubscriberDto> getAllSubscribers(@NonNull ItemContext context,
                                                                         String subContext,
                                                                         @NonNull String itemId) {
        return getAllPagedData(pageNumber ->
                favouriteUserDmClient.findSubscribers(
                        context.getId(), subContext, itemId, pageNumber, FETCH_ALL_PAGE_SIZE, USER_DM_DEFAULT_SORT_FIELD,
                        Sort.Direction.ASC)
        );
    }

    /**
     * Fetch all the pages of data for the page provider.
     *
     * @param pageSupplierByPageNumber the function that supplies the page of favourite items for provided page number
     * @param <T> the type of the item dto
     * @return all the found pages of data
     */
    @NonNull
    private <T> List<T> getAllPagedData(@NonNull IntFunction<OutputPageDto<T>> pageSupplierByPageNumber) {
        int pageNumber = 0;
        List<T> itemDtos = new ArrayList<>();

        while (true) {
            OutputPageDto<T> favouriteItemsPage = pageSupplierByPageNumber.apply(pageNumber);

            // collect all items returned
            itemDtos.addAll(favouriteItemsPage.getContent());

            // 1. no more items returned, so we can exit the loop
            // 2. all items returned, so we can exit the loop
            // 3. not sufficient items returned for current page, so we can exit the loop
            if (favouriteItemsPage.getContent().isEmpty()
                    || itemDtos.size() >= favouriteItemsPage.getPage().getTotalElements()
                    || favouriteItemsPage.getContent().size() < FETCH_ALL_PAGE_SIZE) {
                break;
            }

            pageNumber++;
        }
        return itemDtos;
    }

    private <T extends FavouriteUserItemDetail> FavouriteUserItemDetailProvider<T> getDetailProvider(@NonNull ItemContext context) {
        @SuppressWarnings("unchecked")
        FavouriteUserItemDetailProvider<T> detailProvider = (FavouriteUserItemDetailProvider<T>) favouriteUserItemDetailProviderRegistry
                .getProvider(context)
                .orElseThrow(() -> new WP2BusinessException("Detail provider not found for " + context.getId()));
        return detailProvider;
    }

    private <T extends FavouriteUserItemDetail> OutputFavouriteUserItemDetailDto<T> buildFavouriteUserItemDetailDto(OutputFavouriteUserItemDto itemDto) {
        return OutputFavouriteUserItemDetailDto
                .<T> detailBuilder()
                .id(itemDto.getId())
                .context(itemDto.getContext())
                .subContext(itemDto.getSubContext())
                .itemId(itemDto.getItemId())
                .followed(itemDto.isFollowed())
                .creationTime(itemDto.getCreationTime())
                .updateTime(itemDto.getUpdateTime())
                .build();
    }

    /**
     * Returns if the sort field is supported by the favourite user domain service
     *
     * @param sort the sort field
     * @return {@code true} if supported, {@code false} otherwise.
     */
    private boolean isSupportedUserDomainServiceSort(String sort) {
        return StringUtils.isBlank(sort) || USER_DM_SORT_FIELDS.contains(sort);
    }
}
