package com.finconsgroup.itserr.marketplace.favourite.user.bs.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.bean.FilterProperties;
import com.finconsgroup.itserr.marketplace.core.web.dto.OutputPageDto;
import com.finconsgroup.itserr.marketplace.core.web.enums.SortDirection;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.bean.DetailRequest;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.client.SearchDmClient;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.config.FavouriteUserBsProperties;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.InputFindFavouriteDetailDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.dto.search.OutputCatalogFavouriteDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.enums.ItemContext;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.service.FavouriteUserItemDetailProvider;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.service.FavouriteUserItemDetailProviderRegistry;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.Set;

/**
 * Implementation of {@link FavouriteUserItemDetailProvider} for Catalog Items
 */
@Service
@Slf4j
public class FavouriteUserItemCatalogDetailProvider extends AbstractFavouriteUserItemDetailProvider<OutputCatalogFavouriteDto> {

    private static final String FILTER_KEY_TYPE = "type";
    private static final String FILTER_KEY_STATUS = "status";
    private static final Set<String> SUPPORTED_FILTER_KEYS = Set.of(FILTER_KEY_TYPE, FILTER_KEY_STATUS);

    private final SearchDmClient searchDmClient;

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    public FavouriteUserItemCatalogDetailProvider(@NonNull FavouriteUserItemDetailProviderRegistry favouriteUserItemDetailProviderRegistry,
                                                  @NonNull SearchDmClient searchDmClient,
                                                  @NonNull FavouriteUserBsProperties favouriteUserBsProperties) {
        super(favouriteUserItemDetailProviderRegistry, favouriteUserBsProperties, ItemContext.CATALOG);
        Objects.requireNonNull(searchDmClient, "searchDmClient must not be null");

        this.searchDmClient = searchDmClient;
    }

    @NonNull
    @Override
    public OutputCatalogFavouriteDto getDetailById(@NonNull String itemId) {
        Objects.requireNonNull(itemId, "itemId must not be null");

        InputFindFavouriteDetailDto inputFindFavouriteDetailDto = InputFindFavouriteDetailDto.builder().ids(List.of(itemId)).build();
        OutputPageDto<OutputCatalogFavouriteDto> outputCatalogFavouritePageDto = searchDmClient.getFavourite(
            inputFindFavouriteDetailDto, null, 0, 1, "id", SortDirection.ASC);

        if (CollectionUtils.isEmpty(outputCatalogFavouritePageDto.getContent())) {
            throw new WP2ResourceNotFoundException("Favourite catalog item not found");
        }

        return outputCatalogFavouritePageDto.getContent().getFirst();
    }

    @NonNull
    @Override
    public OutputPageDto<OutputCatalogFavouriteDto> getDetails(@NonNull DetailRequest request) {
        // avoid calling the business service if there are no item ids in the request
        if (request.getItemIds() == null || request.getItemIds().isEmpty()) {
            return OutputPageDto.emptyWithPageSize(request.getPageRequestDto().getPageSize());
        }

        StringBuilder filters = new StringBuilder();
        if (request.getFilters() != null) {
            FilterProperties filterProperties = getFilterProperties();
            for (DetailRequest.Filter filter : request.getFilters()) {
                filters.append(filter.getKey()).append(filterProperties.keyValueSeparator()).append(filter.getValue());
            }
        }

        return searchDmClient.getFavourite(
            InputFindFavouriteDetailDto.builder().ids(request.getItemIds()).build(),
            filters.toString(),
            request.getPageRequestDto().getPageNumber(),
            request.getPageRequestDto().getPageSize(),
            request.getPageRequestDto().getSort(),
            request.getPageRequestDto().getDirection());
    }

    @NonNull
    @Override
    public Set<String> getSupportedFilterKeys() {
        return SUPPORTED_FILTER_KEYS;
    }
}
