package com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.MemberRequestStatus;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.MemberRequestType;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.PrePersist;
import jakarta.persistence.Table;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.UUID;

/**
 * This entity represents a soft deleted {@link ArchivedMemberRequestEntity}.
 */
@Entity
@Table(name = "archived_member_request")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ArchivedMemberRequestEntity extends AbstractUUIDEntity {

    /**
     * ID of user.
     */
    @Column(name = "user_id", nullable = false)
    private UUID userId;

    /**
     * ID of the institutional page.
     */
    @Column(name = "institutional_page_id", nullable = false)
    private UUID institutionalPageId;

    /**
     * type of request (join request or invitation) .
     */
    @Column(name = "request_type", nullable = false)
    private MemberRequestType requestType;

    /**
     * Indicate if the user should be added as wp leader.
     */
    @Column(name = "wp_leader", nullable = false)
    private boolean wpLeader;

    /**
     * invitation, join request or rejection message
     */
    @Column(name = "message")
    private String message;

    /**
     * status of request (approved or rejected) .
     */
    @Column(name = "request_status", nullable = false)
    private MemberRequestStatus requestStatus;

    /**
     * The timestamp when the memberRequest was created
     * in the original (non-archived) table.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the memberRequest was last updated
     * in the original (non-archived) table.
     */
    @Column(name = "update_time", nullable = false, updatable = false)
    private Instant updateTime;

    /**
     * The timestamp when this record was inserted into the archive table,
     * representing the moment the member_request was archived.
     */
    @Column(name = "archival_time", nullable = false, updatable = false)
    private Instant archivalTime;

    /**
     * Sets {@code archivalTime} before the entity is persisted.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PrePersist
    public void onCreate() {
        archivalTime = Instant.now().truncatedTo(ChronoUnit.MICROS);
    }

}
