package com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity;

import com.finconsgroup.itserr.marketplace.core.entity.AbstractUUIDEntity;
import jakarta.persistence.Column;
import jakarta.persistence.Entity;
import jakarta.persistence.FetchType;
import jakarta.persistence.JoinColumn;
import jakarta.persistence.ManyToOne;
import jakarta.persistence.PrePersist;
import jakarta.persistence.Table;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.experimental.SuperBuilder;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.UUID;

import static com.finconsgroup.itserr.marketplace.institutionalpage.dm.util.Constants.MAX_LONG_TEXT_LENGTH;

/**
 * This entity represents a soft deleted {@link ParagraphEntity}.
 */
@Entity
@Table(name = "archived_paragraph")
@Data
@SuperBuilder
@EqualsAndHashCode(callSuper = true)
@NoArgsConstructor
public class ArchivedParagraphEntity extends AbstractUUIDEntity {

    /**
     * Association of the related institutional page for navigation.
     */
    @ManyToOne(fetch = FetchType.LAZY, optional = false)
    @JoinColumn(name = "institutional_page_id", updatable = false)
    private ArchivedInstitutionalPageEntity institutionalPage;

    /**
     * ID of the related institutional page.
     * <p></p>
     * Had to add this column to avoid joins for findByInstitutionalPageId* queries.
     * <a href="https://github.com/spring-projects/spring-data-jpa/issues/1575">JPA Github Issue</a>
     */
    @Column(name = "institutional_page_id", insertable = false, updatable = false)
    private UUID institutionalPageId;

    /**
     * The title of the paragraph.
     */
    @Column(name = "title", nullable = false, updatable = false)
    private String title;

    /**
     * The content of the paragraph.
     */
    @Column(name = "content", length = MAX_LONG_TEXT_LENGTH)
    private String content;

    /**
     * The image url of the paragraph.
     */
    @Column(name = "image_url")
    private String imageUrl;

    /**
     * The resource url of the paragraph.
     */
    @Column(name = "resource_url")
    private String resourceUrl;

    /**
     * list of labels.
     */
    @Column(name = "labels")
    private List<String> labels;

    /**
     * The timestamp when the institutionalPage was created
     * in the original (non-archived) table.
     */
    @Column(name = "creation_time", nullable = false, updatable = false)
    private Instant creationTime;

    /**
     * The timestamp when the institutionalPage was last updated
     * in the original (non-archived) table.
     */
    @Column(name = "update_time", nullable = false, updatable = false)
    private Instant updateTime;

    /**
     * The timestamp when this record was inserted into the archive table,
     * representing the moment the institutionalPage was archived.
     */
    @Column(name = "archival_time", nullable = false, updatable = false)
    private Instant archivalTime;

    /**
     * Sets {@code archivalTime} before the entity is persisted.
     * <p>
     * The timestamp is stored in UTC to ensure consistent and timezone-safe timestamps.
     * The result is truncated to microseconds to match PostgreSQL's default precision
     * for {@code TIMESTAMPTZ} columns (6 digits).
     * Avoids using Hibernate's {@code @CreationTimestamp} to ensure timestamps
     * are immediately available after {@code JpaRepository.save()},
     * without requiring an explicit {@code JpaRepository.flush()}.
     * </p>
     */
    @PrePersist
    public void onCreate() {
        archivalTime = Instant.now().truncatedTo(ChronoUnit.MICROS);
    }

}
