package com.finconsgroup.itserr.marketplace.institutionalpage.dm.service.impl;

import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.DtoBuilder;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.InstitutionalPageHelper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.InstitutionalPageModerationHandler;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.component.MemberHelper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputParagraphDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.InstitutionalPageEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.entity.ParagraphEntity;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.InvalidParagraphPositionException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.UpdateLockedException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.exception.ParagraphNotFoundException;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.mapper.ParagraphMapper;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.service.ParagraphService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.UUID;

/**
 * Default implementation of {@link ParagraphService}
 * to perform operations related to Paragraphs of institutionalPage
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultParagraphService implements ParagraphService {

    private final ParagraphMapper paragraphMapper;
    private final MemberHelper memberHelper;
    private final InstitutionalPageHelper institutionalPageHelper;
    private final DtoBuilder dtoBuilder;
    private final InstitutionalPageModerationHandler institutionalPageModerationHandler;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputInstitutionalPageDto addParagraph(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            Integer position,
            @NonNull InputParagraphDto inputParagraphDto
    ) {
        // 'retrieveLatestForMemberById' already checks membership, but we call 'verifyWpLeaderOrThrow' first
        // to return meaningful errors: if 'institutionalPageId' refers to a published page,
        // it's clearer to return "not a member" instead of "institutional page not found"
        memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        // retrieve institutional page entity
        InstitutionalPageEntity retrievedInstitutionalPage = institutionalPageHelper
                .retrieveLatestForMemberById(userId, institutionalPageId);
        // moderation process
        InstitutionalPageEntity institutionalPage = institutionalPageModerationHandler
                .update(userId, institutionalPageId, retrievedInstitutionalPage);
        // if page is being edited by someone else, then throws error
        UUID updateLockedBy = institutionalPage.getUpdateLockedBy();
        if (updateLockedBy == null || updateLockedBy.equals(userId)) {
            institutionalPage.setUpdateLockedBy(null);
        } else {
            throw new UpdateLockedException(institutionalPageId, updateLockedBy);
        }
        // validate insert position param
        List<ParagraphEntity> paragraphs = institutionalPage.getParagraphs();
        int insertPosition = (position != null) ? position : paragraphs.size();
        if (insertPosition < 0 || insertPosition > paragraphs.size()) {
            throw new InvalidParagraphPositionException(insertPosition, 0, paragraphs.size());
        }
        // build new paragraph and add to paragraphs list
        ParagraphEntity newParagraph = paragraphMapper.toEntity(inputParagraphDto);
        newParagraph.setInstitutionalPage(institutionalPage);
        paragraphs.add(insertPosition, newParagraph);
        // save institutional page and return output dto
        InstitutionalPageEntity savedInstitutionalPage = institutionalPageHelper.save(userId, institutionalPage);
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, savedInstitutionalPage);
        return outputInstitutionalPageDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputInstitutionalPageDto updateParagraph(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull UUID paragraphId,
            @NonNull InputParagraphDto inputParagraphDto
    ) {
        // 'retrieveLatestForMemberById' already checks membership, but we call 'verifyWpLeaderOrThrow' first
        // to return meaningful errors: if 'institutionalPageId' refers to a published page,
        // it's clearer to return "not a member" instead of "institutional page not found"
        memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        // retrieve institutional page entity
        InstitutionalPageEntity retrievedInstitutionalPage = institutionalPageHelper
                .retrieveLatestForMemberById(userId, institutionalPageId);
        // retrieve paragraph or throw
        // do this before calling institutionalPageModerationHandler.update, otherwise paragraphId will be null
        ParagraphEntity foundParagraph = retrievedInstitutionalPage.getParagraphs().stream()
                .filter(paragraph -> paragraph.getId().equals(paragraphId))
                .findFirst()
                .orElseThrow(() -> new ParagraphNotFoundException(paragraphId, institutionalPageId));
        // update paragraph
        paragraphMapper.updateEntity(foundParagraph, inputParagraphDto);
        // moderation process
        InstitutionalPageEntity institutionalPage = institutionalPageModerationHandler
                .update(userId, institutionalPageId, retrievedInstitutionalPage);
        // if page is being edited by someone else, then throws error
        UUID updateLockedBy = institutionalPage.getUpdateLockedBy();
        if (updateLockedBy == null || updateLockedBy.equals(userId)) {
            institutionalPage.setUpdateLockedBy(null);
        } else {
            throw new UpdateLockedException(institutionalPageId, updateLockedBy);
        }
        // save institutional page and return output dto
        InstitutionalPageEntity savedInstitutionalPage = institutionalPageHelper.save(userId, institutionalPage);
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, savedInstitutionalPage);
        return outputInstitutionalPageDto;
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputInstitutionalPageDto removeParagraph(
            @NonNull UUID userId,
            @NonNull UUID institutionalPageId,
            @NonNull UUID paragraphId
    ) {
        // 'retrieveLatestForMemberById' already checks membership, but we call 'verifyWpLeaderOrThrow' first
        // to return meaningful errors: if 'institutionalPageId' refers to a published page,
        // it's clearer to return "not a member" instead of "institutional page not found"
        memberHelper.verifyWpLeaderOrThrow(userId, institutionalPageId);
        // retrieve institutional page entity
        InstitutionalPageEntity retrievedInstitutionalPage = institutionalPageHelper
                .retrieveLatestForMemberById(userId, institutionalPageId);
        // remove paragraph or throw exception if not found
        // do this before calling institutionalPageModerationHandler.update, otherwise paragraphId will be null
        boolean paragraphRemoved = retrievedInstitutionalPage.getParagraphs()
                .removeIf(paragraph -> paragraph.getId().equals(paragraphId));
        if (!paragraphRemoved) {
            throw new ParagraphNotFoundException(paragraphId, institutionalPageId);
        }
        // moderation process
        InstitutionalPageEntity institutionalPage = institutionalPageModerationHandler
                .update(userId, institutionalPageId, retrievedInstitutionalPage);
        // if page is being edited by someone else, then throws error
        UUID updateLockedBy = institutionalPage.getUpdateLockedBy();
        if (updateLockedBy == null || updateLockedBy.equals(userId)) {
            institutionalPage.setUpdateLockedBy(null);
        } else {
            throw new UpdateLockedException(institutionalPageId, updateLockedBy);
        }
        // save institutional page and return output dto
        InstitutionalPageEntity savedInstitutionalPage = institutionalPageHelper.save(userId, institutionalPage);
        OutputInstitutionalPageDto outputInstitutionalPageDto = dtoBuilder.mapEntityToDto(userId, savedInstitutionalPage);
        return outputInstitutionalPageDto;
    }

}