package com.finconsgroup.itserr.marketplace.institutionalpage.dm.controller;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.api.InstitutionalPageApi;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputCreateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputSearchForMemberInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.InputUpdateInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.dto.OutputRequestUpdateDto;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.enums.InstitutionalPageView;
import com.finconsgroup.itserr.marketplace.institutionalpage.dm.service.InstitutionalPageService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.web.bind.annotation.RestController;

import java.util.Set;
import java.util.UUID;

/**
 * REST controller class for handling InstitutionalPage CRUD related API requests.
 *
 * <p>Implements the {@link InstitutionalPageApi} interface.</p>
 */
@Slf4j
@RequiredArgsConstructor
@RestController
public class InstitutionalPageController implements InstitutionalPageApi {

    private final InstitutionalPageService institutionalPageService;

    @Override
    public OutputInstitutionalPageDto createInstitutionalPage(
            final InputCreateInstitutionalPageDto inputCreateInstitutionalPageDto
    ) {
        log.debug("POST /api/v1/dm/institutional-page/institutional-pages, inputCreateInstitutionalPageDto: {}", inputCreateInstitutionalPageDto);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.institutionalPageService.createInstitutionalPage(userId, inputCreateInstitutionalPageDto);
    }

    @Override
    public Page<OutputInstitutionalPageDto> findAllInstitutionalPages(
            final InstitutionalPageView view,
            final boolean includePublishedAndNotMember,
            final Set<String> associationsToLoad,
            final int pageNumber,
            final int pageSize,
            final String sort,
            final Direction direction
    ) {
        log.debug("GET /api/v1/dm/institutional-page/institutional-pages, view: {}, includePublishedAndNotMember: {}", view, includePublishedAndNotMember);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return this.institutionalPageService.findAllInstitutionalPages(userId, view, includePublishedAndNotMember, associationsToLoad, pageable);
    }

    @Override
    public Page<OutputInstitutionalPageDto> search(
            final InstitutionalPageView view,
            final InputSearchForMemberInstitutionalPageDto inputSearchForMemberInstitutionalPageDto,
            final Set<String> associationsToLoad,
            final int pageNumber,
            final int pageSize,
            final String sort,
            final Direction direction
    ) {
        log.debug("POST /api/v1/dm/institutional-page/institutional-pages/search, view: {}, inputSearchForMemberInstitutionalPageDto: {}", view, inputSearchForMemberInstitutionalPageDto);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return institutionalPageService.searchInstitutionalPages(
                userId,
                view,
                associationsToLoad,
                pageable,
                inputSearchForMemberInstitutionalPageDto
        );
    }

    @Override
    public OutputInstitutionalPageDto findInstitutionalPageById(
            final UUID institutionalPageId,
            final InstitutionalPageView view,
            final boolean includePublishedAndNotMember
    ) {
        log.debug("GET /api/v1/dm/institutional-page/institutional-pages/{}, view: {}, includePublishedAndNotMember: {}", institutionalPageId, view, includePublishedAndNotMember);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.institutionalPageService.findInstitutionalPageById(userId, institutionalPageId, view, includePublishedAndNotMember);
    }

    @Override
    public OutputInstitutionalPageDto updateInstitutionalPageById(
            final UUID institutionalPageId,
            final InputUpdateInstitutionalPageDto inputUpdateInstitutionalPageDto
    ) {
        log.debug("PUT /api/v1/dm/institutional-page/institutional-pages/{}, inputUpdateInstitutionalPageDto: {}", institutionalPageId, inputUpdateInstitutionalPageDto);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.institutionalPageService.updateInstitutionalPageById(userId, institutionalPageId, inputUpdateInstitutionalPageDto);
    }

    @Override
    public OutputInstitutionalPageDto deleteInstitutionalPageById(final UUID institutionalPageId) {
        log.debug("DELETE /api/v1/dm/institutional-page/institutional-pages/{}", institutionalPageId);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.institutionalPageService.deleteInstitutionalPageById(userId, institutionalPageId);
    }

    @Override
    public Page<OutputInstitutionalPageDto> findInstitutionalPagesHierarchyByRootId(
            final UUID rootInstitutionalPageId,
            final Set<String> associationsToLoad,
            final int pageNumber,
            final int pageSize,
            final String sort,
            final Direction direction
    ) {
        log.debug("GET /api/v1/dm//institutional-page/institutional-pages/{institutionalPageId}/hierarchy");
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return institutionalPageService.findInstitutionalPagesHierarchyByRootId(
                userId,
                associationsToLoad,
                pageable,
                rootInstitutionalPageId
        );
    }

    @Override
    public OutputInstitutionalPageDto updateFolderId(
            final UUID institutionalPageId,
            final UUID newFolderId
    ) {
        log.debug("PUT /api/v1/dm/institutional-page/institutional-pages/{}/folders/{}", institutionalPageId, newFolderId);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.institutionalPageService.updateFolderId(userId, institutionalPageId, newFolderId);
    }

    @Override
    public OutputInstitutionalPageDto requestModerationInstitutionalPageById(final UUID institutionalPageId) {
        log.debug("PUT /api/v1/dm/institutional-page/institutional-pages/{}/request-moderation", institutionalPageId);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.institutionalPageService.requestModerationInstitutionalPage(userId, institutionalPageId);
    }

    @Override
    public OutputInstitutionalPageDto requestPublicationInstitutionalPageById(final UUID institutionalPageId) {
        log.debug("PUT /api/v1/dm/institutional-page/institutional-pages/{}/request-publication", institutionalPageId);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.institutionalPageService.requestPublicationInstitutionalPage(userId, institutionalPageId);
    }

    @Override
    public OutputRequestUpdateDto requestUpdateInstitutionalPageById(
            final UUID institutionalPageId
    ) {
        log.debug("PUT /api/v1/dm/institutional-page/institutional-pages/{}/request-update", institutionalPageId);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        return this.institutionalPageService.requestUpdateInstitutionalPageById(userId, institutionalPageId);
    }

    @Override
    public void cancelUpdateInstitutionalPageById(
            final UUID institutionalPageId
    ) {
        log.debug("PUT /api/v1/dm/institutional-page/institutional-pages/{}/cancel-update", institutionalPageId);
        UUID userId = JwtTokenHolder.getUserIdOrThrow();
        this.institutionalPageService.cancelUpdateInstitutionalPageById(userId, institutionalPageId);
    }

}
