package com.finconsgroup.itserr.marketplace.metrics.dm.api;

import com.finconsgroup.itserr.marketplace.metrics.dm.dto.InputCreateMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.MetricDtoType;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputMetricEventDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputMetricEventRefDto;
import com.finconsgroup.itserr.marketplace.metrics.dm.dto.OutputMetricsInterestWeightsConfigurationDto;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.security.SecurityRequirement;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import jakarta.validation.constraints.Positive;
import jakarta.validation.constraints.PositiveOrZero;
import org.springframework.core.io.Resource;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Sort;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;

import java.util.UUID;

/**
 * This interface defines the contract for REST API endpoints used for system testing and debugging.
 *
 * <p>
 * These endpoints are intended for internal use only and provide endpoints used for debugging and runtime analysis.
 * </p>
 *
 * <p>Example usage:
 * <pre>
 * GET /api/v1/bs/diagnostics/download-logs - Download application log for analysis
 * </pre>
 * </p>
 */
@Tag(
        name = "Diagnostics",
        description = "Diagnostics API: provides endpoints used for " +
                "debugging and runtime analysis."
)
@SecurityRequirement(name = "BearerAuth")
public interface DiagnosticsApi {

    /**
     * Downloads the application log file for analysis.
     *
     * @return ResponseEntity containing the log file as a Resource
     */
    @Operation(
            summary = "Download log file",
            responses = { @ApiResponse(responseCode = "200", description = "OK"), }
    )
    @GetMapping(path = "/metrics/diagnostics/download-logs", produces = MediaType.APPLICATION_OCTET_STREAM_VALUE)
    ResponseEntity<Resource> downloadLogs();

    /**
     * Creates a new metricEvent.
     *
     * @param metric the metric type
     * @param inputCreateMetricEventDto the input data transfer object containing metricEvent details
     * @return the created {@link OutputMetricEventDto} and HTTP status 201 (Created)
     */
    @Operation(
            summary = "create metricEvent",
            responses = { @ApiResponse(responseCode = "201", description = "Created") }
    )
    @PostMapping(
            value = "/metrics/diagnostics/metrics/{metricId}/events",
            consumes = MediaType.APPLICATION_JSON_VALUE,
            produces = MediaType.APPLICATION_JSON_VALUE
    )
    @ResponseStatus(HttpStatus.CREATED)
    OutputMetricEventDto createMetricEvent(
            @PathVariable("metricId") MetricDtoType metric,
            @Valid @RequestBody InputCreateMetricEventDto inputCreateMetricEventDto);

    /**
     * Retrieves a paginated list of all MetricEvents.
     *
     * @param metric the metric type
     * @param pageNumber the page number to retrieve (default is 0)
     * @param pageSize the number of MetricEvents per page (default is 10)
     * @param sort the field to sort by (default is "id")
     * @param direction the direction of sorting (default is ascending)
     * @return a page of {@link OutputMetricEventRefDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find all MetricEvents",
            responses = { @ApiResponse(responseCode = "200", description = "OK") }
    )
    @GetMapping(value = "/metrics/diagnostics/metrics/{metricId}/events", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    Page<OutputMetricEventRefDto> findAll(
            @PathVariable("metricId") MetricDtoType metric,
            @RequestParam(name = "pageNumber", defaultValue = "0", required = false) @PositiveOrZero int pageNumber,
            @RequestParam(name = "pageSize", defaultValue = "10", required = false) @Positive int pageSize,
            @RequestParam(name = "sort", defaultValue = "id", required = false) String sort,
            @RequestParam(name = "direction", defaultValue = "ASC", required = false) Sort.Direction direction
    );

    /**
     * Retrieves a metricEvent by id.
     *
     * @param metric the metric type
     * @param metricEventId the id of the metricEvent to retrieve
     * @return the found {@link OutputMetricEventDto} and HTTP status 200 (OK)
     */
    @Operation(
            summary = "find metricEvent by id",
            responses = {
                    @ApiResponse(responseCode = "200", description = "OK"),
                    @ApiResponse(responseCode = "404", description = "Not Found"),
            }
    )
    @GetMapping(value = "/metrics/diagnostics/metrics/{metricId}/events/{metricEventId}", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputMetricEventDto findById(
            @PathVariable("metricId") MetricDtoType metric,
            @PathVariable("metricEventId") UUID metricEventId);

    /**
     * Forces an update of the metrics data to ensure it reflects the most current state.
     *
     * @param sync if true, the update will be performed synchronously
     */
    @Operation(
            summary = "Force update metrics",
            responses = {
                    @ApiResponse(responseCode = "202", description = "Accepted (sync=false, will be processed soon)"),
                    @ApiResponse(responseCode = "204", description = "Done (sync=true)"),
            }
    )
    @PutMapping(value = "/metrics/diagnostics/metrics")
    ResponseEntity<Void> updateMetrics(@RequestParam(name = "sync", required = false, defaultValue = "false") boolean sync);

    /**
     * Retrieves the metrics interest weights configuration.
     *
     * @return an {@link OutputMetricsInterestWeightsConfigurationDto} containing the configuration of metrics interest weights as a mapping of metric
     *         identifiers to weights
     */
    @Operation(
            summary = "Retrieve metrics interest weight configuration",
            responses = { @ApiResponse(responseCode = "200", description = "OK") }
    )
    @GetMapping(value = "/metrics/diagnostics/config/metrics-weights", produces = MediaType.APPLICATION_JSON_VALUE)
    @ResponseStatus(HttpStatus.OK)
    OutputMetricsInterestWeightsConfigurationDto getInterestWeights();

}
