package com.finconsgroup.itserr.marketplace.metrics.dm.config;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.task.TaskExecutor;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.annotation.EnableScheduling;
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler;

/**
 * Configuration class for setting up materialized view updates.
 */
@Configuration
@EnableScheduling
@EnableAsync
@RequiredArgsConstructor
public class MVUpdateConfiguration {

    /** Bean name for the materialized view update scheduler */
    public static final String MV_UPDATE_SCHEDULER = "mvUpdateScheduler";
    /** Bean name for the materialized view update executor */
    public static final String MV_UPDATE_EXECUTOR = "mvUpdateExecutor";

    /**
     * Creates and configures a {@link ThreadPoolTaskScheduler} for materialized view updates.
     *
     * @return configured ThreadPoolTaskScheduler instance
     */
    @Bean(name = MV_UPDATE_SCHEDULER)
    public ThreadPoolTaskScheduler mvUpdateScheduler() {
        ThreadPoolTaskScheduler scheduler = new ThreadPoolTaskScheduler();
        scheduler.setPoolSize(1);
        scheduler.setRemoveOnCancelPolicy(true);
        scheduler.initialize();
        scheduler.setThreadFactory(runnable -> new Thread(runnable, "mvUpdate"));
        return scheduler;
    }

    /**
     * Creates a {@link TaskExecutor} using the provided scheduler.
     *
     * @param singleThreadScheduler the scheduler to be used as executor
     * @return TaskExecutor instance
     */
    @Bean(name = MV_UPDATE_EXECUTOR)
    public TaskExecutor mvUpdateExecutor(
            @Qualifier(MV_UPDATE_SCHEDULER) final ThreadPoolTaskScheduler singleThreadScheduler) {
        return singleThreadScheduler;
    }

}
