/**
 * Global exception handler for REST controllers in the metrics system. This class handles custom exceptions related to metricEvent and category operations and
 * returns appropriate HTTP error responses.
 *
 * <p>Example usage:
 * <pre>
 * When a MetricEventNotFoundException is thrown, this handler returns a 404 NOT FOUND response.
 * </pre>
 * </p>
 */
package com.finconsgroup.itserr.marketplace.metrics.dm.exception;

import com.finconsgroup.itserr.marketplace.core.web.exception.ErrorResponseDto;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.context.request.WebRequest;

import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.createResponseEntityError;
import static com.finconsgroup.itserr.marketplace.core.web.exception.ExceptionUtils.logError;
import static org.springframework.http.HttpStatus.CONFLICT;
import static org.springframework.http.HttpStatus.NOT_FOUND;

/**
 * Custom Exception Handler.
 */
@Slf4j
@RestControllerAdvice
public class CustomExceptionHandler {

    /**
     * Handles MetricEventNotFoundException and returns a 404 NOT FOUND error response.
     *
     * @param ex the MetricEventNotFoundException that was thrown
     * @param request the current web request
     * @return a ResponseEntity containing the error response DTO and HTTP status 404
     */
    @ExceptionHandler({ MetricEventNotFoundException.class })
    public ResponseEntity<ErrorResponseDto> handleMetricEventNotFoundException(
            final MetricEventNotFoundException ex,
            final WebRequest request) {
        logError(ex, request);
        return createResponseEntityError(NOT_FOUND, ex.getMessage());
    }

    /**
     * Handles {@link MetricEventConflictException} and returns a 409 CONFLICT error response.
     *
     * @param ex the {@link MetricEventConflictException} thrown due to idempotency conflict
     * @return a {@link ResponseEntity} containing the error response DTO and HTTP status 409 CONFLICT
     */
    @ExceptionHandler({ MetricEventConflictException.class })
    public ResponseEntity<ErrorResponseDto> handleMetricEventConflictException(
            final MetricEventConflictException ex) {
        log.debug("Metric event refused because of conflict");
        log.trace("Exception stacktrace: ", ex);
        return createResponseEntityError(CONFLICT, "Another metric event with the same resource and author already exists");
    }

}
