package com.finconsgroup.itserr.marketplace.search.dm.controller;

import com.finconsgroup.itserr.marketplace.search.dm.api.CatalogApi;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputCatalogDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputFavouriteSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputCatalogDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputCatalogFavouriteDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputCatalogLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.service.CatalogService;
import com.finconsgroup.itserr.marketplace.search.dm.util.SortUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Map;

/**
 * Spring Boot REST Controller implementation for the {@link CatalogApi} endpoints.
 */
@RestController
@RequiredArgsConstructor
@Slf4j
public class CatalogController implements CatalogApi {
    private final CatalogService catalogService;

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        // trim strings before triggering validation, but do not convert to null
        // as otherwise it throws - MissingServletRequestParameterException
        binder.registerCustomEditor(String.class, new StringTrimmerEditor(false));
    }

    @NonNull
    @Override
    public OutputCatalogDto upsertCatalog(String id, InputCatalogDto dto) {
        return catalogService.upsertDocument(dto);
    }

    @NonNull
    @Override
    public OutputCatalogDto getCatalog(String id) {
        return catalogService.getDocument(id);
    }

    @Override
    public void deleteCatalog(String id) {
        catalogService.deleteDocument(id);
    }

    @Override
    public void deleteAll() {
        catalogService.deleteAll();
    }

    @NonNull
    @Override
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        return catalogService.getAutoCompletions(terms);
    }

    @NonNull
    @Override
    public Page<OutputCatalogLocalSearchDto> getLocalSearch(@Nullable String terms,
                                                            @Nullable String filters,
                                                            int pageNumber,
                                                            int pageSize,
                                                            String sort,
                                                            Sort.Direction direction,
                                                            String multiSort) {
        Sort sortToApply = SortUtils.buildSort(sort, direction, multiSort, ":", Map.of());
        Pageable pageable = PageRequest.of(pageNumber, pageSize, sortToApply);
        return catalogService.getLocalSearch(terms, filters, pageable);
    }

    @Override
    public Page<OutputCatalogFavouriteDto> getFavourite(InputFavouriteSearchDto dto, String filters, int pageNumber, int pageSize, String sort, Sort.Direction direction) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize, Sort.by(direction, sort));
        return catalogService.getFavourite(dto.getIds(), filters, pageable);
    }
}
