package com.finconsgroup.itserr.marketplace.search.dm.controller;

import com.finconsgroup.itserr.marketplace.search.dm.api.UserProfileApi;
import com.finconsgroup.itserr.marketplace.search.dm.dto.InputUserProfileDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputGlobalSearchAutoCompleteDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputUserProfileDto;
import com.finconsgroup.itserr.marketplace.search.dm.dto.OutputUserProfileLocalSearchDto;
import com.finconsgroup.itserr.marketplace.search.dm.service.UserProfileService;
import com.finconsgroup.itserr.marketplace.search.dm.util.SortUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.lang.Nullable;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;
import java.util.Map;

/**
 * Spring Boot REST Controller implementation for the {@link UserProfileApi} endpoints.
 */
@RestController
@RequiredArgsConstructor
@Slf4j
public class UserProfileController implements UserProfileApi {
    private final UserProfileService userProfileService;

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        // trim strings before triggering validation, but do not convert to null
        // as otherwise it throws - MissingServletRequestParameterException
        binder.registerCustomEditor(String.class, new StringTrimmerEditor(false));
    }

    @NonNull
    @Override
    public OutputUserProfileDto upsertUserProfile(String id, InputUserProfileDto dto) {
        return userProfileService.upsertDocument(dto);
    }

    @NonNull
    @Override
    public OutputUserProfileDto getUserProfile(String id) {
        return userProfileService.getDocument(id);
    }

    @Override
    public void deleteUserProfile(String id) {
        userProfileService.deleteDocument(id);
    }

    @Override
    public void deleteAll() {
        userProfileService.deleteAll();
    }

    @NonNull
    @Override
    public List<OutputGlobalSearchAutoCompleteDto> getAutoCompletions(@NonNull String terms) {
        return userProfileService.getAutoCompletions(terms);
    }

    @NonNull
    @Override
    public Page<OutputUserProfileLocalSearchDto> getLocalSearch(@Nullable String terms,
                                                                @Nullable String filters,
                                                                int pageNumber,
                                                                int pageSize,
                                                                String sort,
                                                                Sort.Direction direction,
                                                                String multiSort) {

        Sort sortToApply = SortUtils.buildSort(sort, direction, multiSort, ":", Map.of());
        Pageable pageable = PageRequest.of(pageNumber, pageSize, sortToApply);
        return userProfileService.getLocalSearch(terms, filters, pageable);
    }
}
