package com.finconsgroup.itserr.marketplace.search.dm.service;

import com.finconsgroup.itserr.marketplace.search.dm.bean.QuerySearchFields;
import com.finconsgroup.itserr.marketplace.search.dm.bean.SearchRequest;
import com.finconsgroup.itserr.marketplace.search.dm.config.SearchProperties;
import com.finconsgroup.itserr.marketplace.search.dm.entity.RegisteredAuthor;
import com.finconsgroup.itserr.marketplace.search.dm.entity.UserProfile;
import com.finconsgroup.itserr.marketplace.search.dm.entity.UserProfileMinimal;
import com.finconsgroup.itserr.marketplace.search.dm.enums.AssociationCategory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;

import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * Service to sync documents when a user profile has been updated.
 *
 * @param <T>
 */
public interface SyncUserProfileService<T> {

    /**
     * Synchronize the related search documents whenever a user profile is updated.
     *
     * @param entity      the document to sync
     * @param userProfile the user profile that is updated
     */
    void syncDocumentForUserProfile(T entity, UserProfile userProfile);

    /**
     * Performs search for the document index for {@code T} based on the terms and filters, and
     * returns the results as page of dto {@code R}
     *
     * @param searchRequest the search request to search for
     * @param pageable      the page to return
     * @return the page of {@code T} of matching documents
     */
    @NonNull
    Page<T> searchDocumentsForUserProfileSync(@NonNull SearchRequest searchRequest,
                                              @NonNull Pageable pageable);

    /**
     * Returns the search properties for related configuration
     *
     * @return SearchProperties
     */
    @NonNull
    SearchProperties getSearchProperties();

    /**
     * Searches for any documents that contain user profile id in any of the documents.
     * And then syncs the document for the updated user profile.
     *
     * @param userProfile the updated {@link UserProfile}
     */
    default void syncDocumentsForUserProfile(UserProfile userProfile) {
        if (getSearchProperties().syncList() == null) {
            return;
        }

        Optional<SearchProperties.SyncProperties> syncPropertiesOptional = getSearchProperties()
                .syncList().stream()
                .filter(sp -> AssociationCategory.PEOPLE.equals(
                        sp.associationCategory()))
                .findFirst();
        if (syncPropertiesOptional.isEmpty()) {
            return;
        }

        QuerySearchFields querySearchFields = QuerySearchFields
                .builder()
                .fieldType(QuerySearchFields.TYPE_KEYWORD)
                .fieldNames(syncPropertiesOptional.get().foreignKeyProperties())
                .build();
        SearchRequest searchRequest = SearchRequest
                .builder()
                .terms(userProfile.getId())
                .termSearchFields(List.of(querySearchFields))
                .build();

        int page = 0, pageSize = syncPropertiesOptional.get().fetchPageSize();
        while (true) {
            Pageable pageable = PageRequest.of(page, pageSize);
            Page<T> entityPage = searchDocumentsForUserProfileSync(searchRequest, pageable);
            if (entityPage.isEmpty()) {
                break;
            }

            entityPage.forEach(entity -> syncDocumentForUserProfile(entity, userProfile));

            page++;
            if (entityPage.getNumberOfElements() < pageSize || page >= entityPage.getTotalPages()) {
                break;
            }
        }
    }

    /**
     * Checks if the user profile is updated after the last time the minimal user profile was created from it.
     *
     * @param userProfileMinimal the old {@link UserProfileMinimal}
     * @param userProfile        the new {@link UserProfile}
     * @return {@code true}, if updated, {@code false} otherwise.
     */
    default boolean isUserProfileMinimalUpdated(@NonNull UserProfileMinimal userProfileMinimal,
                                                @NonNull UserProfile userProfile) {
        return !Objects.equals(userProfile.getFirstName(), userProfileMinimal.getFirstName())
                || !Objects.equals(userProfile.getLastName(), userProfileMinimal.getLastName())
                || !Objects.equals(userProfile.getEmail(), userProfileMinimal.getEmail())
                || !Objects.equals(userProfile.getOrcid(), userProfileMinimal.getOrcid());
    }

    /**
     * Updates the minimal user profile from the user profile.
     *
     * @param userProfileMinimal the {@link UserProfileMinimal} to update
     * @param userProfile        the new {@link UserProfile}
     */
    default void updateUserProfileMinimal(@NonNull UserProfileMinimal userProfileMinimal,
                                          @NonNull UserProfile userProfile) {
        userProfileMinimal.setFirstName(userProfile.getFirstName());
        userProfileMinimal.setLastName(userProfile.getLastName());
        userProfileMinimal.setEmail(userProfile.getEmail());
        userProfileMinimal.setOrcid(userProfile.getOrcid());
    }

    /**
     * Checks if the user profile is updated after the last time the registered author was created from it.
     *
     * @param registeredAuthor the old {@link RegisteredAuthor}
     * @param userProfile      the new {@link UserProfile}
     * @return {@code true}, if updated, {@code false} otherwise.
     */
    default boolean isRegisteredAuthorUserProfileUpdated(@NonNull RegisteredAuthor registeredAuthor,
                                                         @NonNull UserProfile userProfile) {
        return !Objects.equals(registeredAuthor.getFirstName(), userProfile.getFirstName())
                || !Objects.equals(registeredAuthor.getLastName(), userProfile.getLastName())
                || !Objects.equals(registeredAuthor.getEmail(), userProfile.getEmail())
                || !Objects.equals(registeredAuthor.getOrcid(), userProfile.getOrcid())
                || !Objects.equals(registeredAuthor.getLocation(), userProfile.getWorkingLocation());
    }

    /**
     * Updates the registered author user profile from the user profile.
     *
     * @param registeredAuthor the {@link RegisteredAuthor} to update
     * @param userProfile      the new {@link UserProfile}
     */
    default void updateRegisteredAuthorUserProfile(@NonNull RegisteredAuthor registeredAuthor,
                                                   @NonNull UserProfile userProfile) {
        registeredAuthor.setFirstName(userProfile.getFirstName());
        registeredAuthor.setLastName(userProfile.getLastName());
        registeredAuthor.setEmail(userProfile.getEmail());
        registeredAuthor.setOrcid(userProfile.getOrcid());
        registeredAuthor.setLocation(userProfile.getWorkingLocation());
    }
}
