package com.finconsgroup.itserr.marketplace.favouritesearch.dm.service;

import com.finconsgroup.itserr.marketplace.favouritesearch.dm.dto.InputCreateFavouriteSearchDto;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.dto.OutputFavouriteSearchDto;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.enums.SearchContext;
import com.finconsgroup.itserr.marketplace.favouritesearch.dm.exception.FavouriteSearchNotFoundException;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;

import java.util.UUID;

/**
 * Service for handling business logic related to FavouriteSearches.
 * This interface provides methods to retrieve, create, and delete FavouriteSearches,
 * as well as retrieve FavouriteSearches by search text and user id.
 *
 * <p>Example usage:
 * <pre>
 * Page<OutputFavouriteSearchDto> favouriteSearches =
 *     favouriteSearchService.findAll(userId, "Godfath", pageable);
 * </pre>
 * </p>
 */
public interface FavouriteSearchService {

    /**
     * Creates a new favouriteSearch for the authenticated user using the provided input data.
     *
     * @param userId                        the unique identifier of the user
     * @param inputCreateFavouriteSearchDto the input DTO containing favouriteSearch details
     * @return an OutputFavouriteSearchDto representing the created favouriteSearch
     */
    @NonNull
    OutputFavouriteSearchDto create(
            @NonNull UUID userId,
            @NonNull InputCreateFavouriteSearchDto inputCreateFavouriteSearchDto
    );

    /**
     * Retrieves a paginated list of all FavouriteSearches for the authenticated user.
     *
     * @param userId     the unique identifier of the user
     * @param context    the text to filter FavouriteSearches by context
     * @param searchText the text to filter FavouriteSearches by
     * @param pageable   the pagination information
     * @return a page of OutputFavouriteSearchDto representing the FavouriteSearches
     */
    @NonNull
    Page<OutputFavouriteSearchDto> findAll(
            @NonNull UUID userId,
            SearchContext context,
            @NonNull String searchText,
            @NonNull Pageable pageable
    );

    /**
     * Retrieves a favouriteSearch by id for the authenticated user.
     *
     * @param userId            the unique identifier of the user
     * @param favouriteSearchId the id of the favouriteSearch to retrieve
     * @return an OutputFavouriteSearchDto representing the favouriteSearch details
     * @throws FavouriteSearchNotFoundException if no favouriteSearch is found with the given id
     */
    @NonNull
    OutputFavouriteSearchDto findById(@NonNull UUID userId, @NonNull UUID favouriteSearchId) throws FavouriteSearchNotFoundException;

    /**
     * Deletes a favouriteSearch by id for the authenticated user.
     *
     * @param userId            the unique identifier of the user
     * @param favouriteSearchId the id of the favouriteSearch to delete
     * @throws FavouriteSearchNotFoundException if no favouriteSearch is found with the given id
     */
    void deleteById(@NonNull UUID userId, @NonNull UUID favouriteSearchId) throws FavouriteSearchNotFoundException;

}
