package com.finconsgroup.itserr.marketplace.favourite.user.bs.config;


import com.finconsgroup.itserr.marketplace.core.web.bean.FilterProperties;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.enums.ItemContext;
import jakarta.validation.constraints.NotBlank;
import jakarta.validation.constraints.NotNull;
import lombok.Builder;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.lang.NonNull;

import java.util.Map;
import java.util.Optional;

/**
 * It contains configuration for Favourite User Bs service
 *
 * @param swaggerApiServerUrl swagger api server url
 * @param context the context specific properties map
 * @param filter the filter properties
 * @param messaging the messaging properties
 */
@ConfigurationProperties(prefix = "favourite-user.bs")
@Builder
public record FavouriteUserBsProperties(@NotBlank String swaggerApiServerUrl,
                                        @NotNull Map<String, ContextProperties> context,
                                        @NotNull FilterProperties filter,
                                        @NotNull MessagingProperties messaging) {

    /**
     * It contains configuration for Favourite User Item for each context
     *
     * @param filterEnabled the flag to indicate if filter should be enabled
     */
    @Builder
    public record ContextProperties(boolean filterEnabled) {

    }


    /**
     * Map the context key to {@link ItemContext}
     *
     * @param context the context key
     * @return the mapped {@link ItemContext}
     */
    @NonNull
    public Optional<ItemContext> mapToItemContext(String context) {
        ItemContext itemContext = switch (context) {
            case "catalog" -> ItemContext.CATALOG;
            case "institutional-page" -> ItemContext.INSTITUTIONAL_PAGE;
            case "search" -> ItemContext.SEARCH;
            case "label" -> ItemContext.LABEL;
            case "people" -> ItemContext.PEOPLE;
            default -> null;
        };

        return Optional.ofNullable(itemContext);
    }

    /**
     * Map the {@link ItemContext} to context key
     *
     * @param itemContext the item context
     * @return the mapped context key
     */
    @NonNull
    public Optional<String> mapToContextKey(ItemContext itemContext) {
        String contextKey = switch (itemContext) {
            case ItemContext.CATALOG -> "catalog";
            case ItemContext.INSTITUTIONAL_PAGE -> "institutional-page";
            case ItemContext.SEARCH -> "search";
            case ItemContext.LABEL -> "label";
            case ItemContext.PEOPLE -> "people";
        };

        return Optional.of(contextKey);
    }
}
