package com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.impl;

import com.finconsgroup.itserr.marketplace.favourite.user.bs.enums.ItemContext;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.EventProducer;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.config.properties.MessagingConfigurationProperties;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.dto.ResourceByFolloweeMessageBodyDto;
import com.finconsgroup.itserr.marketplace.favourite.user.bs.messaging.dto.FavouriteItemMessageBodyDto;
import com.finconsgroup.itserr.messaging.exception.WP2MessagingException;
import com.finconsgroup.itserr.messaging.producer.MessageProducer;
import com.finconsgroup.itserr.messaging.producer.ProducerRegistry;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Component;

import java.util.Map;

import static com.finconsgroup.itserr.marketplace.favourite.user.bs.util.Constants.SERVICE_FILTER_KEY;
import static com.finconsgroup.itserr.marketplace.favourite.user.bs.util.Constants.SERVICE_FILTER_VALUE;

/**
 * Default implementation of {@link EventProducer} that publishes resource events (create, update, delete).
 */
@Slf4j
@Component
@ConditionalOnProperty(name = "messaging.enabled", havingValue = "true")
public class DefaultEventProducer implements EventProducer {

    private final MessageProducer messageProducer;

    private final MessagingConfigurationProperties messagingProperties;

    /**
     * Constructs the DefaultEventProducer.
     *
     * @param messagingConfigurationProperties application-specific configuration properties
     * @param producerRegistry the registry providing message producers
     */
    public DefaultEventProducer(
            MessagingConfigurationProperties messagingConfigurationProperties,
            ProducerRegistry producerRegistry
    ) {
        this.messagingProperties = messagingConfigurationProperties;
        this.messageProducer = producerRegistry.getMessageProducer(this.messagingProperties.getEventProducerName());
    }

    @Override
    public void publishFavouriteItemCreatedEvent(@NonNull FavouriteItemMessageBodyDto favouriteItem) {
        publish(favouriteItem.getId(), favouriteItem, messagingProperties.getFavouriteItemCreatedEventType());
    }

    @Override
    public void publishFavouriteItemUpdatedEvent(@NonNull FavouriteItemMessageBodyDto favouriteItem) {
        publish(favouriteItem.getId(), favouriteItem, messagingProperties.getFavouriteItemUpdatedEventType());
    }

    @Override
    public void publishFavouriteItemDeletedEvent(@NonNull FavouriteItemMessageBodyDto favouriteItem) {
        publish(favouriteItem.getId(), favouriteItem, messagingProperties.getFavouriteItemDeletedEventType());
    }

    @Override
    public void publishResourceCreatedByFolloweeEvent(@NonNull ResourceByFolloweeMessageBodyDto resourceByFollowee) {
        String itemContext = resourceByFollowee.getAdditionalData().getItemContext();
        String eventType = null;
        if (ItemContext.CATALOG.getId().equalsIgnoreCase(itemContext)) {
            eventType = messagingProperties.getFavouriteItemFolloweeCatalogCreatedEventType();
        } else if (ItemContext.INSTITUTIONAL_PAGE.getId().equalsIgnoreCase(itemContext)) {
            eventType = messagingProperties.getFavouriteItemFolloweeInstitutionalPageCreatedEventType();
        } else if (ItemContext.EVENT_ID.equalsIgnoreCase(itemContext)) {
            eventType = messagingProperties.getFavouriteItemFolloweeEventCreatedEventType();
        } else if (ItemContext.NEWS_ID.equalsIgnoreCase(itemContext)) {
            eventType = messagingProperties.getFavouriteItemFolloweeNewsCreatedEventType();
        }
        if (StringUtils.isNotBlank(eventType)) {
            publish(resourceByFollowee.getId(), resourceByFollowee, eventType);
        } else {
            log.info("Item context not supported to publish the resource created by followee event - {}", itemContext);
        }
    }

    /**
     * Sends a message to the messaging system.
     *
     * @param resourceId the resource id
     * @param resource the payload of the message
     * @param eventType the type of event to publish
     * @throws WP2MessagingException if message publishing fails
     */
    private <T> void publish(
            @NonNull String resourceId,
            @NonNull T resource,
            @NonNull String eventType) {
        try {
            Map<String, T> messageMap = Map.of(resourceId, resource);
            Map<String, String> filterProperties = Map.of(SERVICE_FILTER_KEY, SERVICE_FILTER_VALUE);
            log.debug("Sending event message, messageMap:{}, eventType: {}, source: {}, filterProperties: {}",
                    messageMap, eventType, messagingProperties.getSource(), filterProperties);
            messageProducer.publishEvent(messageMap, eventType, messagingProperties.getSource(), filterProperties);
            log.info("Successfully published event message for resource with id: {}", resourceId);
        } catch (Exception e) {
            String errorMessage = "Error occurred while sending event message: %s".formatted(e.getMessage());
            throw new WP2MessagingException(errorMessage, e);
        }
    }

}
