package com.finconsgroup.itserr.marketplace.favourite.user.dm.controller;

import com.finconsgroup.itserr.marketplace.core.web.security.jwt.JwtTokenHolder;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.api.FavouriteUserItemApi;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.dto.InputCreateFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.dto.InputPatchFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.dto.OutputFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.dto.OutputFavouriteUserItemSubscriberDto;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.enums.ItemContext;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.service.FavouriteUserItemService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.propertyeditors.StringTrimmerEditor;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.lang.NonNull;
import org.springframework.web.bind.WebDataBinder;
import org.springframework.web.bind.annotation.InitBinder;
import org.springframework.web.bind.annotation.RestController;

import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * Spring Boot REST Controller implementation for the {@link FavouriteUserItemApi} endpoints.
 */
@RestController
@RequiredArgsConstructor
@Slf4j
public class FavouriteUserItemController implements FavouriteUserItemApi {

    private final FavouriteUserItemService favouriteUserItemService;

    @InitBinder
    public void initBinder(WebDataBinder binder) {
        // trim strings before triggering validation, but do not convert to null
        // as otherwise it throws - MissingServletRequestParameterException
        binder.registerCustomEditor(String.class, new StringTrimmerEditor(false));
    }

    @NonNull
    @Override
    public OutputFavouriteUserItemDto create(InputCreateFavouriteUserItemDto dto) {
        return favouriteUserItemService.create(JwtTokenHolder.getUserIdOrThrow(), dto);
    }

    @NonNull
    @Override
    public OutputFavouriteUserItemDto get(UUID favouriteUserItemId) {
        return favouriteUserItemService.findById(JwtTokenHolder.getUserIdOrThrow(), favouriteUserItemId);
    }

    @Override
    public void delete(UUID favouriteUserItemId) {
        favouriteUserItemService.deleteById(JwtTokenHolder.getUserIdOrThrow(), favouriteUserItemId);
    }

    @Override
    public OutputFavouriteUserItemDto patch(UUID favouriteUserItemId, InputPatchFavouriteUserItemDto dto) {
        return favouriteUserItemService.patchById(JwtTokenHolder.getUserIdOrThrow(), favouriteUserItemId, dto);
    }

    @NonNull
    @Override
    public Page<OutputFavouriteUserItemDto> findByContext(ItemContext context, String subContext, int pageNumber,
                                                          int pageSize, String sort, Sort.Direction direction) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return favouriteUserItemService.findByContext(JwtTokenHolder.getUserIdOrThrow(), context, subContext, pageable);
    }

    @Override
    public List<OutputFavouriteUserItemDto> findByContextAndItemIds(ItemContext context, String subContext,
                                                                    String itemIds, String itemIdSeparator) {
        Set<String> itemIdSet = new LinkedHashSet<>(Arrays.asList(itemIds.split(itemIdSeparator)));
        return favouriteUserItemService.findByContextAndItemIds(JwtTokenHolder.getUserIdOrThrow(), context, subContext, itemIdSet);
    }

    @Override
    public Page<OutputFavouriteUserItemSubscriberDto> findSubscribers(ItemContext context, String subContext, String itemId,
                                                                      int pageNumber, int pageSize, String sort,
                                                                      Sort.Direction direction) {
        Pageable pageable = PageRequest.of(pageNumber, pageSize, direction, sort);
        return favouriteUserItemService.findSubscribers(context, subContext, itemId, pageable);
    }

    @Override
    public List<OutputFavouriteUserItemDto> deleteByContextAndItemId(ItemContext context, String subContext, String itemId) {
        return favouriteUserItemService.deleteByContextAndItemId(context, subContext, itemId);
    }
}
