package com.finconsgroup.itserr.marketplace.favourite.user.dm.service.impl;

import com.finconsgroup.itserr.marketplace.core.web.exception.WP2DuplicateResourceException;
import com.finconsgroup.itserr.marketplace.core.web.exception.WP2ResourceNotFoundException;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.dto.InputCreateFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.dto.InputPatchFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.dto.OutputFavouriteUserItemDto;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.dto.OutputFavouriteUserItemSubscriberDto;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.entity.ArchivedFavouriteUserItemEntity;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.entity.FavouriteUserItemEntity;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.enums.ItemContext;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.mapper.ArchivedFavouriteUserItemMapper;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.mapper.FavouriteUserItemMapper;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.repository.ArchivedFavouriteUserItemRepository;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.repository.FavouriteUserItemRepository;
import com.finconsgroup.itserr.marketplace.favourite.user.dm.service.FavouriteUserItemService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.lang.NonNull;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.UUID;

/**
 * Default implementation of {@link FavouriteUserItemService} to perform operations related to favourite user item resources.
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class DefaultFavouriteUserItemService implements FavouriteUserItemService {

    private static final String FORMAT_BUSINESS_KEY = "{context: %s, itemId: %s}";

    private final FavouriteUserItemRepository favouriteUserItemRepository;
    private final FavouriteUserItemMapper favouriteUserItemMapper;
    private final ArchivedFavouriteUserItemRepository archivedFavouriteUserItemRepository;
    private final ArchivedFavouriteUserItemMapper archivedFavouriteUserItemMapper;

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputFavouriteUserItemDto create(@NonNull UUID userId, @NonNull InputCreateFavouriteUserItemDto dto) {
        // Checks that there is no favourite user item with same context and item id
        if (favouriteUserItemRepository.countByUserIdAndContextAndItemId(userId, dto.getContext(), dto.getItemId()) > 0) {
            throw new WP2DuplicateResourceException(FORMAT_BUSINESS_KEY.formatted(dto.getContext(), dto.getItemId()));
        }

        final FavouriteUserItemEntity favouriteUserItemEntity = favouriteUserItemMapper.createDtoToEntity(dto, userId);
        return saveAndReturnDto(favouriteUserItemEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public OutputFavouriteUserItemDto findById(@NonNull UUID userId, @NonNull UUID favouriteUserItemId) {
        FavouriteUserItemEntity favouriteUserItemEntity = findByIdOrThrow(userId, favouriteUserItemId);
        return favouriteUserItemMapper.toOutputDto(favouriteUserItemEntity);
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public void deleteById(@NonNull UUID userId, @NonNull UUID favouriteUserItemId) {
        FavouriteUserItemEntity favouriteUserItemEntity = findByIdOrThrow(userId, favouriteUserItemId);
        deleteEntity(favouriteUserItemEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public OutputFavouriteUserItemDto patchById(@NonNull UUID userId,
                                                @NonNull UUID favouriteUserItemId,
                                                @NonNull InputPatchFavouriteUserItemDto dto) {
        FavouriteUserItemEntity favouriteUserItemEntity = findByIdOrThrow(userId, favouriteUserItemId);
        favouriteUserItemMapper.patchDtoToEntity(favouriteUserItemEntity, dto);
        return saveAndReturnDto(favouriteUserItemEntity);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputFavouriteUserItemDto> findByContext(@NonNull UUID userId, @NonNull ItemContext context,
                                                          String subContext, @NonNull Pageable pageable) {
        Page<FavouriteUserItemEntity> favouriteUserItemEntitiesResult;
        if (StringUtils.isNotBlank(subContext)) {
            favouriteUserItemEntitiesResult = favouriteUserItemRepository.findByUserIdAndContextAndSubContext(
                userId, context, subContext, pageable);
        } else {
            favouriteUserItemEntitiesResult = favouriteUserItemRepository.findByUserIdAndContext(
                userId, context, pageable);
        }
        return favouriteUserItemEntitiesResult.map(favouriteUserItemMapper::toOutputDto);
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public List<OutputFavouriteUserItemDto> findByContextAndItemIds(@NonNull UUID userId, @NonNull ItemContext context,
                                                                    String subContext, @NonNull Set<String> itemIds) {
        if (itemIds.isEmpty()) {
            return List.of();
        }

        List<FavouriteUserItemEntity> favouriteUserItemEntities;
        if (StringUtils.isNotBlank(subContext)) {
            favouriteUserItemEntities = favouriteUserItemRepository.findByUserIdAndContextAndSubContextAndItemIdIn(
                userId, context, subContext, itemIds);
        } else {
            favouriteUserItemEntities = favouriteUserItemRepository.findByUserIdAndContextAndItemIdIn(
                userId, context, itemIds);
        }
        return favouriteUserItemEntities.stream().map(favouriteUserItemMapper::toOutputDto).toList();
    }

    @NonNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED, readOnly = true, noRollbackFor = Exception.class)
    public Page<OutputFavouriteUserItemSubscriberDto> findSubscribers(
            @NonNull ItemContext context, String subContext, @NonNull String itemId, @NonNull Pageable pageable) {

        String subContextFilter = StringUtils.isBlank(subContext) ? null : subContext;
        return favouriteUserItemRepository.findByContextAndItemIdAndFollowedIsTrueAndOptionalSubContext(
                context, itemId, subContextFilter, pageable
        ).map(favouriteUserItemMapper::toOutputSubscriberDto);
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED, rollbackFor = Exception.class)
    public List<OutputFavouriteUserItemDto> deleteByContextAndItemId(@NonNull ItemContext context,
                                                                     String subContext,
                                                                     @NonNull String itemId) {
        String subContextFilter = StringUtils.isBlank(subContext) ? null : subContext;
        List<OutputFavouriteUserItemDto> deletedFavouriteItemDtos = new LinkedList<>();
        favouriteUserItemRepository
                .findAllByContextAndItemIdAndOptionalSubContext(context, itemId, subContextFilter)
                .forEach(entity -> {
                    deletedFavouriteItemDtos.add(favouriteUserItemMapper.toOutputDto(entity));
                    deleteEntity(entity);
                });
        return deletedFavouriteItemDtos;
    }

    private FavouriteUserItemEntity findByIdOrThrow(@NonNull UUID userId, @NonNull UUID favouriteUserItemId) {
        return favouriteUserItemRepository
            .findByUserIdAndId(userId, favouriteUserItemId)
            .orElseThrow(() -> new WP2ResourceNotFoundException(favouriteUserItemId));
    }

    /**
     * Saves the entity and returns the mapped dto.
     *
     * @param favouriteUserItemEntity the entity to save
     * @return the mapped {@link OutputFavouriteUserItemDto}
     */
    private OutputFavouriteUserItemDto saveAndReturnDto(@NonNull FavouriteUserItemEntity favouriteUserItemEntity) {
        final FavouriteUserItemEntity savedFavouriteUserItemEntity = favouriteUserItemRepository.save(favouriteUserItemEntity);
        return favouriteUserItemMapper.toOutputDto(savedFavouriteUserItemEntity);
    }

    /**
     * Archives and then deletes the entity
     *
     * @param favouriteUserItemEntity the entity to delete
     */
    private void deleteEntity(@NonNull FavouriteUserItemEntity favouriteUserItemEntity) {
        persistArchivedCopy(favouriteUserItemEntity);
        favouriteUserItemRepository.delete(favouriteUserItemEntity);
    }

    /**
     * Archives the entity i.e. creates a copy in the archive table.
     *
     * @param favouriteUserItemEntity the entity to archive
     */
    private void persistArchivedCopy(FavouriteUserItemEntity favouriteUserItemEntity) {
        ArchivedFavouriteUserItemEntity archivedFavouriteSearchEntity = archivedFavouriteUserItemMapper.toArchiveEntity(favouriteUserItemEntity);
        archivedFavouriteUserItemRepository.saveAndFlush(archivedFavouriteSearchEntity);
    }
}
