package com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging;

import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.InstitutionalPageIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.client.dm.dto.ModerationIPDmDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputInviteMembersDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputModerationStatusDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputParagraphDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputPatchIPInvitationRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputPatchIPJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputPatchMembershipDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputRemoveMembershipDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.InputSubmitJoinRequestDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputInstitutionalPageDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.dto.OutputPendingMemberRequestsDto;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageCreatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageDeletedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPagePublicationRequestNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageUpdatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.InstitutionalPageStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MembershipRemovalNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.ParagraphCreatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.ParagraphUpdatedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.ParagraphDeletedNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MemberInvitationNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MemberInvitationStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.JoinRequestNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.JoinRequestStatusChangeNotificationData;
import com.finconsgroup.itserr.marketplace.institutional_page.bs.messaging.data.MembershipChangeNotificationData;
import com.finconsgroup.itserr.messaging.dto.MessagingEventDto;

import java.util.List;
import java.util.UUID;

/**
 * Factory interface for creating MessagingEventDto instances for various notification events.
 * <p>
 * This factory provides methods to convert different types of DTOs into MessagingEventDto objects
 * that can be used for publishing notification events through the messaging system.
 */
public interface NotificationMessageFactory {

    /**
     * Creates a notification message for institutional page creation events.
     *
     * @param institutionalPageDto the institutional page data
     * @return a MessagingEventDto containing notification details for page creation
     */
    MessagingEventDto<InstitutionalPageCreatedNotificationData> createInstitutionalPageCreatedNotification(
            OutputInstitutionalPageDto institutionalPageDto
    );

    /**
     * Creates a notification message for institutional page update events.
     *
     * @param institutionalPageDto the institutional page data
     * @return a MessagingEventDto containing notification details for page updates
     */
    MessagingEventDto<InstitutionalPageUpdatedNotificationData> createInstitutionalPageUpdatedNotification(
            OutputInstitutionalPageDto institutionalPageDto
    );

    /**
     * Creates a notification message for institutional page deletion events.
     *
     * @param institutionalPageDto the institutional page data
     * @return a MessagingEventDto containing notification details for page deletion
     */
    MessagingEventDto<InstitutionalPageDeletedNotificationData> createInstitutionalPageDeletedNotification(
            InstitutionalPageIPDmDto institutionalPageDto
    );

    /**
     * Creates a notification message for institutional page moderation status change events.
     *
     * @param inputModerationStatusDto the moderation status input data
     * @param moderationIPDmDto the institutional page data
     * @return a MessagingEventDto containing notification details for status changes
     */
    MessagingEventDto<InstitutionalPageStatusChangeNotificationData> createInstitutionalPageStatusChangeNotification(
            InputModerationStatusDto inputModerationStatusDto,
            ModerationIPDmDto moderationIPDmDto
    );

    MessagingEventDto<InstitutionalPagePublicationRequestNotificationData> createInstitutionalPagePublicationRequestNotification(
            OutputInstitutionalPageDto institutionalPageDto
    );

    /**
     * Creates a notification message for join request events.
     *
     * @param inputSubmitJoinRequestDto the join request input data
     * @param pendingMemberRequestsForUserDto the pending member requests data
     * @return a MessagingEventDto containing notification details for join requests
     */
    MessagingEventDto<JoinRequestNotificationData> createJoinRequestNotification(
            UUID institutionalPageId,
            InputSubmitJoinRequestDto inputSubmitJoinRequestDto,
            OutputPendingMemberRequestsDto pendingMemberRequestsForUserDto
    );

    /**
     * Creates a notification message for join request deletion events.
     *
     * @param institutionalPageIPDmDto the institutional page data from DM
     * @return a MessagingEventDto containing notification details for join request deletion
     */
    MessagingEventDto<JoinRequestNotificationData> createJoinRequestDeletedNotification(
        InstitutionalPageIPDmDto institutionalPageIPDmDto
    );

    /**
     * Creates a notification message for join request status change events.
     *
     * @param institutionalPageIPDmDto the institutional page data from DM
     * @param inputPatchIPJoinRequestDto the join request status change input data
     * @return a MessagingEventDto containing notification details for join request status changes
     */
    MessagingEventDto<JoinRequestStatusChangeNotificationData> createJoinRequestStatusChangeNotification(
            InstitutionalPageIPDmDto institutionalPageIPDmDto,
            InputPatchIPJoinRequestDto inputPatchIPJoinRequestDto
    );

    /**
     * Creates a notification message for member invitation events.
     *
     * @param institutionalPageIPDmDto the institutional page data from DM
     * @param inputInviteMembersDto the member invitation input data
     * @return a MessagingEventDto containing notification details for member invitations
     */
    MessagingEventDto<MemberInvitationNotificationData> createMemberInvitationNotification(
            InstitutionalPageIPDmDto institutionalPageIPDmDto,
            InputInviteMembersDto inputInviteMembersDto
    );

    /**
     * Creates a notification message for member invitation deletion events.
     *
     * @param institutionalPageIPDmDto the institutional page data from DM
     * @param invitedUserId the ID of the user whose invitation was deleted
     * @return a MessagingEventDto containing notification details for invitation deletion
     */
    MessagingEventDto<MemberInvitationNotificationData> createMemberInvitationDeletedNotification(
            InstitutionalPageIPDmDto institutionalPageIPDmDto,
            UUID invitedUserId
    );

    /**
     * Creates a notification message for member invitation status change events.
     *
     * @param institutionalPageIPDmDto the institutional page data from DM
     * @param inputPatchIPInvitationRequestDto the invitation status change input data
     * @param wpLeader true if the invitation is for becoming a wpLeader of the institutional page
     * @param hierarchyInstitutionalPages the list of institutional pages in the hierarchy of the institutional page
     * @return a MessagingEventDto containing notification details for invitation status changes
     */
    MessagingEventDto<MemberInvitationStatusChangeNotificationData> createMemberInvitationStatusChangeNotification(
            InstitutionalPageIPDmDto institutionalPageIPDmDto,
            InputPatchIPInvitationRequestDto inputPatchIPInvitationRequestDto,
            Boolean wpLeader,
            List<OutputInstitutionalPageDto> hierarchyInstitutionalPages
    );

    /**
     * Creates a notification message for membership change events.
     *
     * @param inputPatchMembershipDto the membership change input data
     * @param institutionalPageDto the institutional page data
     * @param hierarchyInstitutionalPages the list of institutional pages in the hierarchy of the institutional page
     * @return a MessagingEventDto containing notification details for membership changes
     */
    MessagingEventDto<MembershipChangeNotificationData> createMembershipChangeNotification(
            InputPatchMembershipDto inputPatchMembershipDto,
            OutputInstitutionalPageDto institutionalPageDto,
            List<OutputInstitutionalPageDto> hierarchyInstitutionalPages
    );

    /**
     * Creates a notification message for membership removal events.
     *
     * @param inputRemoveMembershipDto the membership removal input data
     * @param wpLeader true if the removed member was wp leader or not
     * @param institutionalPageDto the institutional page data
     * @param hierarchyInstitutionalPages the list of institutional pages in the hierarchy of the institutional page
     * @return a MessagingEventDto containing notification details for membership removal
     */
    MessagingEventDto<MembershipRemovalNotificationData> createMembershipRemovalNotification(
            InputRemoveMembershipDto inputRemoveMembershipDto,
            boolean wpLeader,
            OutputInstitutionalPageDto institutionalPageDto,
            List<OutputInstitutionalPageDto> hierarchyInstitutionalPages
    );

    /**
     * Creates a notification message for paragraph creation events.
     *
     * @param inputParagraphDto the paragraph input data
     * @param institutionalPageDto the institutional page data
     * @return a MessagingEventDto containing notification details for paragraph creation
     */
    MessagingEventDto<ParagraphCreatedNotificationData> createParagraphCreatedNotification(
            InputParagraphDto inputParagraphDto,
            OutputInstitutionalPageDto institutionalPageDto
    );

    /**
     * Creates a notification message for paragraph update events.
     *
     * @param inputParagraphDto the paragraph input data
     * @param institutionalPageDto the institutional page data
     * @return a MessagingEventDto containing notification details for paragraph updates
     */
    MessagingEventDto<ParagraphUpdatedNotificationData> createParagraphUpdatedNotification(
            InputParagraphDto inputParagraphDto,
            OutputInstitutionalPageDto institutionalPageDto
    );

    /**
     * Creates a notification message for paragraph deletion events.
     *
     * @param inputParagraphDto the paragraph input data
     * @param institutionalPageDto the institutional page data
     * @return a MessagingEventDto containing notification details for paragraph deletion
     */
    MessagingEventDto<ParagraphDeletedNotificationData> createParagraphDeletedNotification(
            InputParagraphDto inputParagraphDto,
            OutputInstitutionalPageDto institutionalPageDto
    );

    /**
     * Creates a notification message for pending invitation requests on an institutional page.
     *
     * @param institutionalPageDto the institutional page data
     * @param pendingRequestsDto the pending member requests data
     * @return a MessagingEventDto containing notification details for pending invitations
     */
    MessagingEventDto<MemberInvitationNotificationData> createPendingInvitationRequestsNotification(
            OutputInstitutionalPageDto institutionalPageDto,
            OutputPendingMemberRequestsDto pendingRequestsDto
    );

}
